﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/image/image_JpegDecoder.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>

#include "DevMenu_Common.h"
#include "DevMenu_RootSurface.h"
#include "Launcher/DevMenu_Launcher.h"

namespace devmenu {

namespace {

#if defined( NN_BUILD_CONFIG_OS_WIN )

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation in C11 rules for aligned memory allocations.@n
    //!        Memory allocator with alignment for start address of rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void* aligned_alloc( size_t alignment, size_t size ) NN_NOEXCEPT
    {
        return ::_aligned_malloc( size, alignment );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::_aligned_free( address );
    }

#else   // NX, Cafe, and C++11 supported platforms...

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release that has been imitated on the rules of the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::free( address );
    }
#endif

} // ~namespace devmenu::<anonymous>

void ConvertBytesToLargestType( char* outConvertedString, size_t stringSize, double byteSize, bool isSuffixAdded ) NN_NOEXCEPT
{
    const double kb = 1024.0;
    const double mb = kb * 1024.0;
    const double gb = mb * 1024.0;
    const double tb = gb * 1024.0;

    std::string suffix;
    std::string stringFormat = "%.2lf %s";

    if (byteSize > tb)
    {
        byteSize /= tb;
        suffix = "TBytes";
    }
    else if (byteSize > gb)
    {
        byteSize /= gb;
        suffix = "GBytes";
    }
    else if (byteSize > mb)
    {
        byteSize /= mb;
        suffix = "MBytes";
    }
    else if (byteSize > kb)
    {
        byteSize /= kb;
        suffix = "KBytes";
    }
    else
    {
        // Since you can't have a fraction of a byte, set the string format to not include the decimal places
        suffix = "Bytes";
        stringFormat = "%.0lf %s";
    }


    if ( isSuffixAdded )
    {
        nn::util::SNPrintf( outConvertedString, stringSize, stringFormat.c_str(), byteSize, suffix.c_str() );

    }
    else
    {
        nn::util::SNPrintf( outConvertedString, stringSize, "%.2lf", byteSize );
    }
}

void CreateErrorMessage( std::string* pOutStr, const std::string& errorStr, const nn::Result& result, bool addsNewLine ) NN_NOEXCEPT
{
    char errorValueStr[ 32 ];
    nn::util::SNPrintf( errorValueStr, sizeof( errorValueStr ), "(result: 0x%08x)", result.GetInnerValueForDebug() );

    if ( addsNewLine )
    {
        *pOutStr = errorStr + '\n' + std::string( errorValueStr );
    }
    else
    {
        *pOutStr = errorStr + std::string( errorValueStr );
    }
}

void CreateErrorMessage( std::string* pOutStr, const std::string& errorStr, bool addsNewLine ) NN_NOEXCEPT
{

    if ( addsNewLine )
    {
        *pOutStr = errorStr + '\n';
    }
    else
    {
        *pOutStr = errorStr;
    }
}

void CreateErrorMessageView( MessageView* pOutView, const std::string& errorStr, const nn::Result& result, const std::string& buttonStr ) NN_NOEXCEPT
{
    DEVMENU_LOG( "%s (result: 0x%08x)\n", errorStr.c_str(), result.GetInnerValueForDebug() );
    std::string displayStr( 64, '\0' );
    CreateErrorMessage( &displayStr, errorStr, result );
    pOutView->AddMessage( displayStr );
    pOutView->AddButton( buttonStr );
}

void CreateErrorMessageView( devmenu::MessageView* pOutView, const std::string& errorStr, const std::string& buttonStr ) NN_NOEXCEPT
{
    DEVMENU_LOG( "%s\n", errorStr.c_str());
    std::string displayStr( 64, '\0' );
    CreateErrorMessage( &displayStr, errorStr);
    pOutView->AddMessage( displayStr );
    pOutView->AddButton( buttonStr );
}

MessageView* CreateRebootRequestDialog() NN_NOEXCEPT
{
    auto pView = new MessageView( false );

    // Set up the Message And Buttons
    pView->AddMessage( "Reboot device to reflect changes." );
    pView->AddButton( "Close" );

    return pView;
}

std::string GetDelimitedNumberString( int64_t number ) NN_NOEXCEPT
{
    const int delimitedStrLength    = 3;

    std::string baseNumberString    = std::to_string( number );
    std::string outString           = "";
    int numberStrLength             = static_cast<int>( baseNumberString.size() );
    int reaminCount                 = baseNumberString.size() % delimitedStrLength;
    int delimitedCount              = (baseNumberString.size() - reaminCount) / delimitedStrLength;

    if ( numberStrLength <= delimitedStrLength )
    {
        return baseNumberString;
    }

    if ( reaminCount > 0 )
    {
        outString = baseNumberString.substr( 0, reaminCount );
    }

    for ( int i = 0; i < delimitedCount; i++ )
    {
        if ( outString != "" )
        {
            outString.append( "," );
            outString.append( baseNumberString.substr(reaminCount + (i * delimitedStrLength), delimitedStrLength) );
        }
        else
        {
            outString = baseNumberString.substr( reaminCount + (i * delimitedStrLength), delimitedStrLength );
        }
    }

    return outString;
}

/*********************************
 * class MemoryAllocator
 *********************************/

void* MemoryAllocator::AllocAlignedMemory( size_t alignment, size_t size ) NN_NOEXCEPT
{
    return aligned_alloc( alignment, size );
}

void MemoryAllocator::FreeAlignedMemory( void* address ) NN_NOEXCEPT
{
    aligned_free( address );
}

/*********************************
 * class RebootAttentionLabel
 *********************************/

void RebootAttentionLabel::ApplyInitialStyle() NN_NOEXCEPT
{
    m_Style.color.text.set( glv::Color( 1.0f, 0.0f, 0.0f ) );
    m_Style.color.border.set( 1.0f, 0.0f, 0.0f );

    enable( glv::Property::DrawBorder );
    style( &m_Style );
    m_pRebootIcon->style( &m_Style );
    m_pRebootText->style( &m_Style );
    disable( glv::Property::DrawBack | glv::Property::Visible );
}

/*********************************
 * class Button
 *********************************/

Button::Button( const std::string& text, std::function< void() > callback, float fontSize, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT
    : glv::Button( glv::Rect( 80.0f, 20.0f ), true )
    , m_Label( text, false )
    , m_Callback( callback )
{
    FitButton( fontSize, paddingX, paddingY );
    Initialize();
}

Button::Button( const char* text, std::function< void() > callback, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : glv::Button( rect, true )
    , m_Label( text, glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize ) )
    , m_Callback( callback )
{
    this->anchor( anchor );
    Initialize();
}

Button::Button( const char* text, std::function< void() > callback, const glv::Rect& rect, const glv::Label::Spec& spec ) NN_NOEXCEPT
    : glv::Button( rect, true )
    , m_Label( text, spec )
    , m_Callback( callback )
{
    Initialize();
}

Button::Button( const glv::WideString& text, std::function< void() > callback, float fontSize, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT
    : glv::Button( glv::Rect( 80.0f, 20.0f ), true )
    , m_Label( text, false )
    , m_Callback( callback )
{
    FitButton( fontSize, paddingX, paddingY );
    Initialize();
}

Button::Button( const glv::WideString& text, std::function< void() > callback, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : glv::Button( rect, true )
    , m_Label( text, glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize ) )
    , m_Callback( callback )
{
    this->anchor( anchor );
    Initialize();
}

Button::Button( const glv::WideString& text, std::function< void() > callback, const glv::Rect& rect, const glv::Label::Spec& spec ) NN_NOEXCEPT
    : glv::Button( rect, true )
    , m_Label( text, spec )
    , m_Callback( callback )
{
    Initialize();
}

void Button::Initialize() NN_NOEXCEPT
{
    *this << m_Label;
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask );
    attach( []( const glv::Notification& notification )->void { notification.receiver< Button >()->m_Callback(); }, glv::Update::Clicked, this );

    m_StyleLabel[ ButtonStyle_Focusable ] = m_Label.style();
    m_StyleLabel[ ButtonStyle_NotFocusable ].color = glv::Style::standard().color;
    m_StyleLabel[ ButtonStyle_NotFocusable ].color.set( glv::StyleColor::BlackOnWhite );
    m_StyleLabel[ ButtonStyle_NotFocusable ].color.fore.set( 0.8f, 0.8f, 0.8f );
    m_StyleLabel[ ButtonStyle_FocusableButInvalid ] = glv::Style( m_StyleLabel[ ButtonStyle_NotFocusable ] );

    m_StyleButton[ ButtonStyle_Focusable ] = style();
    m_StyleButton[ ButtonStyle_NotFocusable ].color = glv::Style::standard().color;
    m_StyleButton[ ButtonStyle_NotFocusable ].color.fore.set  ( 0.3f, 0.3f, 0.3f );
    m_StyleButton[ ButtonStyle_NotFocusable ].color.border.set( 0.3f, 0.3f, 0.3f );
    m_StyleButton[ ButtonStyle_NotFocusable ].color.back.set  ( 0.15f, 0.15f, 0.15f );
    m_StyleButton[ ButtonStyle_FocusableButInvalid ] = glv::Style( m_StyleButton[ ButtonStyle_NotFocusable ] );
    m_StyleButton[ ButtonStyle_FocusableButInvalid ].color.back.set  ( 0.2f, 0.2f, 0.2f );
}


void Button::OnClicked() NN_NOEXCEPT
{
    m_Callback();
}

void Button::UpdateFocusAndColor( bool isValid, bool isFocusable ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    const glv::Property::t focusableButInvalidProperty = glv::Property::t::Controllable;

    if ( isValid )
    {
        style( &m_StyleButton[ ButtonStyle_Focusable ] );
        m_Label.style( &m_StyleLabel[ ButtonStyle_Focusable ] );
        enable( focusableProperty );
    }
    else
    {
        m_Label.disable( glv::Property::DrawBack );

        if ( !isFocusable )
        {
            style( &m_StyleButton[ ButtonStyle_NotFocusable ] );
            m_Label.style( &m_StyleLabel[ ButtonStyle_NotFocusable ] );
            disable( focusableProperty );
        }
        else
        {
            style( &m_StyleButton[ ButtonStyle_FocusableButInvalid ] );
            m_Label.style( &m_StyleLabel[ ButtonStyle_FocusableButInvalid ] );
            disable( focusableButInvalidProperty );
        }
    }
}

void Button::UpdateLabelText( const char* text ) NN_NOEXCEPT
{
    m_Label.setValue( text );
}

void Button::UpdateLabelText( const glv::WideString& text ) NN_NOEXCEPT
{
    m_Label.setValue( text );
}

void Button::SetCallback( std::function< void() > callback ) NN_NOEXCEPT
{
    m_Callback = callback;
}

void Button::IncreaseLabelSize( glv::space_t increasedX, glv::space_t increasedY ) NN_NOEXCEPT
{
    glv::Button::extent( m_Label.width() + increasedX, m_Label.height() + increasedY );
}

void Button::ChangeLabelSize( glv::space_t width, glv::space_t height ) NN_NOEXCEPT
{
    glv::Button::extent( width, height );
}

Button& Button::AddEventHandler( glv::Event::t event, EventHandlerMap* pHandlerMap ) NN_NOEXCEPT
{
    auto& handlerMap = *pHandlerMap;
    EventHandlerMap::iterator iter = handlerMap.find( event );
    if ( handlerMap.end() != iter )
    {
        this->addHandler( event, *handlerMap[ event ] );
    }
    return *this;
}

glv::Label& Button::GetLabel() NN_NOEXCEPT
{
    return m_Label;
}

void Button::FitButton( float fontSize, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT
{
    m_Label.size( fontSize );
    m_Label.paddingX( paddingX );
    m_Label.paddingY( paddingY );
    m_Label.anchor( glv::Place::CC );
    m_Label.pos( glv::Place::CC, 0.0f, 0.0f ).anchor( glv::Place::CC );
    const glv::space_t width = ( m_Label.width() + 1.0f ) / 2.0f * 2.0f;   // ピクセルだから、センタリング時に 2 の倍数じゃないと...
    const glv::space_t height = ( m_Label.height() + 1.0f ) / 2.0f * 2.0f; // ピクセルだから、センタリング時に 2 の倍数じゃないと...
    glv::Button::extent( width, height );
}

/*********************************
 * class Table
 *********************************/
void Table::RemoveAllChildren() NN_NOEXCEPT
{
    std::vector< glv::View* > viewList;
    RemoveAllChildren( &viewList);
}

void Table::RemoveAllChildren( std::vector< glv::View* >* pOutChildren ) NN_NOEXCEPT
{
    auto pView = this->child;
    while ( nullptr != pView )
    {
        pOutChildren->push_back( pView );
        auto pSibling = pView->sibling;
        pView->remove();
        pView = pSibling;
    }
}

void Table::ReAddOwnselfToParent( glv::View* pParentView, bool isChildrenMoved ) NN_NOEXCEPT
{
    std::vector< glv::View* > children;

    RemoveAllChildren( &children );

    for ( auto& childView : children )
    {
        *this << childView;
    }

    this->arrange().fit( isChildrenMoved );
    *pParentView << this;
}

} // ~namespace devmenu
