﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "CloudBackup/DevMenu_CloudBackupCommons.h"

#include "CloudBackup/DevMenu_CloudBackupScene.h"
#include "CloudBackup/DevMenu_TransferTaskListScene.h"
#include "CloudBackup/DevMenu_ErrorListScene.h"
#include "CloudBackup/DevMenu_AccountInfoScene.h"
#include "CloudBackup/DevMenu_ApplicationInfoScene.h"

namespace devmenu { namespace cloudbackup {

/**************************************
class CloudBackupPage
**************************************/
class CloudBackupPage : public Page, public cloudbackup::ParentPageInterface
{
public:

    CloudBackupPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect)
        , m_pCloudBackupScene(nullptr)
        , m_pTransferTaskListScene(nullptr)
        , m_pErrorListScene(nullptr)
        , m_pAccountInfoScene(nullptr)
        , m_pApplicationInfoScene(nullptr)
    {
        // NOOP
    }

    /**
    * @brief ページがコンテナに追加された後に呼び出されます。
    */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        auto rect = glv::Rect(w, h);

        // ルートのシーン
        m_pCloudBackupScene = new CloudBackupScene(this, rect);
        m_pCloudBackupScene->SetBackButtonCallback([&] { this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); });
        m_pCloudBackupScene->enable(glv::Property::Visible);

        // 転送タスク一覧
        m_pTransferTaskListScene = new TransferTaskListScene(this, rect);
        m_pTransferTaskListScene->SetBackButtonCallback([&] { this->SwitchScene(SceneType_CloudBackupRoot, true); });
        m_pTransferTaskListScene->disable(glv::Property::Visible);

        // エラー一覧
        m_pErrorListScene = new ErrorListScene(this, rect);
        m_pErrorListScene->SetBackButtonCallback([&] { this->SwitchScene(SceneType_CloudBackupRoot, true); });
        m_pErrorListScene->disable(glv::Property::Visible);

        // アカウント情報
        m_pAccountInfoScene = new AccountInfoScene(this, rect);
        m_pAccountInfoScene->SetBackButtonCallback([&] { this->SwitchScene(SceneType_CloudBackupRoot, true); });
        m_pAccountInfoScene->disable(glv::Property::Visible);

        // アプリケーション情報( TODO:セーブデータが主なので名称が変わる可能性がある)
        m_pApplicationInfoScene = new ApplicationInfoScene(this, rect);
        m_pApplicationInfoScene->SetBackButtonCallback([&] { this->SwitchScene(SceneType_AccountInfo, true); });
        m_pApplicationInfoScene->disable(glv::Property::Visible);

        *this << m_pCloudBackupScene;
        *this << m_pTransferTaskListScene;
        *this << m_pErrorListScene;
        *this << m_pAccountInfoScene;
        *this << m_pApplicationInfoScene;

        SetActiveScene(SceneType_CloudBackupRoot);
    }

    /**
    * @brief ページがコンテナから削除された前に呼び出されます。
    */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        // NOOP
    }

    /**
    * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
    */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        // フォーカスを記憶し、処理後に元に戻す。
        auto pRootSurface = GetRootSurfaceContext();
        NN_ASSERT_NOT_NULL(pRootSurface);
        auto pCurrentView = pRootSurface->focusedView();

        GetActiveScene()->OnActivatePage();

        pRootSurface->setFocus(pCurrentView); // フォーカスを元に戻す
    }

    /**
    * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
    */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        // NOOP
    }

    /**
    * @brief バックグラウンド遷移処理です。ページがアクティブな場合にのみ呼ばれます。
    */
    virtual void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE
    {
        // NOOP
    }

    /**
    * @brief フォアグラウンド遷移処理です。ページがアクティブな場合にのみ呼ばれます。
    */
    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        // NOOP
    }

    /**
    * @brief アプリケーションメインループからのコールバックです。
    *
    * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
    * この時点ではまだ glvコンテキストのレンダリングは開始していません。
    */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        GetActiveScene()->OnLoopBeforeSceneRenderer(context, events);
    }

    /**
    * @brief アプリケーションメインループからのコールバックです。
    *
    * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
    */
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        GetActiveScene()->OnLoopAfterSceneRenderer(context, events);
    }

    /**
    * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return nullptr;
    }

    virtual Scene* GetScene(int sceneIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        SceneType type = SceneType(sceneIndex);

        switch (type)
        {
        case SceneType_CloudBackupRoot:
            return m_pCloudBackupScene;

        case SceneType_TransferTaskList:
            return m_pTransferTaskListScene;

        case SceneType_ErrorList:
            return m_pErrorListScene;

        case SceneType_AccountInfo:
            return m_pAccountInfoScene;

        case SceneType_ApplicationInfo:
            return m_pApplicationInfoScene;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    /**
    * @brief シーンを切替えます。子シーンへのインターフェースメソッドの定義です。
    */
    virtual void SwitchScene(SceneType nextScene, bool isBack) NN_NOEXCEPT final NN_OVERRIDE
    {
        Page::SwitchScene(nextScene, isBack);
    }

    /**
    * @brief PageのRootSurfaceContextを取得します。子シーンへのインターフェースメソッドの定義です。
    */
    virtual RootSurfaceContext* GetParentPageRootSurfaceContext() const NN_NOEXCEPT final NN_OVERRIDE
    {
        return GetRootSurfaceContext();
    }

    /**
    * @brief アカウント情報シーンを開きます。子シーンへのインターフェースメソッドの定義です。
    */
    virtual void OpenAccountInfo(const nn::account::Uid& uid) NN_NOEXCEPT final NN_OVERRIDE
    {
        m_pAccountInfoScene->SetAccountUid(uid);
        SwitchScene(SceneType_AccountInfo, false);
    }

private:
    CloudBackupScene* m_pCloudBackupScene;
    TransferTaskListScene* m_pTransferTaskListScene;
    ErrorListScene* m_pErrorListScene;
    AccountInfoScene* m_pAccountInfoScene;
    ApplicationInfoScene* m_pApplicationInfoScene;
};

/**************************************
class PageCreator
**************************************/

template< size_t ID >
class CroudBackupPageCreator : PageCreatorBase
{
public:
    explicit CroudBackupPageCreator(const char* pageName) NN_NOEXCEPT
        : PageCreatorBase(ID, pageName) {}

    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[2];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution(resolution[0], resolution[1]);
        const glv::space_t width = static_cast<glv::space_t>(resolution[0]);
        const glv::space_t height = static_cast<glv::space_t>(resolution[1]);

        const glv::Rect pageBounds(width - 218.f, height - 118.0f);
        return new CloudBackupPage(ID, GLV_TEXT_API_WIDE_STRING("nn::olsc"), pageBounds);
    }
};

#if 0
#define LOCAL_PAGE_CREATOR( _id, _name ) CroudBackupPageCreator< _id > g_CroudBackupPageCreator##_id( _name );
        LOCAL_PAGE_CREATOR(DevMenuPageId_CroudBackup, "CroudBackupPage");
#endif
}} // ~namespace devmenu::cloudbackup, ~namespace devmenu
