﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_Battery.h"
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/psm/psm_Api.h>
#include <nn/psm/psm_System.h>
#include <algorithm>
#include <memory>

namespace devmenu {

namespace {

//#define ENABLE_AUTO_INCREMENTAL_METER   // デバッグ用です
#if defined( ENABLE_AUTO_INCREMENTAL_METER )
    static int g_DebugBatteryPower = 0;
    static int g_AutoIncrementalDirection = 1;
#endif

    static const glv::space_t    ChargeMarkMargin = 4.0f;
    static const glv::space_t    BatteryIconMargin = 8.0f;
    static const char*           BatteryIconTextureResourceFilePath = "Contents:/drawable/HudIconBattery.bntx";

    // TODO: TextureUvRegion[1] のテクスチャ上での実寸幅( width )は27ですが、
    // インジケータ領域の左パディングを 5に統一するためにテクスチャラップ(CLAMP)前提で +1 しています。
    // テクスチャ直すのが面倒という理由ではない、決して。
    static const glv::Rect TextureUvRegion[] =
    {
        glv::Rect(  0.f,  0.f, 36.f, 20.f ),    // icon back for machine.
        glv::Rect( 37.f,  0.f, 28.f, 20.f ),    // icon back for pad controller.
        glv::Rect(  0.f, 21.f, 16.f, 26.f ),    // charge mark.
    };

    NN_FORCEINLINE static const glv::Rect& GetTextureUvRegion( const BatteryIndicator::IconDesign design )
    {
        return TextureUvRegion[ static_cast< unsigned int >( design ) ];
    }
}

//!--------------------------------------------------------------------------------------
//! @brief コンストラクタ
//!--------------------------------------------------------------------------------------
BatteryIndicator::BatteryIndicator( IconDesign iconDesign, bool percentageDisplay, float percentageFontSize ) NN_NOEXCEPT
    : glv::View(),
    m_pDrawable( glv::Resources::GetSharedResourceContext().GetDrawable( BatteryIconTextureResourceFilePath ) ),
    m_ColorSafety( 1.0f ),
    m_ColorCritical( 1.0f, 0, 0.35f ),
    m_IsDisplayPercentage( percentageDisplay ),
    m_FontSize( percentageFontSize ),
    m_IconDesign( iconDesign )
{
    glv::View::disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::FocusHighlight | glv::Property::FocusToTop | glv::Property::HitTest | glv::Property::Controllable | glv::Property::Animate );
    m_pDrawable->SetTextureWrap( GL_CLAMP, GL_CLAMP );

    glv::space_t width;
    glv::space_t height;

    if ( true == m_IsDisplayPercentage )
    {
        font().size( percentageFontSize );
        font().getBounds( width, height, "000%" );
    }
    else
    {
        width = height = 0;
    }

    const glv::Rect& iconBack = GetTextureUvRegion( iconDesign );
    const glv::Rect& markCharge = TextureUvRegion[ 2 ];
    m_Width = width + BatteryIconMargin + iconBack.width() + ChargeMarkMargin + markCharge.width();
    m_Height = ( height > markCharge.height() ) ? height : markCharge.height();
    extent( m_Width, m_Height );
}

//!--------------------------------------------------------------------------------------
//! @brief デストラクタ
//!--------------------------------------------------------------------------------------
BatteryIndicator::~BatteryIndicator() NN_NOEXCEPT
{
    glv::Drawable* pDrawable;
    if ( nullptr != ( pDrawable = m_pDrawable ) )
    {
        m_pDrawable = nullptr;
        pDrawable->Finish();
    }
}

//!--------------------------------------------------------------------------------------
//! @brief 任意座標系での描画
//!--------------------------------------------------------------------------------------
void BatteryIndicator::Draw( glv::space_t originLeft, glv::space_t originTop, int percentage, BatteryChargerState chargerState ) NN_NOEXCEPT
{
    glv::Drawable* pDrawable;
    if ( nullptr != ( pDrawable = m_pDrawable ) )
    {
        glv::draw::enable( glv::draw::Texture2D );

        const glv::space_t sheetWidth = pDrawable->GetIntrinsicWidth();
        const glv::space_t sheetHeight = pDrawable->GetIntrinsicHeight();
        const glv::space_t viewWidth = m_Width;
        const glv::space_t viewHeight = m_Height;
        glv::space_t batteryIconAnchorRight = originLeft + viewWidth;

        // 充電マーク表示
        if ( BatteryIndicator::BatteryChargerState_EnoughPower == chargerState )
        {
            const glv::Rect& iconCharge = TextureUvRegion[ 2 ];
            const glv::space_t markOriginX = iconCharge.left();
            const glv::space_t markOriginY = iconCharge.top();
            const glv::space_t markTexTop = markOriginY / sheetWidth;
            const glv::space_t markTexLeft = markOriginX / sheetHeight;
            const glv::space_t markTexRight = ( markOriginX + iconCharge.width() - 1 ) / sheetWidth;
            const glv::space_t markTexBottom = ( markOriginY + iconCharge.height() - 1 ) / sheetHeight;

            const glv::space_t markTop = originTop + ( ( viewHeight > iconCharge.height() ) ? ( ( viewHeight - iconCharge.height() ) / 2.f ) : 0 );
            const glv::space_t markLeft = originLeft + viewWidth - iconCharge.width();
            const glv::space_t markRight = markLeft + iconCharge.width() - 1;
            const glv::space_t markBottom = markTop + iconCharge.height() - 1;
            batteryIconAnchorRight = markLeft - ChargeMarkMargin;
            pDrawable->SetColorFilter( style().color.border );
            pDrawable->Draw( markLeft, markTop, markRight, markBottom, markTexLeft, markTexTop, markTexRight, markTexBottom );
        }

        // 電池アイコン背景
        const glv::Rect& iconBack = GetTextureUvRegion( m_IconDesign );
        const glv::space_t originX = iconBack.left();
        const glv::space_t originY = iconBack.top();
        const glv::space_t texTop = originY / sheetWidth;
        const glv::space_t texLeft = originX / sheetHeight;
        const glv::space_t texRight = ( originX + iconBack.width() - 1 ) / sheetWidth;
        const glv::space_t texBottom = ( originY + iconBack.height() - 1 ) / sheetHeight;

        const glv::space_t topAnchor = originTop + ( ( viewHeight > iconBack.height() ) ? ( ( viewHeight - iconBack.height() ) / 2.f ) : 0 );
        const glv::space_t leftAnchor = batteryIconAnchorRight - iconBack.width();
        const glv::space_t rightAnchor = batteryIconAnchorRight - 1;
        const glv::space_t bottomAnchor = topAnchor + iconBack.height() - 1;
        pDrawable->SetColorFilter( style().color.border );
        pDrawable->Draw( leftAnchor, topAnchor, rightAnchor, bottomAnchor, texLeft, texTop, texRight, texBottom );

        // 残量 % インジケータ表示
        if ( percentage > 0 )
        {
            const float factor = ( percentage <= 100 ) ? percentage / 100.0f : 1.0f;
            const glv::space_t leftIndicator = leftAnchor + 3;
            const glv::space_t rightIndicator = rightAnchor - 5;
            const glv::space_t batteryLevel = ( ( rightIndicator - leftIndicator ) * factor );
            if ( static_cast< int >( batteryLevel ) >= 1.0f )   // 1画素よりも狭ければ描画しない
            {
                const glv::space_t topIndicator = topAnchor + 3;
                const glv::space_t bottomIndicator = bottomAnchor - 2;
                glv::draw::color( ( percentage < 20 ) ? m_ColorCritical : m_ColorSafety );
                glv::draw::rectangle( leftIndicator, topIndicator, leftIndicator + batteryLevel, bottomIndicator );
                glv::draw::color( 1.0f );
            }
        }

        glv::draw::disable( glv::draw::Texture2D );

        // 残量 % 数値表示
        if ( true == m_IsDisplayPercentage )
        {
            glv::space_t fWidth;
            glv::space_t fHeight;
            char ConvertBuffer[ 16 ];
            nn::util::SNPrintf( ConvertBuffer, sizeof( ConvertBuffer ), "%3d%%", percentage );
            font().getBounds( fWidth, fHeight, ConvertBuffer );
            font().render( ConvertBuffer, leftAnchor - BatteryIconMargin - fWidth, originTop );
        }
    }
}

//!--------------------------------------------------------------------------------------
//! @brief 描画コールバック
//!--------------------------------------------------------------------------------------
void BatteryIndicator::onDraw( glv::GLV& g ) NN_NOEXCEPT
{
    int percentage;
    const BatteryChargerState chargerState = OnQueryBatteryState( percentage );
    Draw( 0, 0, percentage, chargerState );
}

//!--------------------------------------------------------------------------------------
//! @brief 電池残量状態問い合わせ
//!--------------------------------------------------------------------------------------
const BatteryIndicator::BatteryChargerState BatteryIndicator::OnQueryBatteryState( int& outPercentageOnBatteryPower ) NN_NOEXCEPT
{

#if defined( NN_BUILD_CONFIG_OS_HORIZON )

    // 呼び出し負荷計測 ( SDEV1.6 : 0.15～0.20ミリ秒 )

    // 充電状態取得
    BatteryChargerState resultChargerType;
    switch ( nn::psm::GetChargerType() )
    {
    case nn::psm::ChargerType_Unconnected:
        resultChargerType = BatteryChargerState::BatteryChargerState_Unconnected;
        break;
    case nn::psm::ChargerType_EnoughPower:
        resultChargerType = BatteryChargerState::BatteryChargerState_EnoughPower;
        break;
    case nn::psm::ChargerType_LowPower:
        resultChargerType = BatteryChargerState::BatteryChargerState_LowPower;
        break;
    case nn::psm::ChargerType_NotSupported:
        resultChargerType = BatteryChargerState::BatteryChargerState_NotSupported;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 電池残量( 1 ～ 100% )取得
    outPercentageOnBatteryPower = nn::psm::GetBatteryChargePercentage();
    return resultChargerType;

#else

#if defined( ENABLE_AUTO_INCREMENTAL_METER )

    g_DebugBatteryPower += g_AutoIncrementalDirection;
    const int nowPower = g_DebugBatteryPower / 5;
    outPercentageOnBatteryPower = ( nowPower >= 100 ) ? 100 : nowPower;
    if ( nowPower >= 110 )
    {
        g_AutoIncrementalDirection = -1;
    }
    else if ( nowPower <= 0 )
    {
        g_AutoIncrementalDirection = 1;
    }

#else

    outPercentageOnBatteryPower = 35;

#endif

    //return BatteryChargerState::BatteryChargerState_Unconnected;
    return BatteryChargerState::BatteryChargerState_EnoughPower;

#endif

}

} // ~namespace devmenu
