﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenu_ApplicationFileListSelector.h"
#include "DevMenu_Config.h"
#include "DevMenu_Common.h"
#include "DevMenu_ModalView.h"

namespace devmenu { namespace application {

// 相互参照を防ぐためのクラス参照
class ApplicationCatalogPage;

namespace write {

const int64_t MaxDirectoryNameLength = 300;  //!< SD カード内アプリケーション名の最大文字数（終端文字を含む）

/**
* @brief SD カード内のアプリケーションプロパティ型です。
*/
struct FilePropertyType
{
    /**
    * @brief SD カード内のアプリケーションリスト用プロパティデータを構築します。
    */
    void Prepare( const nn::fs::DirectoryEntry& directoryEntry, const char* pFilePath ) NN_NOEXCEPT
    {
        BuildUtf16< MaxDirectoryNameLength >( name, "%s%s", pFilePath, directoryEntry.name );
        const auto delimitedSizeStr = GetDelimitedNumberString( directoryEntry.fileSize );
        BuildUtf16< 32 >( size, "%s", delimitedSizeStr.c_str() );
    }

    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];
};

/**
 * @brief リストビュー
 */
class SdCardListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit SdCardListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size( 20.f );
    }

protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds( outWidth, outHeight, item.name );
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        font.getBounds( outWidth, outHeight, item.size );
        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render( item.name, posName, contentRegion.top() );

        font.getBounds( outWidth, outHeight, item.size );
        const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
        const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
        font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
    }
};

/**
 * @brief GameCard書き込み確認モーダルビュー
 */
class ConfirmView : public MessageView
{
    NN_DISALLOW_COPY( ConfirmView );
    NN_DISALLOW_MOVE( ConfirmView );

public:
    explicit ConfirmView( bool isRadioButtonRequired ) NN_NOEXCEPT;
    virtual ~ConfirmView() NN_NOEXCEPT NN_OVERRIDE;

    void ExtentSize() NN_NOEXCEPT; // Expected to be called in final step

    // Get and Set the Storage ID for the installation target
    nn::ncm::StorageId GetSelectedTargetDevice() NN_NOEXCEPT;

private:
    nn::ncm::StorageId m_WriteLocationStorageId;
};

/**
 * @brief 進捗表示モーダルビュー
 */
class WriteView : public MessageView
{
    NN_DISALLOW_COPY( WriteView );
    NN_DISALLOW_MOVE( WriteView );

public:
    typedef std::function< void() > ButtonCallback;

public:

    /**
     * @brief       コンストラクタです。
     */
    NN_IMPLICIT WriteView( const ButtonCallback& closeCallback ) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~WriteView() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief      メッセージを追加します。
     *
     * @param[in]   pProgressCaption    状態を表示するためのラベル
     * @param[in]   pProgressFileName   GameCard書き込みしているファイル名を表示するためのラベル
     * @param[in]   pProgressLabel      GameCard書き込み容量を表示するラベル
     */
    WriteView& AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton   ボタン
     */
    WriteView& AddButton( glv::Button* pButton ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton     ボタン
     * @param[in]   pFunc               ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam              ボタン押下時に実行される関数への引数
     */
    WriteView& AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT;

    /**
     * @brief       進捗状況バーを追加します。
     *
     * @param[in]   pProgressBar   進捗状況バーのスライダー
     */
    WriteView& AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT;

    /**
     * @brief       GameCard書き込みしているファイル名を設定します。
     *
     * @param[in]   fileName        ファイル名
     */
    void SetProgressFileName( const char* fileName ) NN_NOEXCEPT { m_ProgressFileName.setValue( fileName ); };

    /**
     * @brief       GameCard書き込み進捗メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressLabel( const char* message ) NN_NOEXCEPT { m_ProgressLabel.setValue( message ); };

    /**
     * @brief       直近でGameCard書き込みした合計ファイルのサイズを取得します。
     * @return      ファイルのサイズ
     */
    int64_t GetLastWrittenNspTotalSize() NN_NOEXCEPT { return m_LastWrittenNspTotalSize; };

    /**
     * @brief       直近でGameCard書き込みした合計ファイルのサイズを設定します。
     *
     * @param[in]   totalSize       ファイルサイズ
     */
    void SetLastWrittenNspTotalSize( int64_t totalSize ) NN_NOEXCEPT { m_LastWrittenNspTotalSize = totalSize; };

    /**
     * @brief       進捗状況バーの進捗度を設定します。
     *
     * @param[in]   value       進捗度( 0.0f - 1.0f )
     */
    void SetProgressBarValue( double value ) NN_NOEXCEPT { m_ProgressBar.setValue( value ); };

    /**
     * @brief       GameCard書き込み完了時に表示されるボタンを取得します。
     * @return      ボタン
     */
    glv::Button* GetButton() NN_NOEXCEPT { return &m_CloseButton; };

    /**
     * @brief       GameCard書き込み状態メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressCaption( const char* message ) NN_NOEXCEPT { m_ProgressCaption.setValue( message ); };

    /**
     * @brief       エラー画面を構築します。
     */
    void SetErrorMessageView() NN_NOEXCEPT;


    /**
     * @brief       GameCard書き込み結果を設定します。
     */
    void SetWriteResult( const nn::Result& result ) NN_NOEXCEPT { m_WriteResult = result; };

    /**
     * @brief       GameCard書き込み完了ボタンを設定します。
     */
    void SetCloseButton() NN_NOEXCEPT;

    /**
     * @brief       ButtonsTable に追加された Button を全て削除します。
     */
    void RemoveButtons() NN_NOEXCEPT;

private: // メンバ定数

    static const size_t           InitialActionCapacity; // 実行情報の初期数
    static const glv::space_t     WriteViewWidth;
    static const glv::space_t     ProgressBarMargineX;
    static const glv::space_t     MessageButtonPaddingX;
    static const glv::space_t     MessageButtonPaddingY;
    static const glv::space_t     MessageViewPaddingX;
    static const glv::space_t     MessageViewPaddingY;

    static const glv::Rect        ProgressBarRect;
    static const glv::Label::Spec DefaultLabelSpec;

private: // メンバ関数

    virtual void Initialize() NN_NOEXCEPT;

private: // メンバ変数

    glv::Table                  m_ProgressBarsTable;        //!< スライダー配置テーブル
    glv::Slider                 m_ProgressBar;              //!< 進捗状況を表示するためのバー
    glv::Label                  m_ProgressCaption;          //!< 状態を表示するためのラベル
    glv::Label                  m_ProgressFileName;         //!< GameCard書き込みしているファイル名を表示するためのラベル
    glv::Label                  m_ProgressLabel;            //!< GameCard書き込み容量を表示するラベル
    devmenu::Button             m_CloseButton;              //!< GameCard書き込み完了時に表示されるボタン
    nn::Result                  m_WriteResult;              //!< GameCard書き込み結果
    int64_t                     m_LastWrittenNspTotalSize;   //!< 直近でGameCard書き込みした合計ファイルサイズ（GameCard書き込み完了表示用）
};

/**
 * @brief アプリケーションGameCard書き込みシーン
 */
class WriteScene : public Scene
{
public:
    /**
     * @brief コンストラクタです。
     */
    WriteScene( ApplicationCatalogPage* pParentPage, Scene* pParentScene = nullptr ) NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    /**
    * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief       GameCard書き込み進捗表示を更新します。
     */
    void UpdateWriteProgress() NN_NOEXCEPT;

    /**
    * @brief nspリスト画面に切り替えます。
    */
    void ListupCallBack( nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT;

    /**
    * @brief nspリスト画面で選択されたnspファイルをコンテンツタイプ別に設定します。
    */
    void SetNspFile( const char* pNspFile ) NN_NOEXCEPT;

    /**
    * @brief 選択されたnspテキストを初期状態にクリアします。
    */
    void ClearScreenText() NN_NOEXCEPT;

private:

    class Header : public glv::Group
    {
    public:
        Header( glv::Label* pPageTitleLabel, const glv::Rect& rect ) NN_NOEXCEPT;
    };

    enum WriteState
    {
        WriteState_NotStarted,
        WriteState_Running,
        WriteState_Committed,
        WriteState_Completed,
        WriteState_Failed,
        WriteState_WaitUserAction
    };

private: // メンバ定数

    // シザーボックスの領域設定
    static const glv::space_t HeaderRegionLength;
    static const glv::space_t FooterRegionLength;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginLeft;
    static const glv::space_t ListMarginRight;
    static const glv::space_t ListMarginVertical;
    static const glv::space_t NspFileLeft;

    static const glv::Label::Spec LabelSpecNoNsp;

    static const std::string NspFileNothing;
    static const std::string HeaderName;
    static const std::string HeaderNameApplicationSelected;

private: // メンバ関数

    /**
     * @brief ダイアログを表示して確認を取った後、全 nsp のGameCard書き込みを行います。
     */
    void ConfirmSelector( const FilePropertyType* pProperty = nullptr ) NN_NOEXCEPT;

    /**
     * @brief 選択された nsp ファイルのGameCard書き込みを行います。
     */
    void WriteNspFile() NN_NOEXCEPT;

    /**
     * @brief GameCard書き込み進捗画面の Close ボタンや Next ボタン押下時のコールバック関数です。
     */
    void CloseButtonCallback() NN_NOEXCEPT;

    /**
     * @brief SD カード内の全 nsp ファイルのGameCard書き込みを行います。
     */
    void WriteNspFiles() NN_NOEXCEPT;

    /**
     * @brief nsp ファイルをGameCard書き込みします。
     */
    const nn::Result ExecuteWrite( const char* applicationFilePath, const char* patchFilePath ) NN_NOEXCEPT;

    /**
     * @brief GameCard書き込み結果を設定します。
     */
    void SetWriteResult( const nn::Result& result ) NN_NOEXCEPT;

    /**
     * @brief GameCard書き込みのトリガをかけるスレッドのメイン関数です。
     */
    static void WriteTriggerThreadFunc( void* args ) NN_NOEXCEPT;

    /**
     * @brief GameCard書き込みトリガスレッドを破棄します。
     */
    void DestroyWriteTriggerThread() NN_NOEXCEPT;

    /**
    * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
    *
    * @param[in]   string          表示内容
    * @param[in]   maxLength       表示する文字の上限数
    * @return      整形された文字列
    */
    const std::string ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT;

private: // メンバ変数

    devmenu::Button                                 m_ButtonToCatalog;
    devmenu::Button                                 m_ButtonWrite;
    glv::ScissorBoxView                             m_ListContainer;
    glv::Label*                                     m_pHeaderName;
    glv::Label*                                     m_pLabelApplication;
    glv::Label*                                     m_pLabelPatch;
    char                                            m_ApplicationFileName[MaxDirectoryNameLength];
    char                                            m_PatchFileName[MaxDirectoryNameLength];
    select::FileListSelectorScene*                  m_pFileListSelectorScene;

    devmenu::Button                                 m_ButtonToApplicationList;
    devmenu::Button                                 m_ButtonToPatchList;

    Scene*                                          m_pParentScene;
    ApplicationCatalogPage*                         m_pParentPage;

    // GameCard書き込み状況表示関連
    nn::os::ThreadType                              m_WriteTriggerThread;   //!< GameCard書き込みトリガをかけるスレッド
    nn::os::ThreadType                              m_WriteExecutionThread; //!< GameCard書き込み実行スレッド
    nn::os::Mutex                                   m_Mutex;                //!< 排他ロック
    nn::os::Tick                                    m_StartTick;            //!< GameCard書き込み開始時点の SystemTick 値
    WriteView*                                      m_pWriteView;           //!< GameCard書き込みの進捗表示用のモーダル
    WriteState                                      m_WriteState;           //!< GameCard書き込み状態
};

}}} // ~namespace devmenu::application::write, ~namespace devmenu::application, ~namespace devmenu
