﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenu_Config.h"
#include "DevMenu_Common.h"

namespace devmenu { namespace application {

// 相互参照を防ぐためのクラス参照
class ApplicationCatalogPage;

namespace select {

const int64_t MaxDirectoryNameLength = 300;  //!< SD カード内アプリケーション名の最大文字数（終端文字を含む）

/**
* @brief SD カード内のアプリケーションプロパティ型です。
*/
struct FilePropertyType
{
    /**
    * @brief SD カード内のアプリケーションリスト用プロパティデータを構築します。
    */
    void Prepare( const nn::fs::DirectoryEntry& directoryEntry, const char* pFilePath ) NN_NOEXCEPT
    {
        BuildUtf16< MaxDirectoryNameLength >( name, "%s%s", pFilePath, directoryEntry.name );
        const auto delimitedSizeStr = GetDelimitedNumberString( directoryEntry.fileSize );
        BuildUtf16< 32 >( size, "%s", delimitedSizeStr.c_str() );
    }

    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];
    std::string longName;
};

/**
 * @brief リストビュー
 */
class SdCardListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit SdCardListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size( 20.f );
    }

protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds( outWidth, outHeight, item.name );
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        font.getBounds( outWidth, outHeight, item.size );
        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render( item.name, posName, contentRegion.top() );

        font.getBounds( outWidth, outHeight, item.size );
        const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
        const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
        font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
    }
};

/**
 * @brief SDファイル一覧シーン
 */
class FileListSelectorScene : public Scene
{
public:
    /**
     * @brief コンストラクタです。
     */
    FileListSelectorScene( ApplicationCatalogPage* pParentPage, Scene* pParentScene = nullptr ) NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    /**
    * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief SD カードをマウントしファイルリストを作成。
     */
    const nn::Result MakeNspList() NN_NOEXCEPT;

    /**
    * @brief NSPファイルが選択されているかを判定します。
    */
    bool IsNspFileSelected() NN_NOEXCEPT;

    /**
    * @brief ファイル選択情報を初期化します。
    */
    void InitializeFileSelected() NN_NOEXCEPT;

    /**
    * @brief 選択されたNSPファイルパスを取得します。
    */
    const char* GetSelectedNspFilePath() NN_NOEXCEPT;

    /**
    * @brief コンテンツタイプを設定
    */
    void SetContentType( nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT;

    /**
    * @brief コンテンツタイプを取得
    */
    nn::ncm::ContentMetaType GetContentType() NN_NOEXCEPT;

private: // メンバ定数

    // シザーボックスの領域設定
    static const glv::space_t HeaderRegionLength;
    static const glv::space_t FooterRegionLength;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginLeft;
    static const glv::space_t ListMarginRight;
    static const glv::space_t ListMarginVertical;

    static const glv::Label::Spec LabelSpecNoNsp;

private: // メンバ関数

    /**
     * @brief SD カード内のアプリケーションプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT;

    /**
     * @brief 指定したパス以下のファイルをリストにして取得します。
     */
    static const nn::Result EnumerateFiles( std::vector< FilePropertyType >* pOutEntryList, const char* directoryPath ) NN_NOEXCEPT;

    /**
    * @brief 選択された nsp ファイルパス取得及び画面遷移を行います。
    */
    void GetSelectedFilePathAndChangeScene() NN_NOEXCEPT;

    /**
     * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
     *
     * @param[in]   string          表示内容
     * @param[in]   maxLength       表示する文字の上限数
     * @return      整形された文字列
     */
    static const std::string ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT;

    /**
    * @brief       コンテンツタイプをチェックする
    *
    * @param[out]  pOutValue        チェック結果(true:コンテンツタイプが一致  false:コンテンツタイプが不一致)
    * @param[out]  pOutErrorMessage エラーメッセージ
    * @return      チェックの成否を Result 値で返します。
    */
    nn::Result CheckContentMeta( bool* pOutValue, std::string* pOutErrorMessage ) NN_NOEXCEPT;

private: // メンバ変数

    devmenu::Button                                 m_ButtonToCatalog;
    glv::ScissorBoxView                             m_ListContainer;
    SdCardListView::CollectionType                  m_NspList;
    SdCardListView                                  m_NspListView;
    glv::Label                                      m_LabelNoNsp;
    Scene*                                          m_pParentScene;
    ApplicationCatalogPage*                         m_pParentPage;
    char                                            m_FileName[MaxDirectoryNameLength];
    bool                                            m_IsNspFileSelected;
    nn::ncm::ContentMetaType                        m_ContentMetaType;
};

}}} // ~namespace devmenu::application::select, ~namespace devmenu::application, ~namespace devmenu
