﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Applications/DevMenu_ApplicationsSceneDetail.h"
#include "Common/DevMenu_CommonScene.h"
#include "Common/DevMenu_CommonScrollBox.h"
#include "DevMenu_Common.h"
#include "DevMenu_RootSurface.h"

//#define DEBUG_DATA

namespace devmenu { namespace application {

class ApplicationCatalogPage;

// 相互参照を防ぐためのクラス参照
class InstallScene;
class DeleteContentsScene;

namespace write {
    class WriteScene;
}

namespace select {
    class FileListSelectorScene;
}

class ApplicationListModel
{
public:
    ApplicationListModel() NN_NOEXCEPT;

    enum ExitReason : nn::Bit8
    {
        ExitReason_Normal,
        ExitReason_Canceled
    };

    struct ControlProperty
    {
        char name[ 64 ];
        char displayVersion[ 16 ];
        bool isNameCut;
    };

    struct Item
    {
        nn::ns::ApplicationView view;
        nn::util::optional< ControlProperty > property;
        bool requiresUpdateControl;
    };

public:
    static const nn::Result GetApplicationView( nn::ns::ApplicationView* pOutView, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;
    static const nn::Result GetApplicationControlProperty( nn::util::optional<ControlProperty>* pOutProperty, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

    ExitReason Refresh() NN_NOEXCEPT;
    ExitReason UpdateIfNecessary() NN_NOEXCEPT;

    int64_t GetRefreshedCount() const NN_NOEXCEPT;

    const std::vector< Item >& Get() NN_NOEXCEPT;
    void ResetCancel() NN_NOEXCEPT;
    void Cancel() NN_NOEXCEPT;

private:
    void UpdateOne( ApplicationListModel::Item* pItem ) NN_NOEXCEPT;
    nn::util::optional< Item > Find( const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT;

private:
    nn::os::Mutex             m_UpdateMutex;
    std::vector< Item >       m_ItemList;
    int64_t                   m_RefreshedCount;
    std::atomic_bool          m_IsCanceled;
};

/**
 * @brief カタログのシーン
 */
class CatalogScene : public Scene, NotificationMessageReceiver
{
public:
    /**
     * @brief コンストラクタ
     */
    CatalogScene( const AbstractOperators& op, ApplicationCatalogPage* pParentPage, glv::Rect rect, InstallScene* pChildScene = nullptr ) NN_NOEXCEPT;

    /**
     * @brief InstallScene 遷移ボタン押下時のコールバック関数
     */
    void SwitchToInstallScene() NN_NOEXCEPT;

    /**
    * @brief WriteScene 遷移ボタン押下時のコールバック関数
    */
    void SwitchToWriteScene() NN_NOEXCEPT;

    /**
     * @brief Game Card Image Hash ボタン押下時のコールバック関数
     */
    void ShowGameCardImageHash() NN_NOEXCEPT;

    /**
     * @brief ストレージ容量の更新
     */
    void UpdateStorageSize( bool isSdCardCheckRequired = true, const nn::Result& checkResult = nn::ResultSuccess() ) NN_NOEXCEPT;

    /**
     * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief Install シーンを設定します。
     */
    void SetInstallScene( InstallScene* pScene ) NN_NOEXCEPT
    {
        m_pInstallScene = pScene;
    }

    /**
    * @brief Write シーンを設定します。
    */
    void SetWriteScene( write::WriteScene* pScene ) NN_NOEXCEPT
    {
#if !defined( NN_DEVMENULOTCHECK )
        m_pWriteScene = pScene;
#else
        NN_UNUSED( pScene );
#endif
    }

    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    void ApplyApplicationInfoToList() NN_NOEXCEPT;

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    void OnChangeIntoBackground() NN_NOEXCEPT;

    void OnChangeIntoForeground() NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE;

    void UpdateApplicationListAndStorageSizeIfRequired() NN_NOEXCEPT;

    void StartUpdateThread() NN_NOEXCEPT;

    void StopUpdateThread() NN_NOEXCEPT;

    void CancelUpdateOfApplicationList() NN_NOEXCEPT;

    const ApplicationListModel::Item* GetSelectedApplicationProperty() const NN_NOEXCEPT { return m_pSelectedApplicationProperty; }

    const nn::ncm::ApplicationId& GetSelectedApplicationId() const NN_NOEXCEPT { return m_SelectedApplicationId; }

    void DisplayInstallTicketDialog() NN_NOEXCEPT;

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

    void LaunchOrResumeApplication( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

#endif //defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

private:

    /**
     * @brief View for the layout and information of application on the application list
     */
    class ApplicationTileView : public glv::View
    {
    public:
        ApplicationTileView( glv::space_t width, glv::space_t height ) NN_NOEXCEPT;

        char* GetApplicationStateString() NN_NOEXCEPT;

        void UpdateApplicationTileValues( const ApplicationListModel::Item& item ) NN_NOEXCEPT;

        virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;

    private:
        const char* GetApplicationDownloadStateString( nn::ns::ApplicationDownloadState state ) NN_NOEXCEPT;

        const char* GetApplyDeltaStateString( nn::ns::ApplicationApplyDeltaState state ) NN_NOEXCEPT;

        ApplicationIcon     m_ApplicationIcon;
        IconLabel*          m_pGameCardIcon;
        glv::Label*         m_pApplicationName;
        glv::Label*         m_pApplicationState;
        glv::Label*         m_pApplicationId;
        glv::Label*         m_pApplicationVersion;
        glv::Table*         m_pApplicationNameTable;
        glv::Table*         m_pInfoTable;
        char                m_StateBuffer[ 128 ];
        ApplicationListModel::Item m_Item;
    };

    class TileContainer : public glv::View
    {
    public:
        TileContainer( CatalogScene* pParent, glv::space_t width, glv::space_t height, const ApplicationListModel::Item& item, ApplicationTileView* pApplicationTile ) NN_NOEXCEPT;

        ~TileContainer() NN_NOEXCEPT;

        virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;

        virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;

    private:
        const ApplicationListModel::Item&   m_Item;
        ApplicationTileView*         m_pApplicationTile;
        CatalogScene*                m_pParent;
    };

    class Footer : public glv::Group
    {
    public:
        Footer( CatalogScene* pParent, const glv::Rect& rect ) NN_NOEXCEPT;
        ~Footer() NN_NOEXCEPT;

        void UpdateLaunchButtonText( const char* text ) NN_NOEXCEPT;
        void UpdateLaunchButtonFocus( bool isEnabled ) NN_NOEXCEPT;
        void IncreaseLaunchButtonSize( glv::space_t increasedX, glv::space_t increasedY ) NN_NOEXCEPT;
        void UpdateStorageSize( bool isSdCardCheckRequired, const nn::Result& checkResult ) NN_NOEXCEPT;

    private:

        /**
         * @brief 指定した StorageId の空き容量・総容量の取得
         */
        const nn::Result GetStorageSize( int64_t* pOutFreeSpaceSize, int64_t* pOutTotalSpaceSize, nn::ncm::StorageId storageId ) NN_NOEXCEPT;

        CatalogScene*     m_pParent;
        devmenu::Button   m_LaunchButton;
        devmenu::Button   m_OptionButton;
        glv::Label        m_BuiltInUserSize;
        glv::Label        m_SdCardSize;
    };

private:
    // シザーボックスの領域設定
    static const glv::space_t HeaderRegion;
    static const glv::space_t FooterRegion;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginLeft;
    static const glv::space_t ListMarginRight;

    static const int          MaxNumberOfVisibleApplicationTiles = 6;
    static const int          MaxNumberOfApplicationsSupported = 2048;

private:

    /**
     * @brief  Functions for use in ApplicationTileView
     */
    void SetSelectedApplicationProperty( const ApplicationListModel::Item& item ) NN_NOEXCEPT;

    void SetSelectedApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

    void SetPreviousSelectedApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

    void UpdateFooterLaunchButtonText( const char* text ) NN_NOEXCEPT;

    void UpdateFooterLaunchButtonFocus( bool isEnabled ) NN_NOEXCEPT;

    /**
     * @brief リスト選択結果の受信
     */
    static void OnApplicationTileViewClickNotification( const glv::Notification& notification ) NN_NOEXCEPT;

    /**
     * @brief Application List を更新可能かを判定して返す
     */
    bool CanUpdateItemList() NN_NOEXCEPT;

    /**
     * @brief       AEメッセージ通知受信コールバック。
     *
     * @param[in]   message 受信した通知メッセージ。
     *
     */
    virtual void OnNotificationMessageReceived( NotificationMessageReceiver::Message message ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief チケットのダウンロード（ロットチェック用）
     */
    nn::Result DownloadAllInstalledTicket( nn::ncm::StorageId storageId, nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT;

private:
    const AbstractOperators&            m_Op;
    ApplicationCatalogPage*             m_pParentPage;
    InstallScene*                       m_pInstallScene;
#if !defined( NN_DEVMENULOTCHECK )
    write::WriteScene*                  m_pWriteScene;
#endif

    ApplicationListModel                m_Model;
    int64_t                             m_ModelRefreshedCount;
    nn::os::ThreadType                  m_UpdateThread;
    nn::os::Event                       m_StopEvent;
    std::vector< ApplicationListModel::Item > m_ListItemCopy;
    bool                                m_IsUpdateThreadRunning;
    bool                                m_IsUpdateOfApplicationListEnabled;
    bool                                m_IsRefreshOfApplicationListRequired;

    Button*                             m_pInstallSceneButton;
#if !defined( NN_DEVMENULOTCHECK )
    Button*                             m_pWriteSceneButton;
#endif
    glv::View*                          m_pContainer;
    ScrollableBoxView*                  m_pScrollContainer;
    std::vector< ApplicationTileView* > m_ApplicationTilesList;
    glv::Label*                         m_pLabelNoItem;
    const ApplicationListModel::Item*   m_pSelectedApplicationProperty;
    Footer                              m_Footer;

    nn::ncm::ApplicationId              m_SelectedApplicationId;
    nn::ncm::ApplicationId              m_PreviousSelectedApplicationId;

    std::function< void() >             m_ListUserActionCallback;
    bool                                m_IsNotifiedByTouch;
};

/**
 * @brief アプリケーション一覧機能のページです。
 */
class ApplicationCatalogPage : public devmenu::Page
{
public:
    /**
     * @brief コンストラクタです。
     */
    ApplicationCatalogPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT;

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief バックグラウンド遷移処理です。ページがアクティブな場合にのみ呼ばれます。
     */
    virtual void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief フォアグラウンド遷移処理です。ページがアクティブな場合にのみ呼ばれます。
     */
    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief シーンを切替えます。
     */
    void SwitchScene( devmenu::Scene* pCurrentScene, devmenu::Scene* pNextScene, bool isBack ) NN_NOEXCEPT;

    /**
     * @brief 次にフォーカス表示可能なビューを取得します。
     */
    glv::View* NextFocusableView( glv::View* pBase, int flags ) NN_NOEXCEPT;

    /**
     * @brief 自身がフォーカス可能で Active な状態であるかを判定して返します。
     */
    inline bool IsInFocusState() NN_NOEXCEPT
    {
        return GetRootSurfaceContext()->IsInFocusState( this );
    }

private:

    struct Operators : public AbstractOperators
    {
        static const nn::ncm::ApplicationId InvalidApplicationId;
        static const nn::ns::ApplicationView InvalidApplicationView;

        ApplicationCatalogPage& parent;

        explicit Operators( ApplicationCatalogPage& parent ) NN_NOEXCEPT;

        virtual void ShowLegalInfo( const char* kindOfDocument, const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT final NN_OVERRIDE;

        virtual void SelectApplication( const nn::ns::ApplicationView& applicationView ) const NN_NOEXCEPT final NN_OVERRIDE;

        virtual void ManageApplication( const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT final NN_OVERRIDE;

        virtual void OpenApplicationInfo() const NN_NOEXCEPT final NN_OVERRIDE;

        virtual void CloseApplicationInfo() const NN_NOEXCEPT final NN_OVERRIDE;

        virtual void OpenAutoBootSetting() const NN_NOEXCEPT final NN_OVERRIDE;

        virtual devmenu::RootSurfaceContext* GetRootSurfaceContext() const NN_NOEXCEPT final NN_OVERRIDE;
    } m_Op;

private:
    CatalogScene*                               m_pCatalogScene;
    InstallScene*                               m_pInstallScene;
    DeleteContentsScene*                        m_pDeleteContentsScene;
    DetailScene*                                m_pApplicationInfoScene;
#if !defined( NN_DEVMENULOTCHECK )
    write::WriteScene*                          m_pWriteScene;
#endif
};

}} // ~namespace devmenu::application, ~namespace devmenu
