﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cfloat>

#include <glv_core.h>
#include <glv_ScissorBoxView.h>
#include <glv_viewutility.h>

#include "../DevMenu_Common.h"
#include "DevMenu_CommonCheckBox.h"

namespace devmenu {

class ScrollableBoxView : public glv::ScissorBoxView
{
public:
    ScrollableBoxView( const char* tableLayoutStr, const glv::Rect& rect, glv::space_t scrollBarWidth = 5.0f, glv::space_t marginLeft = 0.0f, glv::space_t marginTop = 0.0f ) NN_NOEXCEPT;

    /**
     * @brief   Constructor with setting ancestor view.
     * @detail  Use this constructor if it is required to search a next fucusable view outside ScrollableBoxView.
     */
    ScrollableBoxView( const char* tableLayoutStr, const glv::Rect& rect,  glv::View* pAncestorView, glv::space_t scrollBarWidth = 5.0f, glv::space_t marginLeft = 0.0f, glv::space_t marginTop = 0.0f ) NN_NOEXCEPT;

    ~ScrollableBoxView() NN_NOEXCEPT
    {
        delete m_pTable;
    }

    glv::Scroll& mode( glv::Scroll::Mode mode ) NN_NOEXCEPT
    {
        return m_Scroll.mode( mode );
    }

    const glv::Rect& GetInnerRect() const NN_NOEXCEPT
    {
        return m_InnerRect;
    }

    glv::View& operator << ( glv::View& newChild ) NN_NOEXCEPT
    {
        return operator << ( &newChild );
    }

    glv::View& operator << ( glv::View* pNewChild ) NN_NOEXCEPT;

    /**
     * @brief   Scrool to the top position.
     * @detail  Scrool to the top position and clear the current focused view.
     */
    void ScrollToTop() NN_NOEXCEPT;

    /**
     * @brief   Scrool to the specified position.
     */
    void ScrollTo( glv::space_t position ) NN_NOEXCEPT;

    /**
     * @brief   Scrool to the specified View.
     */
    void ScrollTo( glv::View* pTargetView ) NN_NOEXCEPT;

    /**
     * @brief   Refresh ScrollableBoxView.
     * @detail  Recreate the inner table and register it to ScrollableBoxView.
     */
    void Refresh() NN_NOEXCEPT;

    /**
     * @brief   Arrange the inner table.
     * @detail  Arrange the inner table and disable FocusHighlight property of ScrollBoxView
     *          if the table height is less than ScroollBox View height.
     */
    void ArrangeTable() NN_NOEXCEPT;

    /**
     * @brief   Update rect values.
     * @detail  Set rect and then update the inner rect for display area and the rect of glv::Scroll variable.
     */
    void SetRectAndUpdateInnerSize( glv::space_t width, glv::space_t height ) NN_NOEXCEPT;

    virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;

private:
    static constexpr glv::space_t InnerRectMarginHorizontal = 32.0f;
    static constexpr glv::space_t InnerRectMarginVertical = 10.0f;
    static constexpr glv::space_t ForcedScrollRate = 0.25f;
    static constexpr glv::space_t AcceptableFloatDelta = FLT_EPSILON * 1000;

private:
    glv::space_t GetAdjustedTopPosition( const glv::View* const pView ) NN_NOEXCEPT;

    /**
     * @brief      Update rect values.
     * @detail     Set rect and then update the inner rect for display area and the rect of glv::Scroll variable.
     *             If the focused view is ScrollableBoxView itself and the valid ancestor view is set,
     *             a root target for search is the ancestor view. In other cases, the root view is ScrollableBoxView itself.
     *
     * @param[in]  pFocusedView   Current focused view
     * @param[in]  flags
     *
     * @return     Returns the next focusable view. Returns nullptr if it doesn't exist,
     */
    const glv::View* GetNextFocusableChild( const glv::View* pFocusedView, glv::MoveFlags flags ) NN_NOEXCEPT;

    /**
     * @brief   Check the view is displayed.
     *
     * @return  Returns true if the view is in the visible region or a part of the view
     */
    bool IsViewDisplayed( const glv::View* const pView ) NN_NOEXCEPT;

    /**
     * @brief   Check the view is displayed.
     * @return  Returns true if the view is in the visible region or a part of the view
     */
    bool IsDescendant( const glv::View* const pTargetView ) const NN_NOEXCEPT;

    /**
     * @brief   Move focus with input.
     *
     * @param[in]  MoveFlags  Input key type of user operation
     * @param[in]  isRepeat   Whether the input is repeat
     *
     * @return  Returns true if it is necessary to do onEvent of parent view.
     */
    bool Move( glv::MoveFlags flags, bool isRepeat ) NN_NOEXCEPT;

    /**
     * @brief  Required to adjust the scroll position
     */
    void OnFocused( const glv::View* const pFocused ) NN_NOEXCEPT;

    // This is an internal helper class which will set callbacks for focus events on the children of the objects that are
    // added to the ScrollableBoxView.
    class ChildrenFocusTraversalAction : public glv::View::TraversalAction
    {
    public:
        explicit ChildrenFocusTraversalAction( ScrollableBoxView* pParent ) NN_NOEXCEPT;
        virtual bool operator()( glv::View* pView, int depth ) NN_NOEXCEPT NN_OVERRIDE;
    private:
        ScrollableBoxView* m_pScrollableBoxView;
    };

private:
    glv::Scroll         m_Scroll;
    std::string         m_TableLayoutString;
    glv::Table*         m_pTable;
    glv::Rect           m_InnerRect;
    const glv::View*    m_pFocusedView;
    glv::View*          m_pAncestorView; // Root view for searching next focusable view
    ChildrenFocusTraversalAction m_ChildrenFocusTraversalAction;
};

} // ~namespace devmenu

