﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <string>

#include <glv_core.h>

#include "../DevMenu_ModalView.h"

namespace devmenu {

class Page;

/*********************************
 * class Refreshable
 *********************************/

class Refreshable
{
public:
    void RequireRefresh() NN_NOEXCEPT
    {
        m_RefreshOnNext.store( true );
    }

    bool IsRefreshRequired() const NN_NOEXCEPT
    {
        return m_RefreshOnNext.load();
    }

    void ClearRefreshRequest() NN_NOEXCEPT
    {
        m_RefreshOnNext.store( false );
    }

    virtual void Refresh() NN_NOEXCEPT = 0;

protected:
    Refreshable() NN_NOEXCEPT
        : m_RefreshOnNext( false )
    {
    }

private:
    std::atomic< bool > m_RefreshOnNext;

};

/*********************************
 * class Scene
 *********************************/

class Scene
    : public glv::Group
    , public Refreshable
{
public:
   /**
    * @brief 子孫 View であるかを判別します。
    * @details シーンに追加された View であれば真、そうでない場合は偽を返します。
    */
    bool IsDescendant( const glv::View* const pView ) const NN_NOEXCEPT;

    /**
     * @brief 最初にフォーカスされる View を取得します。
     * @details 対象ビューがシーンに設定されていない場合は nullptr を返します。
     */
    inline glv::View* GetFirstFocusTargetView() const NN_NOEXCEPT
    {
        return m_pFirstFocusTargetView;
    }

    /**
     * @brief 最後にフォーカスされた View を取得します。
     * @details 対象ビューがシーンに設定されていない場合は nullptr を返します。
     */
    glv::View* GetLastFocusedView() const NN_NOEXCEPT;

    /**
     * @brief   最後にフォーカスされた View を設定します。
     */
    inline void SetLastFocusedView( glv::View* const pView ) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL( pView );
        m_pLastFocusedView = pView;
    }

    /**
     * @brief   Back ボタン押下時に実行されるコールバック関数を設定します。
     */
    inline void SetBackButtonCallback( const std::function< void() >& callback ) NN_NOEXCEPT
    {
        m_BackButtonCallback = callback;
    }

    /**
     * @brief   シーンを切り替えます。
     */
    void SwitchScene( int nextSceneIndex, bool isBack = false ) NN_NOEXCEPT;

    /**
    * @brief    シーンに入る際の処理です。必要に応じてオーバーライドし、Pageにて呼び出してください。
    */
    virtual void OnEnterScene() NN_NOEXCEPT
    {
        // NOOP
    }

    /**
    * @brief    シーンから抜ける際の処理です。必要に応じてオーバーライドし、Pageにて呼び出してください。
    */
    virtual void OnLeaveScene() NN_NOEXCEPT
    {
        // NOOP
    }

    /**
    * @brief    シーンレンダラ前のループ処理です。必要に応じてオーバーライドし、Pageにて呼び出してください。
    */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    /**
    * @brief    シーンレンダラ後のループ処理です。必要に応じてオーバーライドし、Pageにて呼び出してください。
    */
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    /**
    * @brief    ページがアクティブになった際に呼ばれる処理です。必要に応じてオーバーライドし、Pageにて呼び出してください。
    */
    virtual void OnActivatePage() NN_NOEXCEPT
    {
        // NOOP
    }

protected:
    // シザーボックスの領域設定
    static constexpr glv::space_t HeaderRegion = 32.0f;
    static constexpr glv::space_t FooterRegion = 32.0f;

protected:

    Scene( Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed = false, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT;

    explicit Scene( const glv::Rect& rect, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT;

    /**
     * @brief   最初にフォーカスされる View を設定します。
     */
    inline void SetFirstFocusTargetView( glv::View* const pView ) NN_NOEXCEPT
    {
        m_pFirstFocusTargetView = pView;
    }

    /**
     * @brief   RootSurfaceContext のインスタンスを取得します。
     */
    RootSurfaceContext* GetRootSurfaceContext() const NN_NOEXCEPT;

    /**
     * @brief   ModalView の表示を開始します。
     */
    void StartModal( ModalView* pModalView, bool isDisposable = true, bool isDelayExecution = false, bool isAutomaticExitAtBackgroundEnabled = false, bool isFloatingApplicationLaunchBlocked = false ) NN_NOEXCEPT;

    /**
     * @brief   指定した View をフォーカスします。
     */
    void SetFocus( glv::View* const pView ) NN_NOEXCEPT;

    /**
     * @brief   Back ボタン押下時のコールバック関数です。
     */
    void OnBackButtonPressed() NN_NOEXCEPT
    {
        if ( nullptr != m_BackButtonCallback )
        {
            m_BackButtonCallback();
        }
    }

private:
    void Initialize() NN_NOEXCEPT;

private:
    Page* const m_pParentPage;
    glv::View*  m_pFirstFocusTargetView; // 表示開始後に最初にフォーカスするビュー
    glv::View*  m_pLastFocusedView;  // 別シーンに遷移する直前のフォーカス
    std::function< void() > m_BackButtonCallback;
};

/*********************************
 * class ProgressModalSceneBase
 *********************************/
class ProgressModalSceneBase : public Scene
{
public:
    void StartProgressModalThread(
        const std::string& executionStr,
        const std::function< nn::Result( bool* ) >& startCallback,
        const std::function< void( const nn::Result&, bool ) >& completeCallback = nullptr ) NN_NOEXCEPT;

    /**
     * @brief   ModalView にメッセージをセットします。
     */
    void SetModalViewMessage( const std::string& message ) NN_NOEXCEPT;

protected:
    ProgressModalSceneBase() NN_NOEXCEPT
        : Scene( nullptr, glv::Rect( 0.0f ) )
        , m_ExecutionState( ExecutionState_None )
        , m_ProgressModalView( false )
        , m_ProgressMessageBase( "" )
        , m_IsExecutionSuccess( false )
        , m_StartCallback( nullptr )
        , m_CompletionCallback( nullptr )
    {
    }

    ProgressModalSceneBase( Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed = false, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT
        : Scene( pParentPage, rect, isDefaultbackButtonCallbackUsed, backButtonCallback )
        , m_ExecutionState( ExecutionState_None )
        , m_ProgressModalView( false )
        , m_ProgressMessageBase( "" )
        , m_IsExecutionSuccess( false )
        , m_StartCallback( nullptr )
        , m_CompletionCallback( nullptr )
    {
        NN_ASSERT_NOT_NULL( pParentPage );
    }

    /**
     * @brief   ModalView を表示します。
     */
    void StartModalForProgress( bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT;

    /**
     * @brief   ProgressModalView を更新し、処理が完了したらスレッドを破棄します。
     */
    void UpdateExecutionProgress() NN_NOEXCEPT;

private:
    enum ExecutionState
    {
        ExecutionState_None,
        ExecutionState_Preparing,
        ExecutionState_Running,
        ExecutionState_Completed
    };

private:
    static constexpr int ProgressMessageLoopCount = 4;
    static constexpr int ThreadStackSize = 64 * 1024;

private:
    /**
     * @brief   StartProgressModalThreadで 呼ばれるスレッド処理関数です。
     */
    void ProgressDisplayThreadFunction() NN_NOEXCEPT;

    /**
     * @brief   サブスレッド処理が完了した際に UpdateExecutionProgress() 内で呼ばれるコールバックです。
     */
    ExecutionState OnCompleteExecuteProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT;

private:
    nn::os::ThreadType                                m_ProgressDisplayThread;
    ExecutionState                                    m_ExecutionState;
    ProgressModalView                                 m_ProgressModalView;
    std::string                                       m_ProgressMessageBase;
    bool                                              m_IsExecutionSuccess;
    nn::Result                                        m_ExecutionResult;
    std::function< nn::Result( bool* ) >              m_StartCallback;
    std::function< void( const nn::Result&, bool ) >  m_CompletionCallback;
};

} // ~namespace devmenu
