﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <string>
#include <vector>

#include <glv_core.h>
#include <glv_textview.h>

#include <nn/util/util_Span.h>

#include "../DevMenu_RootSurface.h"

namespace devmenu {

/*********************************
 * class DropDown, WideDropDown
 *********************************/

class DropDownBase : public glv::DropDown
{
public:
    DropDownBase( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
        : glv::DropDown( rect, textSize )
    {
        mItemList.font().size( textSize );
        anchor( glv::Place::TR ).pos( -width() - 24.0f, 0.0f );
    }

    virtual const char* className() const { return "DevMenuDropDown"; }

    bool IsShowingList() const NN_NOEXCEPT
    {
        return mItemList.visible();
    }

    virtual bool onEvent( glv::Event::t e, glv::GLV& g ) NN_NOEXCEPT;
};

class WideDropDown : public glv::WideDropDown
{
public:
    WideDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
        : glv::WideDropDown( rect, textSize )
    {
        mItemList.font().size( textSize );
        anchor( glv::Place::TR ).pos( -width() - 24.0f, 0.0f );
    }

    virtual const char* className() const { return "DevMenuDropDown"; }

    bool IsShowingList() const NN_NOEXCEPT
    {
        return mItemList.visible();
    }

    virtual bool onEvent( glv::Event::t e, glv::GLV& g ) NN_NOEXCEPT;
};

/*******************************************
 * class DropDownComparableKeyValue
 *******************************************/

template < typename T >
class DropDownComparableKeyValue : public DropDownBase
{
protected:
    typedef std::pair< const T, const char* > valuePair;

public:
    DropDownComparableKeyValue(
        const glv::Rect& rect, float textSize,
        std::function< void( void*, devmenu::DropDownBase* ) > registerCallback, std::function< void( void* ) > getCallback, std::function< void( const T&, T ) > setCallback = nullptr
    ) NN_NOEXCEPT
        : DropDownBase( rect, textSize )
        , m_RegisterCallback( registerCallback )
        , m_GetCallback( getCallback )
        , m_SetCallback( setCallback )
    {
        mItemList.font().size( textSize * 1.0f );
        RegisterDropDownItems();
        Refresh();
        attach( []( const glv::Notification& notification )->void { notification.receiver< DropDownComparableKeyValue >()->UpdateSettingsValue(); }, glv::Update::Action, this );
    }

    virtual void Refresh() NN_NOEXCEPT
    {
        SetDropDownValue( GetSettingsValue() );
    }

    virtual const char* GetDropDownItemString( const T& value ) NN_NOEXCEPT
    {
        NN_ASSERT( 0 < m_Values.size() );

        for ( const auto& iter: m_Values )
        {
            if ( value == iter.first )
            {
                return iter.second;
            }
        }
        NN_ABORT( "Must not come here\n" );
    }

protected:
    virtual T GetSettingsValue() NN_NOEXCEPT
    {
        T value;
        m_GetCallback( &value );
        return value;
    }

    virtual void SetDropDownValue( const T& value ) NN_NOEXCEPT
    {
        for ( const auto& iter: m_Values )
        {
            if ( value == iter.first )
            {
                setValue( iter.second );
                return;
            }
        }
        NN_ABORT( "Must not come here\n" );
    }

    virtual void RegisterDropDownItems() NN_NOEXCEPT
    {
        m_RegisterCallback( &m_Values, this );
    }

    virtual void UpdateSettingsValue() NN_NOEXCEPT
    {
        auto& selectedValue = m_Values.at( mSelectedItem ).first;
        auto currentValue = GetSettingsValue();
        if ( selectedValue != currentValue && nullptr != m_SetCallback )
        {
            m_SetCallback( selectedValue, currentValue );
        }
    }

    std::function< void( void* outValues, devmenu::DropDownBase* pDropDown ) > m_RegisterCallback;
    std::function< void( void* outValue ) > m_GetCallback;
    std::function< void( const T& selectedValue, T currentValue ) > m_SetCallback;
    std::vector< valuePair > m_Values;
};


/*******************************************
 * class DropDownStringKeyValueBase
 *******************************************/
class DropDownStringKeyValueBase : public DropDownBase
{
protected:
    typedef std::pair< std::string, std::string > valuePair;

public:
    DropDownStringKeyValueBase( const glv::Rect& rect, float textSize, const std::function< void( const std::string& , const std::string& ) >& callback ) NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    virtual const std::string& GetDropDownItemString( const std::string& value ) NN_NOEXCEPT;

protected:
    virtual std::string GetSettingsValue() NN_NOEXCEPT = 0;

    virtual void SetDropDownValue( const std::string& value ) NN_NOEXCEPT;

    virtual void RegisterDropDownItems() NN_NOEXCEPT = 0;

    virtual void UpdateSettingsValue() NN_NOEXCEPT;

    std::function< void( const std::string& selectedValue, std::string currentValue ) > m_Callback;
    std::vector< valuePair > m_Values;
};

/*********************************
* class SceneSelectDropDown
*********************************/
class SceneSwitchDropDown : public DropDownBase
{
public:
    static constexpr glv::space_t DefaultWidth = 320.f;
    static constexpr glv::space_t DefaultHeight = 35.f;
    static constexpr glv::space_t DefaultHorizontalMargin = 16.f;
    static constexpr glv::space_t DefaultVerticalMargin = 8.f;

    /**
        @brief  Create a dropdown component which calls pParent->SwitchScene(selectedItem) when the selected item changes.

        @param[in] pParent              The page to add this item.
        @param[in] sceneTitles          The titles of the scenes. The order must match pParent->GetScene(int sceneIndex),
                                        meaning sceneTitles[0] must be pParent->GetScene(0)'s title, sceneTitles[1] must be pParent->GetScene(1)'s title and so on.
        @param[in] selectedSceneIndex   The initial title index to display.
        @param[in] rect                 The place and size of this item. The anchor is fixed to glv::Place::t::TL.
        @param[in] textSize             The font size.
    */
    SceneSwitchDropDown(
        devmenu::Page* pParent,
        nn::util::Span<const char*> sceneTitles,
        int selectedSceneIndex = 0,
        const glv::Rect& rect = glv::Rect(DefaultHorizontalMargin, DefaultVerticalMargin, DefaultWidth, DefaultHeight),
        float textSize = CommonValue::InitialFontSize
    ) NN_NOEXCEPT
        : DropDownBase(rect, textSize)
        , m_pParent(pParent)
    {
        NN_SDK_REQUIRES_GREATER(static_cast<int>(sceneTitles.size()), selectedSceneIndex);
        mItemList.font().size(textSize);

        for( const auto& title : sceneTitles )
        {
            this->addItem(title);
        }
        mSelectedItem = selectedSceneIndex;
        this->setValue(sceneTitles[selectedSceneIndex]);

        this->attach([](const glv::Notification& notification)->void {
            auto pSelf = notification.receiver< SceneSwitchDropDown >();
            pSelf->m_pParent->SwitchScene(pSelf->mSelectedItem);
        }, glv::Update::Action, this
        );
        this->anchor(glv::Place::t::TL).pos(rect.l, rect.t);
    }

private:
    devmenu::Page* m_pParent;
};

} // ~namespace devmenu
