﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../DevMenu_Config.h"
#include "../DevMenu_RootSurface.h"
#include <nn/bconfig/bconfig_Api.h>

namespace devmenu { namespace bconfig {

namespace {

typedef void ( *DropDownCallback ) ( void* arg );

void DefaultDropDownCallback( void* arg )
{
    NN_UNUSED( arg );
}

} // end of unnamed namespace

struct BootConfigSettingItem
{
    const               char* label;
    const               char* dropDownItems[ 6 ];
    DropDownCallback    dropDownCallback;
};


const int MaxBootConfigItem = 6;
BootConfigSettingItem bconfigItems[] = {
    {
        "Memory Mode (SDEV Only)",
#ifdef NN_DEVMENUSYSTEM
        { "Auto", "4GB", "6GB", "4GB (sdk)", "4GB (internal)", "6GB (internal)" },
#else
        { "Auto", "4GB    ", "6GB    ", nullptr, nullptr, nullptr },
#endif
        DefaultDropDownCallback,
    },
    {
        "Fill Memory with Non-Zero Value",
        { "Disabled", "Enabled", nullptr, nullptr, nullptr, nullptr },
        DefaultDropDownCallback,
    },
    {
        "User Exception Handler",
        { "Disabled", "Enabled", nullptr, nullptr, nullptr, nullptr },
        DefaultDropDownCallback,
    },
    {
        "Performance Monitoring Unit",
        { "Disabled", "Enabled", nullptr, nullptr, nullptr, nullptr },
        DefaultDropDownCallback,
    },
    {
        "Red Screen at System Error",
        { "Disabled", "Enabled", nullptr, nullptr, nullptr, nullptr },
        DefaultDropDownCallback,
    },
    {
        "Initial Tick Value",
        { "0", "7 days", "14 days", "30 days", "90 days", "<Custom>" },
        DefaultDropDownCallback,
    },
};

enum ItemIndex
{
    ItemIndex_MemoryMode = 0,
    ItemIndex_FillMemory,
    ItemIndex_UserException,
    ItemIndex_Pmu,
    ItemIndex_ShowError,
    ItemIndex_InitialTickValue,
};

enum MemoryMode
{
    MemoryMode_Auto         = 0x00,
    MemoryMode_4g           = 0x01,
    MemoryMode_4gInternal   = 0x02,
    MemoryMode_4gSdk        = 0x03,
    MemoryMode_6g           = 0x11,
    MemoryMode_6gInternal   = 0x12,
};

inline nn::Bit8 GetBootConfigMemoryMode( int index )
{
    switch ( index )
    {
    case 0:
        return bconfig::MemoryMode_Auto;
    case 1:
        return bconfig::MemoryMode_4g;
    case 2:
        return bconfig::MemoryMode_6g;
    case 3:
        return bconfig::MemoryMode_4gSdk;
    case 4:
        return bconfig::MemoryMode_4gInternal;
    case 5:
        return bconfig::MemoryMode_6gInternal;
    default:
        NN_ABORT( "Invalid drop down index: %d.", index );
    }
}

inline int GetMemoryModeItemIndex( nn::Bit8 memoryMode )
{
    switch ( memoryMode )
    {
    case bconfig::MemoryMode_Auto:
        return 0;
    case bconfig::MemoryMode_4g:
        return 1;
    case bconfig::MemoryMode_6g:
        return 2;
    case bconfig::MemoryMode_4gSdk:
        return 3;
    case bconfig::MemoryMode_4gInternal:
        return 4;
    case bconfig::MemoryMode_6gInternal:
        return 5;
    default:
        NN_ABORT( "Invalid memory mode: 0x%02X", memoryMode );
    }
}

nn::Bit64 TickInitialValueTable[] = {
    0,
    0x00000A8FAAAC0000ull, // 7 days
    0x0000151F55580000ull, // 14 days
    0x00002D4324980000ull, // 30 days
    0x000087C96DC80000ull, // 90 days
};

const int TickInitialValueIndexCustom = NN_ARRAY_SIZE(TickInitialValueTable);

inline nn::Bit64 ConvertIndexToIntialTickValue(int index)
{
    NN_ABORT_UNLESS(index >= 0 && index < NN_ARRAY_SIZE(TickInitialValueTable));
    return TickInitialValueTable[index];
}

inline int ConvertIntialTickValueToIndex(nn::Bit64 value)
{
    for (int i = 0; i < NN_ARRAY_SIZE(TickInitialValueTable); i++)
    {
        if (value == TickInitialValueTable[i])
        {
            return i;
        }
    }
    return TickInitialValueIndexCustom; // <Custom>
}

}} // ~namespace devmenu::bconfig, ~namespace devmenu
