﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsSceneSyncTask.h"

#include <nn/nn_Abort.h>

#include <nn/util/util_FormatString.h>

namespace devmenu { namespace accounts {

SyncTaskScene::Closer::Closer(const AbstractOperators& op, SyncTaskScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Close", [&]{CloseEventCallback();}, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void SyncTaskScene::Closer::CloseEventCallback() NN_NOEXCEPT
{
    m_Op.PopScene(&m_Parent, m_Parent.m_pPreviousScene);
    m_Parent.m_pPreviousScene = nullptr;
}

SyncTaskScene::Label::Label(SyncTaskScene& parent, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Placer()
    , m_pTitle(nullptr)
    , m_pDescription(nullptr)
    , m_Mode(Mode_Progress)
    , m_Result(nn::ResultSuccess())
{
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}
void SyncTaskScene::Label::SetProgress() NN_NOEXCEPT
{
    m_Mode = Mode_Progress;
}
void SyncTaskScene::Label::SetResult(nn::Result result) NN_NOEXCEPT
{
    m_Mode = Mode_Result;
    m_Result = result;
}
void SyncTaskScene::Label::Update() NN_NOEXCEPT
{
    if (m_pTitle)
    {
        delete m_pTitle;
        m_pTitle = nullptr;
    }
    if (m_pDescription)
    {
        delete m_pDescription;
        m_pDescription = nullptr;
    }

    switch (m_Mode)
    {
    case Mode_Progress:
        {
            m_pTitle = new glv::Label("In Progress...", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
            m_pDescription = new glv::Label("");
        }
        break;

    case Mode_Result:
        {
            const char* title;
            char description[32] = "";
            if (m_Result.IsSuccess())
            {
                title = "Success";
            }
            else
            {
                title = "Failed";
                nn::util::SNPrintf(description, sizeof(description), "%03d%04d", m_Result.GetModule(), m_Result.GetDescription());
            }
            m_pTitle = new glv::Label(title, glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
            m_pDescription = new glv::Label(description, glv::Label::Spec(glv::Place::TL, 0, 0, accounts::Default::FontSize));
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_Placer = glv::Placer(*this, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 0.0f);
    m_Placer << m_pTitle << m_pDescription;
}

SyncTaskScene::SyncTaskScene(AbstractOperators& op, glv::Rect rect) NN_NOEXCEPT
    : Scene(rect)
    , m_Base(0, 0, rect.width(), rect.height())
    , m_Placer(m_Base, glv::Direction::S, glv::Place::TC, rect.width() / 2.0f, 0.0f, 32)
    , m_Closer(op, *this, glv::Rect(384, 48))
    , m_Label(*this, glv::Rect(width(), 64))
    , m_pFunction(nullptr)
    , m_Result(nn::ResultSuccess())
    , m_IsCompleted(false)
    , m_pStack(nullptr)
    , m_IsThreadCreated(false)
    , m_pPreviousScene(nullptr)
{
    m_pStack = std::malloc(StackSize + nn::os::ThreadStackAlignment);
    NN_ABORT_UNLESS(m_pStack);

    *this << m_Base;

    m_Label.enable(glv::Property::Visible);
    m_Closer.disable(glv::Property::Visible);
}
SyncTaskScene::~SyncTaskScene() NN_NOEXCEPT
{
    if (m_IsThreadCreated)
    {
        nn::os::DestroyThread(&m_Thread);
    }

    std::free(m_pStack);

    Clear();
}
glv::View* SyncTaskScene::GetPrimaryView() NN_NOEXCEPT
{
    return m_Closer.visible() ? &m_Closer : nullptr;
}
void SyncTaskScene::SetContext(Scene* pPrev, std::function<nn::Result(void)> pFunction) NN_NOEXCEPT
{
    m_pPreviousScene = pPrev;

    m_pFunction = pFunction;

    m_Label.SetProgress();

    m_Closer.disable(glv::Property::Visible);

    m_IsCompleted = false;

    NN_SDK_ASSERT(!m_IsThreadCreated);

    void* pAligned = reinterpret_cast<void*>((reinterpret_cast<uintptr_t>(m_pStack) +
        ((nn::os::ThreadStackAlignment) - 1)) & ~((nn::os::ThreadStackAlignment) - 1));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, WorkerThread, this,
        pAligned, StackSize, nn::os::DefaultThreadPriority));
    m_IsThreadCreated = true;

    nn::os::StartThread(&m_Thread);
}
void SyncTaskScene::Update() NN_NOEXCEPT
{
    if (!m_IsThreadCreated)
    {
        return;
    }

    if (m_IsCompleted)
    {
        m_Label.SetResult(m_Result);
        m_Closer.enable(glv::Property::Visible);

        RequireRefresh();

        if (m_IsThreadCreated)
        {
            nn::os::DestroyThread(&m_Thread);
            m_IsThreadCreated = false;
        }
    }
}
void SyncTaskScene::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    m_Placer = glv::Placer(m_Base, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 0.0f);

    m_Label.remove();
    m_Closer.remove();
}
void SyncTaskScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    m_Label.Update();
    m_Placer << m_Label;
    if (m_Closer.visible())
    {
        m_Placer << m_Closer;
    }
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetPrimaryView());
}
void SyncTaskScene::WorkerThread(void* pArg) NN_NOEXCEPT
{
    SyncTaskScene* pScene = reinterpret_cast<SyncTaskScene*>(pArg);

    pScene->m_Result = pScene->m_pFunction();
    pScene->m_IsCompleted = true;
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
