﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsSceneList.h"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiDebug.h>

#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>

namespace devmenu { namespace accounts {

/* AccountView --------------------------------------------------------------------------------
*/

const char AccountView::UserContextView::LayoutStr[] = ">";
AccountView::UserContextView::UserContextView(const UserContext& context, glv::space_t width, glv::space_t height) NN_NOEXCEPT
    : glv::View(glv::Rect(width, height), glv::Place::TL)
    , m_Layout(LayoutStr, .0f, 13.0f, glv::Rect(width, height))
    , m_OpenLabel("Open", glv::Label::Spec(glv::Place::CC, 0, 0, 16))
    , m_LastOpenedLabel("LastOpened", glv::Label::Spec(glv::Place::CC, 0, 0, 16))
    , m_Spacer(width - 8, 1)
{
    *this << (m_Layout << m_OpenLabel << m_LastOpenedLabel << m_Spacer);
    m_Layout.arrange().fit(false);
    m_Layout.disable(glv::Property::HitTest);

    m_OpenStyle.color.set(glv::StyleColor::Preset::WhiteOnBlack);
    m_OpenStyle.color.text.set(context.IsOpen? 1.0f: 0.4f, 1.0f);
    m_OpenLabel.style(&m_OpenStyle);

    m_LastOpenedStyle.color.set(glv::StyleColor::Preset::WhiteOnBlack);
    m_LastOpenedStyle.color.text.set(context.IsLastOpened? 1.0f: 0.4f, 1.0f);
    m_LastOpenedLabel.style(&m_LastOpenedStyle);

    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}

const char AccountView::UserInfoView::LayoutStr[] = ". . x," "< < <," "| | <";
AccountView::UserInfoView::UserInfoView(AccountView& parent, const UserInfo& info, glv::space_t width, glv::space_t height) NN_NOEXCEPT
    : glv::Button(glv::Rect(width, height), true)
    , m_Layout(LayoutStr, 0, 0, glv::Rect(width, height))
    , m_Icon(64, 64, 8, 8)
    , m_Spacer(width - 80, 1)
    , m_SpacerX16(16, 1)
    , m_Parent(parent)
{
#if defined ( NN_CUSTOMERSUPPORTTOOL )
    NN_UNUSED(m_Parent);
#endif
    m_pNickname = new glv::Label(info.nickname, glv::Label::Spec(glv::Place::BL, 0, 0, Default::FontSize));

    auto n = nn::util::SNPrintf(m_StatusString, sizeof(m_StatusString), "UID:");
    NN_ASSERT(n < sizeof(m_StatusString));
    n += UidToString(&m_StatusString[n], sizeof(m_StatusString) - n, info.id);
    NN_ASSERT(n < sizeof(m_StatusString));

    if (info.nsaId)
    {
        n += nn::util::SNPrintf(&m_StatusString[n], sizeof(m_StatusString) - n, ", NSAID:");
        NN_ASSERT(n < sizeof(m_StatusString));
        n += NsaIdToString(&m_StatusString[n], sizeof(m_StatusString) - n, info.nsaId);
        NN_ASSERT(n < sizeof(m_StatusString));
    }
    m_pStatus = new glv::Label(m_StatusString, glv::Label::Spec(glv::Place::TL, 0, 0, Default::FontSizeSmall));

    info.icon
        ? m_Icon.SetTexture(info.icon)
        : m_Icon.ClearTexture();

    *this << (m_Layout << m_Spacer << m_Icon << m_SpacerX16 << m_pNickname << m_pStatus);
    m_Layout.arrange().fit(false);
    m_Layout.disable(glv::Property::HitTest);

    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask );
#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    attach( []( const glv::Notification& n )->void
    {
        accounts::AccountView& parentView = n.receiver< UserInfoView >()->m_Parent;
        parentView.m_Op.OpenUserInfo( parentView.m_User );
    }, glv::Update::Clicked, this );
#endif
}
AccountView::UserInfoView::~UserInfoView() NN_NOEXCEPT
{
    delete m_pNickname;
    delete m_pStatus;
}

AccountView::AccountView(AbstractOperators& op, const nn::account::Uid& user, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : View(r, anchor)
    , m_Placer(*this, glv::Direction::E, glv::Place::TL, 0.0f, 0.0f, 0.0f)
    , m_Op(op)
    , m_User(user)
{
#if defined ( NN_CUSTOMERSUPPORTTOOL )
    NN_UNUSED(m_Op);
#endif
    NN_ASSERT(user);

    // UserContext --------------------------------------------
    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListOpenUsers(&count, users, nn::account::UserCountMax));
    bool isOpened = false;
    for (int i = 0; i < count; ++ i)
    {
        if (users[i] == m_User)
        {
            isOpened = true;
            break;
        }
    }
    nn::account::Uid lastOpened;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetLastOpenedUser(&lastOpened));
    UserContext context = { isOpened, user == lastOpened };
    m_pUserContext = new UserContextView(context, 116.0f, r.height());

    // UserInfo -----------------------------------------------
    UserInfo info = { user };
    GetNickname(info.nickname, sizeof(info.nickname) / sizeof(info.nickname[0]), user);
    info.icon = LoadUserIcon(user);
    info.nsaId = GetNetworkServiceAccountId(user);
    m_pUserInfo = new UserInfoView(*this, info, r.width() - 116.0f, r.height());

    // Register -----------------------------------------------
    m_Placer << m_pUserContext;
    m_Placer << m_pUserInfo;
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}
AccountView::~AccountView() NN_NOEXCEPT
{
    delete m_pUserContext;
    delete m_pUserInfo;
}

/* ボタン類 ---------------------------------------------------------------------------------------
*/

ListScene::FwdbgOpener::FwdbgOpener(const AbstractOperators& op, ListScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Firmware settings", [&] { m_Op.OpenFwdbg(); } , r, anchor)
    , m_Op(op)
{
}

ListScene::Adder::Adder(const AbstractOperators& op, ListScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("+ Add a user account", [&]{ m_Op.AddUser(); } , r, anchor)
    , m_Op(op)
{
}

/* シーン ----------------------------------------------------------------------------------------
*/

ListScene::ListScene(AbstractOperators& op, glv::Rect rect) NN_NOEXCEPT
    : Scene(rect)
    , m_Op(op)
    , m_Table("p,v,p,d", 0.0f, 0.0f)
    , m_Header(
        "Accounts",
        glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize))
#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    , m_Footer(
        GLV_TEXT_API_WIDE_STRING("↑↓:Focus  A:Select"),
        glv::Label::Spec(glv::Place::BR, 0.0f, 0.0f, CommonValue::InitialFontSize))
#else
    , m_Footer(
        GLV_TEXT_API_WIDE_STRING("↑↓:Focus"),
        glv::Label::Spec(glv::Place::BR, 0.0f, 0.0f, CommonValue::InitialFontSize))
#endif
    , m_TopButtons("p", 10.0f, 10.0f)
    , m_ScrollableContainer("<", glv::Rect(width(), 0.0f))
    , m_LabelNoItem(
        "No accounts",
        glv::Label::Spec(glv::Place::CC, 0, 0, Default::FontSize))
    , m_FwdbgOpener(op, *this, glv::Rect(rect.width() - ListScene::ButtonHorizontalMargin, ListScene::ButtonHeight), glv::Place::CC)
    , m_pAdder(nullptr)
    , m_pPrimaryView(nullptr)
{
    for (auto& p : m_pAccounts)
    {
        p = nullptr;
    }
}
ListScene::~ListScene() NN_NOEXCEPT
{
    Clear();
}
glv::View* ListScene::GetPrimaryView() const NN_NOEXCEPT
{
    return m_pPrimaryView;
}

void ListScene::Clear() NN_NOEXCEPT
{
    // 一旦 focus を外す
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    m_Header.remove();
    m_FwdbgOpener.remove();
    if (m_pAdder)
    {
        delete m_pAdder;
        m_pAdder = nullptr;
    }
    for (auto& p: m_pAccounts)
    {
        if (p)
        {
            delete p;
            p = nullptr;
        }
    }
    m_LabelNoItem.remove();
    m_ScrollableContainer.Refresh(); // 全部の子がなくなったのでテーブルをリフレッシュする (必要がある。残念。)
    m_ScrollableContainer.remove();
    m_TopButtons.remove();
    m_Footer.remove();
    m_Table.remove();

    m_pPrimaryView = nullptr;
}

void ListScene::Refresh() NN_NOEXCEPT
{
    // TODO フォーカス管理をしっかりする (シーンへの入場, 退場, リフレッシュ前後のフォーカスがポイントか？)
    // TODO 本当はリフレッシュが必要な部分だけ (ある AccountView だけとか) をリフレッシュしたほうが良い

    Clear();

    ClearRefreshRequest();

    // Contents
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));

#if !defined ( NN_CUSTOMERSUPPORTTOOL )

#if defined(DEVMENU_ACCOUNTS_FWDBG_ENABLE)
    m_TopButtons << m_FwdbgOpener;
#endif

    if (count < nn::account::UserCountMax)
    {
        NN_SDK_ASSERT(m_pAdder == nullptr);
        m_pAdder = new Adder(m_Op, *this, glv::Rect(width() - ListScene::ButtonHorizontalMargin, ListScene::ButtonHeight), glv::Place::CC);
        m_TopButtons << m_pAdder;
    }
#endif

    m_TopButtons.arrange();
    m_ScrollableContainer.SetRectAndUpdateInnerSize(width(), height() - m_Header.h - m_TopButtons.h - m_Footer.h);

    const auto& inner = m_ScrollableContainer.GetInnerRect();
    if (count > 0)
    {
        glv::Rect entitySize(inner.width(), 80.0f);
        nn::account::Uid users[nn::account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
        for (auto i = 0; i < count; ++i)
        {
            NN_SDK_ASSERT(m_pAccounts[i] == nullptr);
            auto& u = m_pAccounts[i];
            u = new AccountView(m_Op, users[i], entitySize);
            m_ScrollableContainer << u;
        }
    }
    else
    {
        m_ScrollableContainer << m_LabelNoItem;
    }

    m_ScrollableContainer.ArrangeTable();

    m_Table << m_Header << m_TopButtons << m_ScrollableContainer << m_Footer;
    m_Table.arrange();
    *this << m_Table;

#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    // 最初に選択される要素の決定
    m_pPrimaryView = &m_FwdbgOpener;
#else
    m_pPrimaryView = &m_ScrollableContainer;
#endif

    // focus を自身に付けなおす
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetPrimaryView());

    // focusableChild を更新する
    m_Op.SetFocusableChild( GetPrimaryView() );
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
