﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_Country.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_AccountsSceneDetail.h"
#include "../DevMenu_Common.h"
#include "../DevMenu_Config.h"
#include "../Launcher/DevMenu_LauncherLibraryAppletApis.h"

namespace devmenu { namespace accounts {
namespace {

nn::Result WaitAsyncContext(nn::account::AsyncContext&& async) NN_NOEXCEPT
{
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
    e.Wait();
    return async.GetResult();
}

} // ~namespace devmenu::accounts::<anonymous>

/* ボタン類 -------------------------------------------------------------------------------------
*/

DetailScene::Closer::Closer(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Back", [&]{CloseEventCallback();} , r, anchor)
    , m_Op(op)
{
}
void DetailScene::Closer::CloseEventCallback() NN_NOEXCEPT
{
    m_Op.CloseUserInfo();
}

DetailScene::Buttons::Deleter::Deleter(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Delete user account", [&]{DeleteEventCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::Deleter::DeleteEventCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        m_Op.DeleteUser(user);
    }
}
void DetailScene::Buttons::Deleter::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        IsNetworkServiceAccountRegistered(user)
            ? disable(glv::Property::Visible)
            : enable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaRegistrar::SaRegistrar(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Register NSA", [&]{RegisrationCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaRegistrar::RegisrationCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        m_Op.BeginSyncTask(
            [user]() -> nn::Result {
                nn::account::NetworkServiceAccountAdministrator admin;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

                nn::account::AsyncContext context;
                NN_ABORT_UNLESS_RESULT_SUCCESS(admin.RegisterAsync(&context));
                nn::os::SystemEvent e;
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&e));
                e.Wait();
                NN_RESULT_DO(context.GetResult());

                auto r = admin.CheckNetworkServiceAccountAvailability();
                NN_RESULT_THROW_UNLESS(r.IsSuccess() || nn::account::ResultNetworkServiceAccountUnavailable::Includes(r), r);
                if (r.IsSuccess())
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.SynchronizeProfileAsync(&context));
                    NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&e));
                    e.Wait();
                    NN_RESULT_DO(context.GetResult());
                }
                NN_RESULT_SUCCESS;
            },
            &m_Parent);
    }
}
void DetailScene::Buttons::SaRegistrar::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        IsNetworkServiceAccountRegistered(user)
            ? disable(glv::Property::Visible)
            : enable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaUnregistrar::SaUnregistrar(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Unregister NSA", [&]{UnregisrationCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaUnregistrar::UnregisrationCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        m_Op.BeginSyncTask(
            [user]() -> nn::Result {
                return nn::ns::UnregisterNetworkServiceAccount(user);
            },
            &m_Parent);
    }
}
void DetailScene::Buttons::SaUnregistrar::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        IsNetworkServiceAccountRegistered(user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaEnsurer::SaEnsurer(DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Ensure NSA with UI", [&]{EnsuranceCallback();} , r, anchor)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaEnsurer::EnsuranceCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        auto r = launcher::EnsureNetworkServiceAccountAvailable(user);
        if (!r.IsSuccess())
        {
            if (nn::account::ResultCancelledByUser::Includes(r))
            {
                NN_LOG("[Accounts / NSA: Ensure] User cancelled\n");
            }
            else
            {
                NN_LOG("[Accounts / NSA: Ensure] EnsureNetworkServiceAccountAvailable failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
            }
        }
        m_Parent.RequireRefresh();
    }
}
void DetailScene::Buttons::SaEnsurer::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        !GetNetworkServiceAccountId(user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaProfileSync::SaProfileSync(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Sync user profile", [&]{SynchronizationCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaProfileSync::SynchronizationCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

        nn::account::AsyncContext context;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.SynchronizeProfileAsync(&context));
        m_Op.BeginAsyncTask(std::move(context), &m_Parent);
    }
}
void DetailScene::Buttons::SaProfileSync::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        GetNetworkServiceAccountId(user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaProfileUpload::SaProfileUpload(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Force upload user profile", [&] {UploadCallback(); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaProfileUpload::UploadCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

        nn::account::AsyncContext context;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.UploadProfileAsync(&context));
        m_Op.BeginAsyncTask(std::move(context), &m_Parent);
    }
}
void DetailScene::Buttons::SaProfileUpload::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        GetNetworkServiceAccountId(user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::SaNotificationTokenUpload::SaNotificationTokenUpload(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Upload notification token", [&]{UploadCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::SaNotificationTokenUpload::UploadCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        m_Op.BeginSyncTask(
            [=]() -> nn::Result
            {
                NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()),
                    nn::npns::ResultNetworkIsNotAvailable());

                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Suspend());

                NN_UTIL_SCOPE_EXIT
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Resume());
                    nn::npns::FinalizeForSystem();
                };

                char jid[nn::npns::JidLength + 1] = {};
                nn::Result result = nn::npns::GetJid(jid, sizeof (jid));

                if (result.IsFailure())
                {
                    NN_RESULT_DO(nn::npns::CreateJid());
                }

                NN_RESULT_DO(nn::npns::UploadTokenToBaaS(user));

                return nn::ResultSuccess();
            },
            &m_Parent);
    }
}
void DetailScene::Buttons::SaNotificationTokenUpload::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::NetworkServiceAccountManager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));

        auto r = manager.CheckNetworkServiceAccountAvailability();
        (r.IsSuccess() || nn::account::ResultNintendoAccountLinkageRequired::Includes(r))
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::NaCacheUpdate::NaCacheUpdate(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Update NA info cache", [&]{SynchronizationCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::NaCacheUpdate::SynchronizationCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        m_Op.BeginSyncTask(
            [user]()->nn::Result {
                nn::account::NetworkServiceAccountManager manager;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));

                nn::account::AsyncContext context;
                NN_ABORT_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNintendoAccountInfoAsync(&context));
                NN_RESULT_DO(WaitAsyncContext(std::move(context)));

                NN_ABORT_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNetworkServiceLicenseInfoAsync(&context));
                NN_RESULT_DO(WaitAsyncContext(std::move(context)));

                NN_RESULT_SUCCESS;
            }, &m_Parent);
    }
}
void DetailScene::Buttons::NaCacheUpdate::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        char screenName[128];
        GetNintendoAccountScreenName(screenName, sizeof(screenName), user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::NicknameSelector::NicknameSelector(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Edit nickname", [&]{ SelectNicknameCallback(); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::NicknameSelector::SelectNicknameCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if( user )
    {
        m_Op.OpenNicknameSelector(user);
    }
}

DetailScene::Buttons::ProfileImageSelector::ProfileImageSelector(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Edit profile image", [&]{ SelectProfileImageCallback(); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::ProfileImageSelector::SelectProfileImageCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if( user )
    {
        m_Op.OpenProfileImageSelector(user);
    }
}

DetailScene::Buttons::DebugCacheClearance::DebugCacheClearance(DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("[Debug] Invalidate cache", [&]{ SetCallback(); }, r, anchor)
    , m_Parent(parent)
{
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set(glv::StyleColor::BlackOnWhite);
    pStyle->color.fore.set(0.8, 0.8, 0.8);
    pStyle->color.back.set(0.3, 0.0, 0.1);
    style(pStyle);
    enable(glv::Property::DrawBack | glv::Property::DrawBorder);
}
void DetailScene::Buttons::DebugCacheClearance::SetCallback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::DebugInvalidateUserResourceCache(user);
        m_Parent.RequireRefresh();
    }
}
void DetailScene::Buttons::DebugCacheClearance::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        GetNetworkServiceAccountId(user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::DebugNaBadStatusSelector::DebugNaBadStatusSelector(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("[Debug] Set user state to invalid", [&]{ SetBadStatusCallback(); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set(glv::StyleColor::BlackOnWhite);
    pStyle->color.fore.set(0.8, 0.8, 0.8);
    pStyle->color.back.set(0.3, 0.0, 0.1);
    style(pStyle);
    enable(glv::Property::DrawBack | glv::Property::DrawBorder);
}
void DetailScene::Buttons::DebugNaBadStatusSelector::SetBadStatusCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if( user )
    {
        m_Op.OpenBadStatusSelector(user);
    }
}
void DetailScene::Buttons::DebugNaBadStatusSelector::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        char screenName[128];
        GetNintendoAccountScreenName(screenName, sizeof(screenName), user)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::PlayDataInfoViewer::PlayDataInfoViewer(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("View play data", [&] { SetCallback(); }, r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::PlayDataInfoViewer::SetCallback() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if( user )
    {
        m_Op.OpenPlayDataInfo(user);
    }
}

DetailScene::Buttons::NaImplicitRecovery::NaImplicitRecovery(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Try recover user state", [&]{Callback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}
void DetailScene::Buttons::NaImplicitRecovery::Callback() NN_NOEXCEPT
{
    NN_ASSERT(visible());

    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

        nn::account::AsyncContext context;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.TryRecoverNintendoAccountUserStateAsync(&context));
        m_Op.BeginAsyncTask(std::move(context), &m_Parent);
    }
}
void DetailScene::Buttons::NaImplicitRecovery::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        nn::account::NetworkServiceAccountManager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));

        auto r = manager.CheckNetworkServiceAccountAvailability();
        nn::account::ResultNintendoAccountStateInteractionRequired::Includes(r)
            ? enable(glv::Property::Visible)
            : disable(glv::Property::Visible);
    }
}

DetailScene::Buttons::Buttons(const AbstractOperators& op, DetailScene& parent, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Placer(*this, glv::Direction::S, glv::Place::TC, r.width() / 2.0f, 0.0f, 16)
    , m_Deleter(op, parent, glv::Rect(r.width(), 42))
    , m_SaRegistrar(op, parent, glv::Rect(r.width(), 42))
    , m_SaUnregistrar(op, parent, glv::Rect(r.width(), 42))
    , m_SaEnsurer(parent, glv::Rect(r.width(), 42))
    , m_SaProfileSync(op, parent, glv::Rect(r.width(), 42))
    , m_SaProfileUpload(op, parent, glv::Rect(r.width(), 42))
    , m_SaNotificationTokenUpload(op, parent, glv::Rect(r.width(), 42))
    , m_NaCacheUpdate(op, parent, glv::Rect(r.width(), 42))
    , m_NicknameSelector(op, parent, glv::Rect(r.width(), 42))
    , m_ProfileImageSelector(op, parent, glv::Rect(r.width(), 42))
    , m_DebugIdTokenClearance(parent, glv::Rect(r.width(), 42))
    , m_DebugNaBadStatusSelector(op, parent, glv::Rect(r.width(), 42))
    , m_PlayDataInfoViewer(op, parent, glv::Rect(r.width(), 42))
    , m_NaImplicitRecovery(op, parent, glv::Rect(r.width(), 42))
{
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}
void DetailScene::Buttons::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(!glv::GLV::valid(&g));
    NN_UNUSED(g);

    m_Placer = glv::Placer(*this, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 12.0f);
}
void DetailScene::Buttons::Update() NN_NOEXCEPT
{
    Clear();

    m_Placer << m_PlayDataInfoViewer;
    m_Placer << m_NicknameSelector;
    m_Placer << m_ProfileImageSelector;

    m_SaProfileSync.Update();
    if (m_SaProfileSync.visible())
    {
        m_Placer << m_SaProfileSync;
    }
    m_SaProfileUpload.Update();
    if (m_SaProfileUpload.visible())
    {
        m_Placer << m_SaProfileUpload;
    }
    m_SaNotificationTokenUpload.Update();
    if (m_SaNotificationTokenUpload.visible())
    {
        m_Placer << m_SaNotificationTokenUpload;
    }
    m_NaCacheUpdate.Update();
    if (m_NaCacheUpdate.visible())
    {
        m_Placer << m_NaCacheUpdate;
    }

    m_SaEnsurer.Update();
    if (m_SaEnsurer.visible())
    {
        m_Placer << m_SaEnsurer;
    }
    m_NaImplicitRecovery.Update();
    if (m_NaImplicitRecovery.visible())
    {
        m_Placer << m_NaImplicitRecovery;
    }
    m_SaRegistrar.Update();
    if (m_SaRegistrar.visible())
    {
        m_Placer << m_SaRegistrar;
    }
    m_SaUnregistrar.Update();
    if (m_SaUnregistrar.visible())
    {
        m_Placer << m_SaUnregistrar;
    }

    m_Deleter.Update();
    if (m_Deleter.visible())
    {
        m_Placer << m_Deleter;
    }

    m_DebugIdTokenClearance.Update();
    if (m_DebugIdTokenClearance.visible())
    {
        m_Placer << m_DebugIdTokenClearance;
    }

    m_DebugNaBadStatusSelector.Update();
    if (m_DebugNaBadStatusSelector.visible())
    {
        m_Placer << m_DebugNaBadStatusSelector;
    }
}

/* 名前表示 ---------------------------------------------------------------------------------------
*/

DetailScene::Detail::Name::Name(Detail& parent, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Parent(parent)
    , m_Placer()
    , m_pNickname(nullptr)
    , m_pUid(nullptr)
{
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}
void DetailScene::Detail::Name::Update() NN_NOEXCEPT
{
    if (m_pNickname)
    {
        delete m_pNickname;
        m_pNickname = nullptr;
    }
    if (m_pUid)
    {
        delete m_pUid;
        m_pUid = nullptr;
    }

    m_Placer = glv::Placer(*this, glv::Direction::S, glv::Place::TC, 0.0f, 0.0f, 16.0f);
    auto user = m_Parent.m_Parent.m_User;
    if (user)
    {
        glv::CharTypeU16 nickname[64];
        GetNickname(nickname, sizeof(nickname) / sizeof(nickname[0]), user);
        m_pNickname = new glv::Label(nickname, glv::Label::Spec(glv::Place::TC, 0, 0, CommonValue::InitialFontSize));
        m_Placer << m_pNickname;

        nn::account::NetworkServiceAccountManager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
        auto availability = manager.CheckNetworkServiceAccountAvailability();
        NN_ABORT_UNLESS(availability.IsSuccess() || nn::account::ResultNetworkServiceAccountUnavailable::Includes(availability));

        devmenu::Buffer b(4096);
        nn::account::CachedNintendoAccountInfoForSystemService info;
        auto r = manager.LoadCachedNintendoAccountInfo(&info, b.GetPointer(), b.GetSize());
        auto isNaAvailable = r.IsSuccess();

        {
            char str[512];

            int n = 0;

            n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "UID: ");
            n += UidToString(&str[n], sizeof(str) - n, user);

            n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "\nNSA Status: ");
            if (availability.IsSuccess())
            {
                n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "Available");
            }
            else
            {
                n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "%04d", availability.GetDescription());
            }

            n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "\nNSA ID: ");
            auto nsaId = GetNetworkServiceAccountId(user);
            if (nsaId)
            {
                n += NsaIdToString(&str[n], sizeof(str) - n, nsaId);
            }
            else
            {
                n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "(not available)");
            }

            n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "\nNA: ");
            if (isNaAvailable)
            {
                size_t sz;
                const auto* countryStr = info.GetCountry(&sz);
                nn::util::Country country;
                auto isCountryAvailable = nn::util::GetCountryFromIso31661Alpha2(&country, countryStr, sz);
                n += nn::util::SNPrintf(
                    &str[n], sizeof(str) - n,
                    "\"%s\"(%d,%c,%c,%c)",
                    info.GetScreenName(&sz),
                    isCountryAvailable ? country : 0,
                    info.GetAnalyticsOptedInFlag() ? 'A' : '-',
                    info.IsChild() ? 'C' : '-',
                    info.IsLinkedWithNintendoNetwork() ? 'N' : '-');
            }
            else
            {
                n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "(not available)");
            }

            nn::account::CachedNetworkServiceLicenseInfo licenseInfo;
            auto isLicenseInfoCached = manager.GetCachedNetworkServiceLicenseInfo(&licenseInfo).IsSuccess();

            n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "\nLicense: ");
            if (isLicenseInfoCached)
            {
                switch (licenseInfo.kind)
                {
                case nn::account::NetworkServiceLicenseKind_None:
                    n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "No license");
                    break;
                case nn::account::NetworkServiceLicenseKind_Common:
                    {
                        nn::time::CalendarTime cal;
                        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&cal, nullptr, licenseInfo.expiration));
                        n += nn::util::SNPrintf(
                            &str[n], sizeof(str) - n,
                            "Expires at %04d-%02d-%02d %02d:%02d",
                            cal.year, cal.month, cal.day, cal.hour, cal.minute);
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else
            {
                n += nn::util::SNPrintf(&str[n], sizeof(str) - n, "(not available)");
            }

            m_pUid = new glv::Label(str, glv::Label::Spec(glv::Place::TC, 0, 0, Default::FontSizeSmall));
            m_Placer << m_pUid;
        }
    }
} // NOLINT(readability/fn_size)

/* 詳細表示 ---------------------------------------------------------------------------------------
*/

DetailScene::Detail::Detail(DetailScene& parent, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Parent(parent)
    , m_Placer(*this, glv::Direction::S, glv::Place::TC, r.width() / 2.0f, 0.0f, 24.0f)
    , m_Icon(256, 256, 16.0f, .0f)
    , m_Name(*this, glv::Rect(r.width(), 128))
{
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);

    m_Placer << m_Icon;
    m_Placer << m_Name;
}
void DetailScene::Detail::Update() NN_NOEXCEPT
{
    auto user = m_Parent.m_User;
    if (user)
    {
        auto icon = LoadUserIcon(user);
        icon
            ? m_Icon.SetTexture(icon)
            : m_Icon.ClearTexture();

        m_Name.Update();
    }
}

/* シーン ----------------------------------------------------------------------------------------
*/
DetailScene::DetailScene(const AbstractOperators& op, glv::Rect rect) NN_NOEXCEPT
    : Scene(rect)
    , m_User(nn::account::InvalidUid)
    , m_Base(0, 0, rect.width(), rect.height())
    , m_Placer()
    , m_Header(
        "Account detail",
        glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize))
    , m_Closer(op, *this, glv::Rect(rect.width() / 2.0f - 48, 48))
    , m_pLayout(nullptr)
    , m_Detail(*this, glv::Rect(rect.width() / 2.0f - 48, 400))
    , m_Buttons(op, *this, glv::Rect(rect.width() / 2.0f - 48, 480))
{
    *this << m_Base;
#if 0
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set(glv::StyleColor::BlackOnWhite);
    pStyle->color.fore.set(0.8, 0.8, 0.8);
    pStyle->color.back.set(0.0, 0.6, 0.0);
    style(pStyle);
    enable(glv::Property::DrawBack | glv::Property::DrawBorder);
#endif
}
DetailScene::~DetailScene() NN_NOEXCEPT
{
    DetailScene::Clear();
}
glv::View* DetailScene::GetPrimaryView() NN_NOEXCEPT
{
    return &m_Closer;
}
void DetailScene::SetAccount(nn::account::Uid user) NN_NOEXCEPT
{
    m_User = user;
}
const nn::account::Uid &DetailScene::GetAccount() NN_NOEXCEPT
{
    return m_User;
}
void DetailScene::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    m_Placer = glv::Placer(m_Base, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 16.0f);
    m_Closer.remove();
    if (m_pLayout != nullptr)
    {
        delete m_pLayout;
    }
    m_pLayout = new glv::Table("p>,"
                               "<|,", 16.0f, .0f, glv::Rect(width(), 0));
#if 0
    {
        glv::Style* pStyle = new glv::Style();
        pStyle->color.set(glv::StyleColor::BlackOnWhite);
        pStyle->color.fore.set(0.8, 0.8, 0.8);
        pStyle->color.back.set(0.0, 0.0, 0.6);
        m_pLayout->style(pStyle);
        m_pLayout->enable(glv::Property::DrawBack | glv::Property::DrawBorder);
    }
#endif

    m_Detail.remove();
    m_Buttons.remove();
}
void DetailScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    *m_pLayout << m_Closer;
    m_Buttons.Update();
    *m_pLayout << m_Buttons;
    m_Detail.Update();
    *m_pLayout << m_Detail;
    m_pLayout->arrange();

    m_Placer << m_Header;
    m_Placer << m_pLayout;

    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetPrimaryView());
}

nn::Result DetailScene::OnNicknameSelected(const NicknameSelection& selection) NN_NOEXCEPT
{
    nn::account::Nickname nickname;

    if (NicknameSelection::SelectionKind::Keyboard == selection.kind)
    {
        nn::account::GetNickname(&nickname, m_User);

        launcher::KeyboardConfig keyboardConfig;
        keyboardConfig.preset = launcher::KeyboardConfig::Preset::Preset_UserName;
        keyboardConfig.keyboardMode = launcher::KeyboardConfig::KeyboardMode::KeyboardMode_Full;
        keyboardConfig.isPredictionEnabled = false;
        keyboardConfig.isUseUtf8 = true;
        keyboardConfig.isUseNewLine = false;
        keyboardConfig.textMaxLength = 10;
        keyboardConfig.textMinLength = 1;
        keyboardConfig.headerTextUtf8 = "Enter user name";
        keyboardConfig.guideTextUtf8 = "Input user name";

        NN_RESULT_DO(launcher::LaunchSoftwareKeyboardAndGetString(
            nickname.name, sizeof(nickname.name), nickname.name, keyboardConfig));
    }
    else
    {
        nn::util::Strlcpy(nickname.name, selection.data, sizeof(nickname.name));
    }

    nn::account::ProfileEditor profileEditor;
    nn::account::GetProfileEditor(&profileEditor, m_User);
    profileEditor.SetNickname(nickname);
    profileEditor.Flush();

    NN_RESULT_SUCCESS;
}

void DetailScene::OnProfileImageSelected(const char* name) NN_NOEXCEPT
{
    nn::account::ProfileEditor profileEditor;
    nn::account::GetProfileEditor(&profileEditor, m_User);

    char path[64] = {};
    nn::util::SNPrintf(path, sizeof(path), "Contents:/accounts/%s", name);

    devmenu::Buffer b(128 * 1024);
    nn::fs::FileHandle file;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    NN_SDK_ASSERT(static_cast<size_t>(size) <= b.GetSize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, b.GetPointer(), size));

    profileEditor.FlushWithImage(b.GetPointer(), size);
}
void DetailScene::OnDebugUserBadStatusSelected(const nn::Result& result) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, m_User));
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.DebugSetNetworkServiceAccountAvailabilityError(result));
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
