﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsSceneAsyncTask.h"
#include "DevMenu_AccountsConfig.h"

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>

#include <nn/util/util_FormatString.h>

namespace devmenu { namespace accounts {

/* ボタン類 -------------------------------------------------------------------------------------
*/

void AsyncTaskScene::Canceller::CancelEventCallback() NN_NOEXCEPT
{
    if (m_Parent.m_pContext != nullptr)
    {
        m_Parent.Cancel();
    }
}
AsyncTaskScene::Canceller::Canceller(const AbstractOperators& op, AsyncTaskScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Cancel", [&]{CancelEventCallback();} , r, anchor)
    , m_Parent(parent)
{
}

void AsyncTaskScene::Closer::CloseEventCallback() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Parent.m_pContext == nullptr);
    m_Op.PopScene(&m_Parent, m_Parent.m_pPreviousScene);
    m_Parent.m_pPreviousScene = nullptr;
}
AsyncTaskScene::Closer::Closer(const AbstractOperators& op, AsyncTaskScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Close", [&]{CloseEventCallback();} , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}

/* 処理状況表示 ---------------------------------------------------------------------------------------
*/

AsyncTaskScene::Label::Label(AsyncTaskScene& parent, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Placer()
    , m_pTitle(nullptr)
    , m_pDescription(nullptr)
    , m_Mode(Mode_Progress)
    , m_Result(nn::ResultSuccess())
{
    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
}
void AsyncTaskScene::Label::SetProgress() NN_NOEXCEPT
{
    m_Mode = Mode_Progress;
}
void AsyncTaskScene::Label::SetResult(nn::Result result) NN_NOEXCEPT
{
    m_Mode = Mode_Result;
    m_Result = result;
}
void AsyncTaskScene::Label::Update() NN_NOEXCEPT
{
    if (m_pTitle)
    {
        delete m_pTitle;
        m_pTitle = nullptr;
    }
    if (m_pDescription)
    {
        delete m_pDescription;
        m_pDescription = nullptr;
    }

    switch (m_Mode)
    {
    case Mode_Progress:
        {
            m_pTitle = new glv::Label("In Progress...", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
            m_pDescription = new glv::Label("");
        }
        break;

    case Mode_Result:
        {
            const char* title;
            char description[32] = "";
            if (m_Result.IsSuccess())
            {
                title = "Success";
            }
            else if (nn::account::ResultCancelled::Includes(m_Result))
            {
                title = "Cancelled";
            }
            else
            {
                title = "Failed";
                nn::util::SNPrintf(description, sizeof(description), "%03d%04d", m_Result.GetModule(), m_Result.GetDescription());
            }
            m_pTitle = new glv::Label(title, glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
            m_pDescription = new glv::Label(description, glv::Label::Spec(glv::Place::TL, 0, 0, accounts::Default::FontSize));
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_Placer = glv::Placer(*this, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 0.0f);
    m_Placer << m_pTitle << m_pDescription;
}

/* シーン ----------------------------------------------------------------------------------------
*/
AsyncTaskScene::AsyncTaskScene(const AbstractOperators& op, glv::Rect rect) NN_NOEXCEPT
    : Scene(rect)
    , m_Base(0, 0, rect.width(), rect.height())
    , m_Placer(m_Base, glv::Direction::S, glv::Place::TC, rect.width() / 2.0f, 0.0f, 32)
    , m_Canceller(op, *this, glv::Rect(384, 48))
    , m_Closer(op, *this, glv::Rect(384, 48))
    , m_Label(*this, glv::Rect(width(), 64))
    , m_pContext(nullptr)
{
    *this << m_Base;

    m_Closer.disable(glv::Property::Visible);
    m_Canceller.disable(glv::Property::Visible);

#if 0
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set(glv::StyleColor::BlackOnWhite);
    pStyle->color.fore.set(0.8, 0.8, 0.8);
    pStyle->color.back.set(0.0, 0.6, 0.0);
    style(pStyle);
    enable(glv::Property::DrawBack | glv::Property::DrawBorder);
#endif
}
AsyncTaskScene::~AsyncTaskScene() NN_NOEXCEPT
{
    if (m_pContext != nullptr)
    {
        m_pContext->Cancel();
        m_pWaiter->Wait();
        delete m_pWaiter;
        delete m_pContext;
    }
    AsyncTaskScene::Clear();
}
glv::View* AsyncTaskScene::GetPrimaryView() NN_NOEXCEPT
{
    return (m_Closer.visible()
        ? reinterpret_cast<glv::View*>(&m_Closer)
        : (m_Canceller.visible()
            ? &m_Canceller
            : nullptr));
}
void AsyncTaskScene::SetContext(nn::account::AsyncContext&& context, Scene* pPrev) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_pContext == nullptr);

    m_pContext = new nn::account::AsyncContext(std::move(context));
    m_pWaiter = new nn::os::SystemEvent();
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pContext->GetSystemEvent(m_pWaiter));
    m_pPreviousScene = pPrev;

    m_Label.SetProgress();

    m_Closer.disable(glv::Property::Visible);
    m_Canceller.enable(glv::Property::Visible);
}
void AsyncTaskScene::Cancel() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_pContext != nullptr);

    bool done;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pContext->HasDone(&done));
    if (done)
    {
        return;
    }

    m_pContext->Cancel();
    m_Canceller.disable(glv::Property::Visible);
    RequireRefresh();
}
void AsyncTaskScene::Update() NN_NOEXCEPT
{
    if (m_pContext == nullptr)
    {
        return;
    }

    if (m_pWaiter->TryWait())
    {
        m_Label.SetResult(m_pContext->GetResult());
        delete m_pWaiter;
        delete m_pContext;
        m_pContext = nullptr;

        m_Canceller.disable(glv::Property::Visible);
        m_Closer.enable(glv::Property::Visible);
        RequireRefresh();
    }
}
void AsyncTaskScene::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(nullptr);

    m_Placer = glv::Placer(m_Base, glv::Direction::S, glv::Place::TC, width() / 2.0f, 0.0f, 0.0f);

    m_Label.remove();
    m_Canceller.remove();
    m_Closer.remove();
}
void AsyncTaskScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    m_Label.Update();
    m_Placer << m_Label;
    if (m_Canceller.visible())
    {
        m_Placer << m_Canceller;
    }
    if (m_Closer.visible())
    {
        m_Placer << m_Closer;
    }
    auto& g = reinterpret_cast<glv::GLV&>(this->root());
    NN_ASSERT(glv::GLV::valid(&g));
    g.setFocus(GetPrimaryView());
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
