﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_Argument.h>
#include <nn/result/result_HandlingUtility.h>

#include "ControllerSerialFlashTool_Csv.h"
#include "ControllerSerialFlashTool_FileSystem.h"
#include "ControllerSerialFlashTool_SerialFlash.h"
#include "ControllerSerialFlashTool_Utility.h"
#include "ControllerSerialFlashTool_UniquePad.h"

namespace
{
const nn::hid::NpadIdType NpadIds[] = { nn::hid::NpadId::No1,
                                        nn::hid::NpadId::No2,
                                        nn::hid::NpadId::No3,
                                        nn::hid::NpadId::No4,
                                        nn::hid::NpadId::No5,
                                        nn::hid::NpadId::No6,
                                        nn::hid::NpadId::No7,
                                        nn::hid::NpadId::No8,
                                        nn::hid::NpadId::Handheld };
//!< コマンド定義を表す構造体です。
struct Command final
{
    //!< コマンドの名前
    const char* name;

    //!< コマンドの実体
    bool (*execute)(const ::std::vector<::std::string>&) NN_NOEXCEPT;
};

//!< プログラム引数を返します。
::std::vector<::std::string> GetArgs() NN_NOEXCEPT;
//!< コマンド定義を返します。
const Command* GetCommand(const ::std::string& name) NN_NOEXCEPT;

//!< 設定のエクスポートをディスパッチします。
bool DispatchExportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;

//!< 設定のインポートをディスパッチします。
bool DispatchImportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT;
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));
    // Set supported NpadStyle
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleJoyLeft::Mask |
        nn::hid::NpadStyleJoyRight::Mask);

    // プログラム引数を取得
    ::std::vector<::std::string> args = GetArgs();

    if (args.size() < 2)
    {
        // コマンドが指定されなかった場合は失敗
        PrintError("Argument Missing. Please specify command and path");
    }
    else
    {
        // コマンド名を退避
        ::std::string name = args[1];

        // コマンド引数以外を除去
        args.erase(args.begin(), args.begin() + 2);

        // ファイルシステム用のアロケータを設定
        ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);

        if (name.find("@") == 0)
        {
            name = name.substr(1);
        }

        const Command* const pCommand = GetCommand(name);

        if (pCommand == nullptr)
        {
            // コマンドが見つからなかった場合は失敗
            PrintError("Command Not Found. Only \"Export\" or \"Import\" is supported");
        }
        else
        {
            // 検出されたコマンドを実行
            pCommand->execute(args);
        }
    }
    return;
}

namespace
{
::std::vector<::std::string> GetArgs() NN_NOEXCEPT
{
    const int argc = ::nn::os::GetHostArgc();

    char** const argv = ::nn::os::GetHostArgv();

    ::std::vector<::std::string> args;

    for (int i = 0; i < argc; ++i)
    {
        args.push_back(argv[i]);
    }

    return args;
}

const Command* GetCommand(const ::std::string& name) NN_NOEXCEPT
{
    static const Command Commands[] =
    {
        { "Export", DispatchExportSettings },
        { "Import", DispatchImportSettings },
    };

    for (const Command& command : Commands)
    {
        if (name == command.name)
        {
            return &command;
        }
    }

    return nullptr;
}


bool DispatchExportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintError("Specify a path to export CSV file\n");
        return false;
    }
    else
    {
        nn::hid::system::UniquePadId uniquePadId;
        SerialFlashCsvType csv;

        COMMAND_DO(GetUniquePad(&uniquePadId));
        COMMAND_DO(ReadSerialFlashAsCsvText(&csv, uniquePadId));
        auto text = GetStringFromCsv(csv);

        HostFsFile hostFsFile;
        std::string filePath = args[0];
        COMMAND_DO(AppendOutputFileName(&filePath, uniquePadId));
        COMMAND_DO(OpenFile(&hostFsFile, filePath, ::nn::fs::OpenMode_Write));
        COMMAND_DO(SetFileSize(hostFsFile, text.size()));
        COMMAND_DO(WriteFile(hostFsFile, 0, text.data(), text.size()));
        COMMAND_DO(FlushFile(hostFsFile));

        return true;
    }
}

bool DispatchImportSettings(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    if (args.size() < 1)
    {
        // ファイルパスが指定されなかった場合は失敗
        PrintError("Specify a path to CSV file to import\n");
        return false;
    }
    else
    {
        nn::hid::system::UniquePadId uniquePadId;

        COMMAND_DO(GetUniquePad(&uniquePadId));

        HostFsFile hostFsFile;
        COMMAND_DO(OpenFile(&hostFsFile, args[0], ::nn::fs::OpenMode_Read));

        ::std::string text;

        auto fileSize = int64_t();
        COMMAND_DO(GetFileSize(&fileSize, hostFsFile));

        if (0 < fileSize)
        {
            ::std::unique_ptr<::std::vector<char>> ptr;
            COMMAND_DO(AllocateBuffer(&ptr, static_cast<size_t>(fileSize)));

            auto readSize = size_t();
            COMMAND_DO(
                ReadFile(&readSize, ptr->data(), ptr->size(), hostFsFile, 0));

            text = ::std::string(ptr->data(), readSize);
        }
        SerialFlashCsvType csvData;
        COMMAND_DO(ParseCsv(&csvData, text));
        COMMAND_DO(WriteSerialFlashFromCsv(csvData, uniquePadId));
        return true;
    }
}

}
