﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/result/result_HandlingUtility.h>

#include "ControllerFirmwareUpdater_Util.h"

namespace
{

// サポートする UniquePadType
const nn::hid::system::UniquePadType SupportedDeviceTypes[] =
{
    nn::hid::system::UniquePadType_FullKeyController,
    nn::hid::system::UniquePadType_RightController,
    nn::hid::system::UniquePadType_LeftController
};

// サポートする UniquePadInterface
const nn::hid::system::UniquePadInterface SupportedInterfaces[] =
{
    nn::hid::system::UniquePadInterface_Bluetooth,
    nn::hid::system::UniquePadInterface_Rail
};

// リソースデータを扱うファイルシステムのマウント名
const char* const RomMountName = "rom";

::nn::lmem::HeapHandle g_HeapHandle;
char* g_MountCache = nullptr;

// サポートしているデバイスタイプか判定
bool IsSupportedDeviceType(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    auto deviceType = nn::hid::system::GetUniquePadType(id);
    for (auto deviceTypeToTest : SupportedDeviceTypes)
    {
        if (deviceType == deviceTypeToTest)
        {
            return true;
        }
    }

    return false;
}

// サポートしている接続インターフェースか判定
bool IsSupportedInterface(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    nn::hid::system::UniquePadInterface padInterface;
    if (nn::hid::system::GetUniquePadInterface(&padInterface, id).IsFailure())
    {
        return false;
    }

    for (auto interfaceToTest : SupportedInterfaces)
    {
        if (padInterface == interfaceToTest)
        {
            return true;
        }
    }

    return false;
}

void* AllocateForFs(size_t size) NN_NOEXCEPT
{
    return ::nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
};

void DeallocateForFs(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::nn::lmem::FreeToExpHeap(g_HeapHandle, p);
};

}  // anonymous

bool IsSupportedDevice(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    return IsSupportedDeviceType(id) && IsSupportedInterface(id);
}

void InitializeFileSystem() NN_NOEXCEPT
{
    // 拡張ヒープ領域
    static const size_t HeapMemorySize = 8 * 1024;
    static char s_HeapMemory[HeapMemorySize];

    // FS 用の拡張ヒープを作成
    g_HeapHandle = ::nn::lmem::CreateExpHeap(
        s_HeapMemory,
        sizeof(s_HeapMemory),
        ::nn::lmem::CreationOption_NoOption);

    // アロケータを登録
    ::nn::fs::SetAllocator(AllocateForFs, DeallocateForFs);

    // RomFs をマウント
    size_t cacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::fs::QueryMountRomCacheSize(&cacheSize));

    g_MountCache = reinterpret_cast<char*>(
        ::nn::lmem::AllocateFromExpHeap(g_HeapHandle, cacheSize));
    NN_ABORT_UNLESS_NOT_NULL(g_MountCache);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::fs::MountRom(RomMountName, g_MountCache, cacheSize));
}

void FinalizeFileSystem() NN_NOEXCEPT
{
    ::nn::fs::Unmount(RomMountName);
    ::nn::lmem::FreeToExpHeap(g_HeapHandle, g_MountCache);
    ::nn::lmem::DestroyExpHeap(g_HeapHandle);
}

void FirmwareVersionReader::SetUniquePadId(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_UniquePadId = id;
    m_IsValid     = true;
    InvalidateVersion();
}

void FirmwareVersionReader::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_IsValid = false;
    InvalidateVersion();
}

bool FirmwareVersionReader::IsValid() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_IsValid;
}

bool FirmwareVersionReader::HasValidVersion() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_HasValidVersion;
}

bool FirmwareVersionReader::Get(nn::hid::system::FirmwareVersion* pOutVersion) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsValid && m_HasValidVersion)
    {
        *pOutVersion = m_Version;
        return true;
    }

    return false;
}

void FirmwareVersionReader::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsValid)
    {
        return;
    }

    if (m_HasValidVersion)
    {
        return;
    }

    if (nn::hid::system::GetFirmwareVersion(
        &m_Version,
        m_UniquePadId).IsSuccess())
    {
        m_HasValidVersion = true;
    }
}

void FirmwareVersionReader::InvalidateVersion() NN_NOEXCEPT
{
    m_HasValidVersion = false;
}

