﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ローカル通信を有効にする
//#define ENABLE_LDN

#include <vector>
#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/init/init_Malloc.h>
#include <nn/oe.h>
#include <nn/oe/oe_HomeButtonControl.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/fs.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#if defined(ENABLE_LDN)
#include <nn/ldn.h>
#endif  // if defined(ENABLE_LDN)

#include "ControllerFirmwareUpdater_PadManager.h"
#include "ControllerFirmwareUpdater_Util.h"
#include "sgx/SimpleGfx.h"

namespace {

const int FrameRate = 60;
const auto FrameWait = nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate);

// 接続可能な Npad
const nn::hid::NpadIdType NpadIds[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld
};

const int NpadIdCountMax = NN_ARRAY_SIZE(NpadIds);

// コントローラーのインジケーター LED 点灯パターン
const uint8_t IndicatorPattern[] = { 0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0D, 0x06, 0x00 };

// アップデートトリガをかけるための L/R/ZL/ZR ボタンの押下状態
struct UpdateTriggerState
{
    bool isLeftLrPressed;
    bool isRightLrPressed;

    void Clear() NN_NOEXCEPT
    {
        isLeftLrPressed  = false;
        isRightLrPressed = false;
    }
};
UpdateTriggerState g_UpdateTriggerStates[nn::hid::system::UniquePadIdCountMax];

// 進捗状況に関する情報
struct ProgressState
{
    bool    existsMcu;
    uint8_t bluetooth;
    uint8_t mcu;

    // メンバの初期化
    void Clear() NN_NOEXCEPT
    {
        existsMcu = false;
        bluetooth = 0;
        mcu       = 0;
    }

    // 指定した UniquePadId に対応する値で初期化
    void Setup(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
    {
        Clear();

        switch (nn::hid::system::GetUniquePadType(id))
        {
        case nn::hid::system::UniquePadType_FullKeyController:
        case nn::hid::system::UniquePadType_RightController:
            existsMcu = true;
            break;

        default:
            break;
        }
    }
};
ProgressState g_ProgressState = { false, 0, 0 };

// ペアリング実行中
bool g_IsOnControllerPairing = false;

// アップデート実行中
bool g_IsUpdating = false;

// 座標指定
const float Display_LineOffset          = 44;
const float Display_ColumnOffset        = 96;

const nns::sgx::Point2D TitlePosition       = { {  48,  16 } };
const nns::sgx::Point2D DebugModePosition   = { { 880,  28 } };
const nns::sgx::Point2D StatusPosition      = { {  48, 108 } };
const nns::sgx::Point2D DeviceListPosition  = { {  48, 288 } };

// 背景色
const nns::sgx::Color BackgroundColor = { { 0xF0, 0xF0, 0xF0, 0xFF } };

// フォントスケール
const float TitleScale          = 1.3f;
const float SectionTitleScale   = 1.0f;
const float MessageScale        = 0.8;
const float IndicatorTextScale  = 0.53f;
const float IndicatorLedScale   = 0.33f;
const float ProgressScale       = 0.65f;
const float CheckMarkScale      = 1.45f;
const float ControllerInfoScale = 0.7f;
const float LoadingIconScale    = 0.85f;

PadManager g_PadManager;
WaitIcon   g_WaitIcon;

nn::os::SystemEventType g_UniquePadUpdateEvent;

nns::sgx::ImageData g_IconImage;
nns::sgx::ImageData g_BarImage;

// 自動モードに関する設定
struct AutoRunMode
{
    bool isEnabled;     // 自動モード有効フラグ

    bool isLeftDone;    // JoyLeft の処理終了 (成否は問わない)
    bool isRightDone;   // JoyRight の処理終了 (成否は問わない)

    // 設定のクリア
    void Clear()
    {
        isEnabled   = false;
        isLeftDone  = false;
        isRightDone = false;
    }
} g_AutoRunMode = { false };

// デバッグモード
bool g_IsDebugMode = false;

//------------------------------------------------------------------------------
//  コマンドライン処理
//------------------------------------------------------------------------------

bool ParseCommandLine()
{
    const int  ArgumentLengthMax   = 256;
    const char ArgumentHelp[]      = "--help";
    const char ArgumentAutoRun[]   = "--auto";
    const char ArgumentDebugMode[] = "--debug";

    // 引数の一致判定
    const auto isArgumentMatch = [](const char* arg, const char* target)
    {
        NN_ASSERT_NOT_NULL(arg);
        NN_ASSERT_NOT_NULL(target);

        int targetLength = nn::util::Strnlen(target, ArgumentLengthMax);
        return nn::util::Strncmp(arg, target, targetLength) == 0 &&
            nn::util::Strnlen(arg, ArgumentLengthMax) == targetLength;
    };

    g_AutoRunMode.Clear();

    // argv[0] (プログラム名) は飛ばす
    for (int i = 1; i < nn::os::GetHostArgc(); i++)
    {
        auto* arg = nn::os::GetHostArgv()[i];
        if (isArgumentMatch(arg, ArgumentHelp))
        {
            // ヘルプを表示して終了
            NN_LOG("usage: ControllerFirmwareUpdater [arguments]\n\n");
            NN_LOG("Available arguments:\n");
            NN_LOG("  --auto\tUpdate rail attached controllers automatically.\n");
            NN_LOG("  --debug\tLaunch as debug mode.\n");
            NN_LOG("  --help\tDisplay this text.\n");
            NN_LOG("\n");
            return false;
        }
        else if (isArgumentMatch(arg, ArgumentAutoRun))
        {
            // レール接続された Joy-Con を自動更新するモード
            g_AutoRunMode.isEnabled = true;
        }
        else if (isArgumentMatch(arg, ArgumentDebugMode))
        {
            // デバッグ/巻き戻し用のデータを使用するモード
            g_IsDebugMode = true;
        }
    }

    return true;
}


//------------------------------------------------------------------------------
//  コントローラーの操作
//------------------------------------------------------------------------------

// コントローラーを初期化
void InitializeController()
{
    nn::hid::InitializeNpad();
    nn::hid::debug::InitializeFirmwareUpdate();

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

    // 使用するNpadを設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Single);

    for (auto id : NpadIds)
    {
        // ツール起動後は念のため接続済みのコントローラーをすべて切断する
        nn::hid::DisconnectNpad(id);
        nn::hid::SetNpadJoyAssignmentModeDual(id);
    }

    nn::hid::system::BindUniquePadConnectionEvent(
        &g_UniquePadUpdateEvent,
        nn::os::EventClearMode_ManualClear);

    g_PadManager.Refresh();
}

// L/ZL の同時押し判定
bool CheckLAndZlRelease(nn::hid::NpadButtonSet& buttons, bool isConnected, UpdateTriggerState* pTriggerState)
{
    if (isConnected)
    {
        if (buttons.Test<nn::hid::NpadButton::ZL>() == true &&
            buttons.Test<nn::hid::NpadButton::L>() == true)
        {
            pTriggerState->isLeftLrPressed = true;
        }
        if (buttons.Test<nn::hid::NpadButton::ZL>() == false &&
            buttons.Test<nn::hid::NpadButton::L>() == false &&
            pTriggerState->isLeftLrPressed == true)
        {
            pTriggerState->isLeftLrPressed = false;
            return true;
        }
    }
    else
    {
        pTriggerState->isLeftLrPressed = false;
    }

    return false;
}

// R/ZR の同時押し判定
bool CheckRAndZrRelease(nn::hid::NpadButtonSet& buttons, bool isConnected, UpdateTriggerState* pTriggerState)
{
    if (isConnected)
    {
        if (buttons.Test<nn::hid::NpadButton::ZR>() == true &&
            buttons.Test<nn::hid::NpadButton::R>() == true)
        {
            pTriggerState->isRightLrPressed = true;
        }
        if (buttons.Test<nn::hid::NpadButton::ZR>() == false &&
            buttons.Test<nn::hid::NpadButton::R>() == false &&
            pTriggerState->isRightLrPressed == true)
        {
            pTriggerState->isRightLrPressed = false;
            return true;
        }
    }
    else
    {
        pTriggerState->isRightLrPressed = false;
    }
    return false;
}

// 無線ペアリングを開始
void StartControllerPairing()
{
    // 自動モードでは無線ペアリングしない
    if (g_AutoRunMode.isEnabled)
    {
        return;
    }

    if (g_IsOnControllerPairing == false)
    {
        nn::btm::system::StartGamepadPairing();
        g_IsOnControllerPairing = true;
    }
}

// 無線ペアリングを停止
void StopControllerPairing()
{
    if (g_IsOnControllerPairing == true)
    {
        nn::btm::system::CancelGamepadPairing();
        g_IsOnControllerPairing = false;
    }
}

// FW 更新の開始
void StartFirmwareUpdate(int deviceIndex) NN_NOEXCEPT
{
    nn::hid::system::UniquePadId id;
    if (g_PadManager.GetUniquePadId(&id, deviceIndex).IsFailure())
    {
        return;
    }

    StopControllerPairing();

    // FW 更新中は HOME 禁止
    nn::oe::BeginBlockingHomeButton();

    if (g_IsDebugMode)
    {
        nn::hid::debug::StartFirmwareUpdateForRevert(id);
    }
    else
    {
        nn::hid::debug::StartFirmwareUpdate(id);
    }

    g_IsUpdating = true;
    g_ProgressState.Setup(id);

    g_PadManager.RequestInvalidateFirmwareVersion();
    g_PadManager.UpdateFirmwareUpdatingDevice();
}

// アップデート開始用にボタンが押されたかどうかの判定
bool IsUpdateTriggered(int index) NN_NOEXCEPT
{
    nn::hid::NpadIdType npadId;
    if (g_PadManager.GetNpadId(&npadId, index).IsFailure())
    {
        g_UpdateTriggerStates[index].Clear();
        return false;
    }

    auto style = nn::hid::GetNpadStyleSet(npadId);

    nn::hid::system::UniquePadId uniquePadId;
    if (g_PadManager.GetUniquePadId(&uniquePadId, index).IsFailure())
    {
        return false;
    }
    auto padType = nn::hid::system::GetUniquePadType(uniquePadId);

    if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
    {
        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, npadId);

        if (CheckLAndZlRelease(
            state.buttons,
            state.attributes.Test<nn::hid::NpadJoyAttribute::IsLeftConnected>() &&
            padType == nn::hid::system::UniquePadType_LeftController,
            &g_UpdateTriggerStates[index]) == true)
        {
            return true;
        }
        if (CheckRAndZrRelease(
            state.buttons,
            state.attributes.Test<nn::hid::NpadJoyAttribute::IsRightConnected>() &&
            padType == nn::hid::system::UniquePadType_RightController,
            &g_UpdateTriggerStates[index]) == true)
        {
            return true;
        }
    }
    else if (style.Test<nn::hid::NpadStyleFullKey>() == true)
    {
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, npadId);

        bool isConnected = state.attributes.Test<nn::hid::NpadAttribute::IsConnected>();
        if (CheckLAndZlRelease(state.buttons, isConnected, &g_UpdateTriggerStates[index]) == true ||
            CheckRAndZrRelease(state.buttons, isConnected, &g_UpdateTriggerStates[index]) == true)
        {
            return true;
        }
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>() == true)
    {
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, npadId);

        if (CheckLAndZlRelease(
            state.buttons,
            state.attributes.Test<nn::hid::NpadJoyAttribute::IsLeftConnected>() &&
            padType == nn::hid::system::UniquePadType_LeftController,
            &g_UpdateTriggerStates[index]) == true)
        {
            return true;
        }
        if (CheckRAndZrRelease(
            state.buttons,
            state.attributes.Test<nn::hid::NpadJoyAttribute::IsRightConnected>() &&
            padType == nn::hid::system::UniquePadType_RightController,
            &g_UpdateTriggerStates[index]) == true)
        {
            return true;
        }
    }
    else
    {
        g_UpdateTriggerStates[index].Clear();
    }

    return false;
}

// コントローラーの更新
void UpdateController()
{
    if (nn::os::TryWaitSystemEvent(&g_UniquePadUpdateEvent))
    {

        nn::os::ClearSystemEvent(&g_UniquePadUpdateEvent);
        g_PadManager.Refresh();
    }

    g_PadManager.Update();

    if (g_AutoRunMode.isEnabled || g_IsUpdating)
    {
        // アップデート中はアップデート開始操作を受け付けない
        for (int i = 0; i < g_PadManager.GetPadCountMax(); i++)
        {
            g_UpdateTriggerStates[i].Clear();
        }
        return;
    }

    for (int i = 0; i < g_PadManager.GetPadCount(); i++)
    {
        // バージョン取得中は操作を受け付けない
        {
            ::nn::hid::system::FirmwareVersion version;
            NN_UNUSED(version);
            if (!g_PadManager.GetFirmwareVersion(&version, i))
            {
                g_UpdateTriggerStates[i].Clear();
                continue;
            }
        }

        if (IsUpdateTriggered(i))
        {
            StartFirmwareUpdate(i);
            break;
        }
    }
}


//------------------------------------------------------------------------------
//  コントローラーの状態の出力
//------------------------------------------------------------------------------

// 影付き文字列を描画
void DrawShadowText(float x, float y, const char* text, const nns::sgx::Color& color)
{
    nns::sgx::SetTextColor(nns::sgx::Colors::DarkGray());
    nns::sgx::DrawText(x + 1, y + 1, text);
    nns::sgx::SetTextColor(color);
    nns::sgx::DrawText(x, y, text);
}

// 単一の進捗バーの描画
void PrintProgressBar(float x, float y, const char* caption, uint8_t progress, float animRate)
{
    NN_ASSERT_NOT_NULL(caption);

    bool isCompleted = progress >= 100;

    const float gaugeLength = 300;
    float fillLength = gaugeLength * progress / 100;

    // 背景
    nns::sgx::FillRectangle(x, y + 24, gaugeLength, 16, nns::sgx::Colors::Gray());

    // 本体
    float barWidth  = g_BarImage.size.width  / 3;
    float barHeight = g_BarImage.size.height / 2;
    nns::sgx::Rectangle destRect = { { x, y + 24, fillLength, 16 } };
    if (isCompleted)
    {
        nns::sgx::Rectangle srcRect = { { 0, barHeight, barWidth, barHeight } };
        nns::sgx::DrawImage(g_BarImage, destRect, srcRect);
    }
    else
    {
        float barX = barWidth * 2 * animRate;
        nns::sgx::Rectangle srcRect = { { barX, 0, barWidth, barHeight } };
        nns::sgx::DrawImage(g_BarImage, destRect, srcRect);
    }

    // 枠
    nns::sgx::DrawRectangle(x, y + 24, gaugeLength, 16, nns::sgx::Colors::Black(), 1);

    nns::sgx::SetTextScale(ProgressScale);
    nns::sgx::DrawText(x, y, "%s", caption);
    nns::sgx::DrawText(x + gaugeLength + 8, y + 22, "%d%%", progress);

    // 完了時 (100%) はチェックマークを描画
    if (isCompleted)
    {
        nns::sgx::SetTextScale(CheckMarkScale);
        nns::sgx::SetTextColor(nns::sgx::Colors::Green());
        nns::sgx::DrawText(x + 140, y, "\uE14B");  // チェックマーク
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    }
}

// 全進捗バーの描画
void PrintAllProgressBar(float x, float y)
{
    static int s_BarDuration = 0;

    float animRate = 1.0f - s_BarDuration / 90.0f;
    s_BarDuration = (s_BarDuration + 1) % 90;

    PrintProgressBar(x, y, "Bluetooth", g_ProgressState.bluetooth, animRate);

    if (g_ProgressState.existsMcu)
    {
        const float mcuOffsetX = 420;
        PrintProgressBar(x + mcuOffsetX, y, "NFC", g_ProgressState.mcu, animRate);
    }
};

// 指定した NpadId に対応するインジケーターの表示パターンを取得
uint8_t GetIndicatorPattern(const nn::hid::NpadIdType& npadId)
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        if (npadId == NpadIds[i])
        {
            return IndicatorPattern[i];
        }
    }

    return 0;
}

// コントローラーインジケータの描画
void PrintIndicator(float x, float y, int idIndex)
{
    nn::hid::NpadIdType id;
    if (g_PadManager.GetNpadId(&id, idIndex).IsFailure())
    {
        return;
    }

    if (id == nn::hid::NpadId::Handheld)
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Smoke());
        nns::sgx::SetTextScale(IndicatorTextScale);
        nns::sgx::DrawText(x - 16, y - 16, "OnRail");
    }
    else
    {
        nns::sgx::SetTextScale(IndicatorLedScale);
        auto pattern = GetIndicatorPattern(id);
        for (int i = 0; i < 4; i++)
        {
            int mask = 1 << i;
            nns::sgx::SetTextColor((pattern & mask) != 0 ? nns::sgx::Colors::LimeGreen() : nns::sgx::Colors::DimGray());
            nns::sgx::DrawText(x - 10 + i * 10, y - 12, "■");
        }
    }
}

float PrintAndStep(float x, float y, const char* text)
{
    NN_ASSERT_NOT_NULL(text);

    nns::sgx::Size size;
    nns::sgx::GetTextDrawSize(&size, text);
    nns::sgx::DrawText(x, y, text);

    return size.width;
}

// アップデート開始前の説明を描画
void PrintStartUpdate(float x, float y)
{
    nns::sgx::SetTextScale(MessageScale);

    bool isDeviceConnected = false;
    for (int i = 0; i < g_PadManager.GetPadCount(); i++)
    {
        nn::hid::system::UniquePadId id;
        if (g_PadManager.GetUniquePadId(&id, i).IsSuccess() &&
            IsSupportedDevice(id))
        {
            isDeviceConnected = true;
            break;
        }
    }

    if (isDeviceConnected)
    {
        const float buttonOffsetY = -10;
        float currentX = x;
        currentX += PrintAndStep(currentX, y, "Press ");

        nns::sgx::SetTextColor(nns::sgx::Colors::Crymson());
        nns::sgx::SetTextScale(MessageScale * 1.5f);
        currentX += PrintAndStep(currentX, y + buttonOffsetY, "\uE0A4+\uE0A6");  // L+ZL

        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
        nns::sgx::SetTextScale(MessageScale);
        currentX += PrintAndStep(currentX, y, " or ");

        nns::sgx::SetTextColor(nns::sgx::Colors::Crymson());
        nns::sgx::SetTextScale(MessageScale * 1.5f);
        currentX += PrintAndStep(currentX, y + buttonOffsetY, "\uE0A5+\uE0A7");  // R+ZR

        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
        nns::sgx::SetTextScale(MessageScale);
        currentX += PrintAndStep(currentX, y, " and RELEASE to start firmware update.");
    }
    else
    {
        nns::sgx::DrawText(x, y,
            "Connect controller to update firmware.\n"
            "You can register new controller by pressing SYNC button.");
    }
}

// コントローラーのデバイスタイプを描画
void PrintDeviceType(float x, float y, nn::hid::system::UniquePadType deviceType)
{
    nns::sgx::SetTextColor(nns::sgx::Colors::Black());

    // アイコン + デバイスタイプを描画
    switch (deviceType)
    {
    case nn::hid::system::UniquePadType_FullKeyController:
        nns::sgx::DrawText(x, y, "\uE12C Pro Controller");
        break;
    case nn::hid::system::UniquePadType_LeftController:
        nns::sgx::DrawText(x, y, "\uE123 Joy-Con (L)");
        break;
    case nn::hid::system::UniquePadType_RightController:
        nns::sgx::DrawText(x, y, "\uE124 Joy-Con (R)");
        break;
    default:
        nns::sgx::DrawText(x, y, "\uE152 Unknown Device ");
        break;
    }
}

// FW アップデートの要否を判定
::nn::Result IsUpdateAvailable(
    bool* pOutNeedsUpdate,
    ::nn::hid::system::UniquePadId id,
    const ::nn::hid::system::FirmwareVersion& currentVersion,
    const ::nn::hid::system::FirmwareVersion& newVersion)
{
    NN_ASSERT_NOT_NULL(pOutNeedsUpdate);

    // エラーの有無を判定するため、先に更新確認 API を呼ぶ
    NN_RESULT_DO(::nn::hid::system::IsFirmwareUpdateAvailable(pOutNeedsUpdate, id));

    // バージョン一致判定
    auto isSameVersion = [](
        const ::nn::hid::system::FirmwareVersion& currentVersion,
        const ::nn::hid::system::FirmwareVersion& newVersion)
    {
        return
            currentVersion.major == newVersion.major &&
            currentVersion.minor == newVersion.minor &&
            currentVersion.micro == newVersion.micro &&
            currentVersion.revision == newVersion.revision;
    };

    if (g_IsDebugMode)
    {
        // デバッグモードでは、公開 API の結果を使用せずバージョンのみで判定する
        *pOutNeedsUpdate = !isSameVersion(currentVersion, newVersion);
    }
    else if (!isSameVersion(currentVersion, newVersion))
    {
        // 不具合で FW 巻き戻し等が発生した場合に、要更新扱いしたい FW があればここで判定
    }

    NN_RESULT_SUCCESS;
}

// FW アップデートの要否を表示
void PrintDeviceUpdateNeeds(
    float x,
    float y,
    ::nn::hid::system::UniquePadId id,
    const ::nn::hid::system::FirmwareVersion& currentVersion)
{
    // FW 更新中は何もしない
    if (g_PadManager.IsFirmwareUpdating(id))
    {
        return;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    };

    // 更新先 FW バージョンの確認
    auto getAvailableVersion = g_IsDebugMode ?
        ::nn::hid::debug::GetAvailableFirmwareVersionForRevert :
        ::nn::hid::system::GetAvailableFirmwareVersion;

    ::nn::hid::system::FirmwareVersion newVersion;
    if (getAvailableVersion(&newVersion, id).IsFailure())
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Crymson());
        nns::sgx::DrawText(x, y, "Update is unavailable");
        return;
    }

    // 更新有無の表示
    bool needsUpdate = false;
    auto result = IsUpdateAvailable(&needsUpdate, id, currentVersion, newVersion);
    if (nn::hid::system::ResultFirmwareUpdateRequireBluetooth::Includes(result))
    {
        DrawShadowText(x, y, "Need update with Bluetooth Connection. Detach from rail.", nns::sgx::Colors::Orange());
    }
    else if (nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(result) ||
        needsUpdate)
    {
        char text[256];
        nn::util::SNPrintf(text, sizeof(text), "%s %02x.%02x.%02x.%02x",
            g_IsDebugMode ? "Available" : "Need update to",
            newVersion.major, newVersion.minor, newVersion.micro, newVersion.revision);
        DrawShadowText(x, y, text, nns::sgx::Colors::Orange());
    }
    else if (result.IsSuccess())
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Smoke());
        nns::sgx::DrawText(x, y, "Already updated");
    }
}

// コントローラーの情報を描画
void PrintDeviceInfo(int idIndex,
                     float x,
                     float y,
                     nn::hid::system::UniquePadId id,
                     nn::hid::system::UniquePadType deviceType)
{
    const float versionOffset     = 220;
    const float statusOffset      = 460;
    const float loadingIconOffset =  26;

    // FW 更新中表示
    if (g_PadManager.IsFirmwareUpdating(id))
    {
        const float fillWidth = 800;
        nns::sgx::FillRectangle(x - 16, y - 16, fillWidth, Display_LineOffset, nns::sgx::Colors::Navy().BlendAlpha(16));
        nns::sgx::SetTextColor(nns::sgx::Colors::RoyalBlue());

        g_WaitIcon.Draw(x + statusOffset, y);
        nns::sgx::DrawText(x + statusOffset + loadingIconOffset, y, "Updating...");
    }

    // 各種情報を表示
    PrintDeviceType(x, y, deviceType);
    PrintIndicator(x, y, idIndex);

    nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    nns::sgx::SetTextScale(ControllerInfoScale);

    ::nn::hid::system::FirmwareVersion version;
    if (g_PadManager.GetFirmwareVersion(&version, idIndex))
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
        nns::sgx::DrawText(x + versionOffset, y, "Ver. %02x.%02x.%02x.%02x", version.major, version.minor, version.micro, version.revision);

        PrintDeviceUpdateNeeds(x + statusOffset, y, id, version);
    }
    else if (!g_PadManager.IsFirmwareUpdating(id))
    {
        g_WaitIcon.Draw(x + versionOffset, y);
        nns::sgx::DrawText(x + versionOffset + loadingIconOffset, y, "Reading version...");
    }
}

// 接続されているコントローラーの一覧を描画
void PrintDeviceList(float x, float y)
{
    const float lineY      = y + 36;
    const float lineLength = 800;

    nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    nns::sgx::SetTextScale(SectionTitleScale);
    nns::sgx::DrawText(x, y, "\uE122 Connected controllers");
    nns::sgx::DrawLine(x, lineY, x + lineLength, lineY, nns::sgx::Colors::Gray(), 1);
    nns::sgx::SetTextScale(ControllerInfoScale);

    float dx = x + 16;
    float dy = y + 60;

    bool isDeviceConnected = false;

    for (int i = 0; i < g_PadManager.GetPadCount(); i++)
    {
        nn::hid::system::UniquePadId id;
        if (g_PadManager.GetUniquePadId(&id, i).IsFailure())
        {
            continue;
        }

        auto deviceType = nn::hid::system::GetUniquePadType(id);
        PrintDeviceInfo(i, dx, dy, id, deviceType);
        dy += Display_LineOffset;
        isDeviceConnected = true;
    }

    if (!isDeviceConnected)
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Smoke());
        nns::sgx::DrawText(dx, dy, "No controller connected");
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    }
}

// タイトルの描画
void PrintTitle()
{
    const nns::sgx::Point2D titleOffset = { {  72, 12 } };
    const nns::sgx::Point2D lineOffset  = { { -16, 72 } };
    const float lineLength = 1216;

    auto& textColor = g_IsDebugMode ? nns::sgx::Colors::Orange() : nns::sgx::Colors::Black();

    float x = TitlePosition.x;
    float y = TitlePosition.y;
    nns::sgx::DrawImage(g_IconImage, x, y);

    x += titleOffset.x;
    y += titleOffset.y;
    nns::sgx::SetTextScale(TitleScale);
    DrawShadowText(x, y, "Controller Firmware Updater", textColor);

    x = TitlePosition.x + lineOffset.x;
    y = TitlePosition.y + lineOffset.y;
    nns::sgx::DrawLine(x, y, x + lineLength, y, nns::sgx::Colors::DimGray(), 1);

    if (g_IsDebugMode)
    {
        DrawShadowText(DebugModePosition.x, DebugModePosition.y, "【 DEBUG MODE 】", textColor);
    }

    nns::sgx::SetTextColor(nns::sgx::Colors::Black());
}

// FW アップデートの進捗情報を表示
void PrintFirmwareUpdateStatusProgress(
    float x,
    float y,
    nn::hid::debug::FirmwareUpdateStage stage,
    uint8_t progress)
{
    // Bluetooth 更新の進捗率を更新
    auto updateBluetoothProgress = [](uint8_t progress)
    {
        g_ProgressState.bluetooth = progress;
    };

    // MCU 更新の進捗率を更新
    auto updateMcuProgress = [](uint8_t progress)
    {
        g_ProgressState.mcu = progress;
    };

    float textX = x + 32;

    if (g_IsUpdating)
    {
        // ローディングアイコン
        nns::sgx::SetTextScale(LoadingIconScale);
        nns::sgx::SetTextColor(nns::sgx::Colors::NeonBlue());
        g_WaitIcon.Draw(x, y);
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());

        // 進捗バー
        const float barOffsetY = 92;
        PrintAllProgressBar(textX, y + barOffsetY);
    }

    nns::sgx::SetTextScale(MessageScale);

    switch (stage)
    {
    case nn::hid::debug::FirmwareUpdateStage_Preparing:
        nns::sgx::DrawText(textX, y, "Preparing firmware update...");
        break;
    case nn::hid::debug::FirmwareUpdateStage_BluetoothDownload:
        nns::sgx::DrawText(textX, y, "Downloading Bluetooth firmware...");
        updateBluetoothProgress(std::min<uint8_t>(progress / 2, 49));
        break;
    case nn::hid::debug::FirmwareUpdateStage_BluetoothVerify:
        nns::sgx::DrawText(textX, y, "Verifying Bluetooth firmware...");
        updateBluetoothProgress(progress / 2 + 49);
        break;
    case nn::hid::debug::FirmwareUpdateStage_BluetoothCommit:
        nns::sgx::DrawText(textX, y, "Commiting Bluetooth birmware...");
        updateBluetoothProgress(99);
        break;
    case nn::hid::debug::FirmwareUpdateStage_Reboot1:
        nns::sgx::DrawText(textX, y,
            "Rebooting... Please Wait.\n"
            "If this do not complete in 60 seconds, connect controller manually by pressing a button.");
        updateBluetoothProgress(100);
        break;

    case nn::hid::debug::FirmwareUpdateStage_Reboot2:
        nns::sgx::DrawText(textX, y, "Rebooting... Please Wait.");
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuVersionCheck:
        nns::sgx::DrawText(textX, y, "Checking NFC Firmware version...");
        updateMcuProgress(1);
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuBoot:
        nns::sgx::DrawText(textX, y, "Booting NFC into firmware update mode...");
        updateMcuProgress(1);
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuErasing:
        nns::sgx::DrawText(textX, y, "Erasing NFC firmware...");
        updateMcuProgress(progress);
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuWriting:
        nns::sgx::DrawText(textX, y, "Writing NFC firmware...");
        updateMcuProgress(progress);
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuReboot:
        nns::sgx::DrawText(textX, y, "Rebooting NFC...");
        updateMcuProgress(progress);
        break;
    case nn::hid::debug::FirmwareUpdateStage_McuVerify:
        nns::sgx::DrawText(textX, y, "Verifying NFC firmware...");
        updateMcuProgress(progress);
        break;
    case nn::hid::debug::FirmwareUpdateStage_Completed:
        DrawShadowText(textX, y, "Firmware update completed.", nns::sgx::Colors::LimeGreen());
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

}  // NOLINT(impl/function_size)

// FW アップデートのエラー情報を表示
void PrintFirmwareUpdateStatusError(float x, float y, nn::Result result)
{
    nns::sgx::SetTextScale(MessageScale);

    if (nn::hid::system::ResultFirmwareUpdateNotStarted::Includes(result))
    {
        nns::sgx::DrawText(x, y, "No firmware update in progress.");
    }
    else
    {
        if (nn::hid::system::ResultUniquePadDisconnected::Includes(result))
        {
            nns::sgx::DrawText(x, y, "Controller disconnected.");
        }
        else  if (nn::hid::ResultFirmwareUpdateBluetoothVerifyError::Includes(result))
        {
            nns::sgx::DrawText(x, y, "Bluetooth firmware verify error occurred.");
        }
        else  if (nn::hid::ResultFirmwareUpdateBluetoothUnknownError::Includes(result))
        {
            nns::sgx::DrawText(x, y, "Bluetooth firmware unknown error occurred.");
        }
        else  if (nn::hid::ResultFirmwareUpdateMcuVerifyError::Includes(result))
        {
            nns::sgx::DrawText(x, y, "Nfc firmware verify error occurred.");
        }
        nns::sgx::SetTextColor(nns::sgx::Colors::Crymson());
        nns::sgx::DrawText(x, y, "\nRetry firmware update.");
        nns::sgx::SetTextColor(nns::sgx::Colors::Black());
    }

    PrintStartUpdate(x, y + Display_ColumnOffset);
}

// FW アップデートの状況を表示
void PrintFirmwareUpdateStatus(
    float x,
    float y,
    nn::hid::debug::FirmwareUpdateStage stage,
    uint8_t progress,
    nn::Result result)
{
    if (result.IsSuccess())
    {
        PrintFirmwareUpdateStatusProgress(x, y, stage, progress);
    }
    else
    {
        PrintFirmwareUpdateStatusError(x, y, result);
    }
}

// 画面描画
void Print(nn::hid::debug::FirmwareUpdateStage stage, uint8_t progress, nn::Result result)
{
    nns::sgx::BeginRender();

    PrintTitle();
    PrintFirmwareUpdateStatus(StatusPosition.x, StatusPosition.y, stage, progress, result);
    PrintDeviceList(DeviceListPosition.x, DeviceListPosition.y);

    nns::sgx::EndRender();
}

// フレーム毎の更新処理
void UpdateFrame(bool isFocused)
{
    // BG 中に描画するとループが止まるので、描画は FG 中のみ
    if (!isFocused)
    {
        nn::os::SleepThread(FrameWait);
        return;
    }

    uint8_t progress;
    nn::hid::debug::FirmwareUpdateStage stage;
    auto result = nn::hid::debug::GetFirmwareUpdateStage(&stage, &progress);

    bool isPrevUpdating = g_IsUpdating;
    g_IsUpdating = result.IsSuccess() && stage != nn::hid::debug::FirmwareUpdateStage_Completed;
    if (g_IsUpdating)
    {
        StopControllerPairing();
    }
    else
    {
        if (isPrevUpdating)
        {
            g_PadManager.UpdateFirmwareUpdatingDevice();
        }

        g_PadManager.InvalidateFirmwareVersionIfNeeded();
        StartControllerPairing();
        nn::oe::EndBlockingHomeButton();
    }

    UpdateController();
    Print(stage, progress, result);

    g_WaitIcon.Update();
}

// 自動実行モードの更新
void UpdateAutoRunMode()
{
    if (!g_AutoRunMode.isEnabled)
    {
        return;
    }

    if (g_IsUpdating)
    {
        return;
    }

    // 指定した type の Handheld デバイスを指すインデックスを探す
    auto findHandheldDeviceIndex = [](nn::hid::system::UniquePadType type)
    {
        for (int i = 0; i < g_PadManager.GetPadCount(); i++)
        {
            nn::hid::system::UniquePadId id;
            if (g_PadManager.GetUniquePadId(&id, i).IsFailure())
            {
                continue;
            }

            // レール接続でないデバイスは無視
            nn::hid::system::UniquePadInterface interface;
            if (nn::hid::system::GetUniquePadInterface(&interface, id).IsFailure() ||
                interface != nn::hid::system::UniquePadInterface_Rail)
            {
                continue;
            }

            if (nn::hid::system::GetUniquePadType(id) == type)
            {
                return i;
            }
        }

        return -1;
    };

    if (!g_AutoRunMode.isLeftDone)
    {
        // レール接続された左コンを更新。未接続ならスキップ
        auto index = findHandheldDeviceIndex(nn::hid::system::UniquePadType_LeftController);
        if (index >= 0)
        {
            NN_APP_LOG0("<AutoMode> Updating left controller\n");
            StartFirmwareUpdate(index);
        }
        else
        {
            NN_APP_LOG0("<AutoMode> Left controller is not connected on rail. Skipped.\n");
        }

        g_AutoRunMode.isLeftDone = true;
        return;
    }
    else if (!g_AutoRunMode.isRightDone)
    {
        // レール接続された右コンを更新。未接続ならスキップ
        auto index = findHandheldDeviceIndex(nn::hid::system::UniquePadType_RightController);
        if (index >= 0)
        {
            NN_APP_LOG0("<AutoMode> Updating right controller\n");
            StartFirmwareUpdate(index);
        }
        else
        {
            NN_APP_LOG0("<AutoMode> Right controller is not connected on rail. Skipped.\n");
        }

        g_AutoRunMode.isRightDone = true;
        return;
    }
}

// プログラムの実行を継続するか
bool IsRunning(bool* pOutIsFocused)
{
    NN_ASSERT_NOT_NULL(pOutIsFocused);

    // 通知メッセージの処理
    nn::oe::Message message;
    if (nn::oe::TryPopNotificationMessage(&message))
    {
        switch (message)
        {
        case nn::oe::MessageFocusStateChanged:
            {
                auto state = nn::oe::GetCurrentFocusState();
                *pOutIsFocused = (state == nn::oe::FocusState_InFocus);
                if (!(*pOutIsFocused))
                {
                    // フォーカスが外れたら無線ペアリングを停止
                    StopControllerPairing();
                }
            }
            break;
        case nn::oe::MessageExitRequest:
            // アプリ終了
            return false;
        default:
            // 何もしない
            break;
        }
    }

    // 自動実行モードの終了判定
    if (g_AutoRunMode.isEnabled)
    {
        if (!g_IsUpdating &&
            g_AutoRunMode.isLeftDone &&
            g_AutoRunMode.isRightDone)
        {
            // 更新が完了したらアプリ終了
            return false;
        }
    }

    return true;
}

}

extern "C" void nnMain()
{
    if (!ParseCommandLine())
    {
        return;
    }

    InitializeFileSystem();

    nns::sgx::Initialize();
    nns::sgx::SetClearColor(BackgroundColor);
    nns::sgx::LoadImage(&g_IconImage, "rom:/UpdaterIcon.bmp");
    nns::sgx::LoadImage(&g_BarImage, "rom:/ProgressBar.bmp");

    InitializeController();

#if defined(ENABLE_LDN)
    // ローカル通信を有効化 (desc に ldn:u を追加する必要あり)
    nn::ldn::Initialize();
    nn::ldn::OpenAccessPoint();

    {
        nn::ldn::NetworkConfig network = {};
        network.intentId     = nn::ldn::MakeIntentId(0, 0);
        network.channel      = nn::ldn::AutoChannel;
        network.nodeCountMax = 1;
        network.localCommunicationVersion = 0;

        nn::ldn::SecurityConfig security = {};
        security.securityMode   = nn::ldn::SecurityMode_Product;
        security.passphraseSize = nn::ldn::PassphraseSizeMin;

        nn::ldn::UserConfig user = {};
        user.userName[0] = 'A';

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ldn::CreateNetwork(network, security, user));
    }
#endif  // if defined(ENABLE_LDN)

    // 終了通知を受け取る
    nn::oe::EnterExitRequestHandlingSection();

    // フォーカス変化通知を受け取る
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

    bool isFocused = true;  // 起動時は Focus 状態であることが保証されている
    do
    {
        UpdateFrame(isFocused);
        UpdateAutoRunMode();
    } while (IsRunning(&isFocused));

    NN_APP_LOG0("Finished\n");

#if defined(ENABLE_LDN)
    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
    nn::ldn::Finalize();
#endif  // if defined(ENABLE_LDN)

    StopControllerPairing();
    nn::os::DestroySystemEvent(&g_UniquePadUpdateEvent);

    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Suspend);
    nn::oe::LeaveExitRequestHandlingSection();
    nn::oe::EndBlockingHomeButton();  // 念のため

    nns::sgx::DestroyImage(&g_IconImage);
    nns::sgx::DestroyImage(&g_BarImage);
    nns::sgx::Finalize();
    FinalizeFileSystem();
}
