﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/bluetooth/bluetooth_Api.h>

#include "BluetoothSettingTool_Program.h"

using namespace BluetoothSettingTool;

namespace
{

typedef struct
{
    bool isBoostEnabled;
    bool isAfhEnabled;
} BtSettingStatus;

typedef struct
{
    void (*function)(void);
} BluetoothSettingToolFunc;

const std::string ENABLED = "Enabled";
const std::string DISABLED = "Disabled";

void ChangeBluetoothBoostSetting(void);
bool g_BluetoothBoostEnabled = true;

void ChangeBluetoothAfhSetting(void);
bool g_BluetoothAfhEnabled = true;

bool g_BluetoothSettingChanged = false;

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

void Initialize()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif
    nn::bluetooth::InitializeBluetoothDriver();

    g_BluetoothBoostEnabled = nn::bluetooth::IsBluetoothBoostSettingEnabled();
    g_BluetoothAfhEnabled = nn::bluetooth::IsBluetoothAfhSettingEnabled();
}

void Finalize()
{
    NPad::Finalize();
    nn::bluetooth::FinalizeBluetoothDriver();
}

void ChangeBluetoothBoostSetting()
{
    NN_LOG("%s called\n", __FUNCTION__);

    bool before = g_BluetoothBoostEnabled;

    nn::bluetooth::EnableBluetoothBoostSetting(!g_BluetoothBoostEnabled);

    g_BluetoothBoostEnabled = nn::bluetooth::IsBluetoothBoostSettingEnabled();

    if(before == g_BluetoothBoostEnabled)
    {
        NN_LOG("Failed to change bluetooth boost setting\n");
    }
    else
    {
        g_BluetoothSettingChanged = true;
    }

    return;
}

void ChangeBluetoothAfhSetting()
{
    NN_LOG("%s called\n", __FUNCTION__);

    bool before = g_BluetoothAfhEnabled;

    nn::bluetooth::EnableBluetoothAfhSetting(!g_BluetoothAfhEnabled);

    g_BluetoothAfhEnabled = nn::bluetooth::IsBluetoothAfhSettingEnabled();

    if(before == g_BluetoothAfhEnabled)
    {
        NN_LOG("Failed to change bluetooth AFH setting\n");
    }
    else
    {
        g_BluetoothSettingChanged = true;
    }

    return;
}

}   // namespace

class BluetoothSettingToolApp : public Application
{
    private:
        Page                                  m_Page;
        Label                                 m_Title;
        Label                                 m_SettingItemsInd;
        Label                                 m_SettingStatusInd;
        Selector<BluetoothSettingToolFunc>    m_Selector;

        Label                                 m_NeedReboot;
        bool                                  m_IsNeedRebootDisplayed;

        Label                                 m_Manual;

        BluetoothSettingToolFunc              m_BluetoothBoostSettingFunc;
        Label                                 m_BluetoothBoostEnabled;

        BluetoothSettingToolFunc              m_BluetoothAfhSettingFunc;
        Label                                 m_BluetoothAfhEnabled;

        bool                                  m_Selected;

    public:
        BluetoothSettingToolApp();
        virtual ~BluetoothSettingToolApp();
        virtual void ApplicationImpl();
};

BluetoothSettingToolApp::BluetoothSettingToolApp()
{
    m_Selected = false;

    m_BluetoothBoostSettingFunc.function = ChangeBluetoothBoostSetting;
    m_BluetoothAfhSettingFunc.function = ChangeBluetoothAfhSetting;

    m_Page.Add(m_Title);
    m_Title.Text        = "Bluetooth Setting Tool";
    m_Title.X           = DISPLAY_CONTENT_START_X;
    m_Title.Y           = DISPLAY_CONTENT_START_Y;
    m_Title.Alignment   = MiddleLeft;
    m_Title.FitSize();

    m_Page.Add(m_SettingItemsInd);
    m_SettingItemsInd.Text        = "Setting List:";
    m_SettingItemsInd.X           = DISPLAY_CONTENT_START_X;
    m_SettingItemsInd.Y           = DISPLAY_CONTENT_START_Y + DISPLAY_LINE_HEIGHT * 2;
    m_SettingItemsInd.Alignment   = MiddleLeft;
    m_SettingItemsInd.FitSize();

    m_Page.Add(m_SettingStatusInd);
    m_SettingStatusInd.Text       = "Status:";
    m_SettingStatusInd.X          = Display::GetInstance().GetWidth() / 2 + 120;
    m_SettingStatusInd.Y          = DISPLAY_CONTENT_START_Y + DISPLAY_LINE_HEIGHT * 2;
    m_SettingStatusInd.Alignment  = MiddleLeft;
    m_SettingStatusInd.FitSize();

    m_Page.Add(m_Selector);

    m_Selector.X        = DISPLAY_CONTENT_START_X;
    m_Selector.Y        = DISPLAY_CONTENT_START_Y + 90;
    m_Selector.Height   = DISPLAY_CONTENT_HEIGHT;

    m_Selector.Register("Change Bluetooth Boost Setting", &m_BluetoothBoostSettingFunc);
    m_Selector.Register("Change Bluetooth AFH Setting", &m_BluetoothAfhSettingFunc);

    m_Page.Add(m_BluetoothBoostEnabled);
    m_BluetoothBoostEnabled.Text       = ENABLED;       // TORIAEZU
    m_BluetoothBoostEnabled.X          = Display::GetInstance().GetWidth() / 2 + 150;
    m_BluetoothBoostEnabled.Y          = DISPLAY_CONTENT_START_Y + DISPLAY_LINE_HEIGHT * 3;
    m_BluetoothBoostEnabled.Alignment  = MiddleLeft;
    m_BluetoothBoostEnabled.FitSize();
    m_BluetoothBoostEnabled.SetFgColor(ToColor(GREEN_YELLOW));

    m_Page.Add(m_BluetoothAfhEnabled);
    m_BluetoothAfhEnabled.Text       = ENABLED;       // TORIAEZU
    m_BluetoothAfhEnabled.X          = Display::GetInstance().GetWidth() / 2 + 150;
    m_BluetoothAfhEnabled.Y          = DISPLAY_CONTENT_START_Y + DISPLAY_LINE_HEIGHT * 4;
    m_BluetoothAfhEnabled.Alignment  = MiddleLeft;
    m_BluetoothAfhEnabled.FitSize();
    m_BluetoothAfhEnabled.SetFgColor(ToColor(GREEN_YELLOW));

    m_NeedReboot.Text = "REBOOT the system to apply setting changes";
    m_NeedReboot.X = DISPLAY_CONTENT_START_X;
    m_NeedReboot.Y = DISPLAY_PAD_START_Y - DISPLAY_LINE_HEIGHT * 2;
    m_NeedReboot.Alignment  = MiddleLeft;
    m_NeedReboot.FitSize();
    m_NeedReboot.SetFgColor(ToColor(RED));

    m_Page.Add(m_Manual);
    m_Manual.X = DISPLAY_PAD_START_X;
    m_Manual.Y = DISPLAY_PAD_START_Y;
    m_Manual.Text = "<Up/Down>:Select   <A>:Change Setting";
    m_Manual.FitSize();
}

BluetoothSettingToolApp::~BluetoothSettingToolApp()
{

}

void BluetoothSettingToolApp::ApplicationImpl()
{
    static nn::os::Tick tick;
    static bool isHoldStart;
    static bool isHoldSelect;
    static bool isHoldZr;
    static bool isHoldZl;

    Pad &pad = Pad::GetInstance();
    bool isTriggered = pad.IsTrigger(Button::A) || pad.IsTrigger(Button::B) ||
                       pad.IsTrigger(Button::X) || pad.IsTrigger(Button::Y) ||
                       pad.IsTrigger(Button::L) || pad.IsTrigger(Button::R) ||
                       pad.IsTrigger(Button::ZL) || pad.IsTrigger(Button::ZR) ||
                       pad.IsTrigger(Button::LEFT) || pad.IsTrigger(Button::RIGHT) ||
                       pad.IsTrigger(Button::UP) || pad.IsTrigger(Button::DOWN) ||
                       pad.IsTrigger(Button::START) || pad.IsTrigger(Button::SELECT);

    if(g_BluetoothSettingChanged && !m_IsNeedRebootDisplayed)
    {
        m_IsNeedRebootDisplayed = true;
        m_Page.Add(m_NeedReboot);
    }

    if(g_BluetoothBoostEnabled)
    {
        m_BluetoothBoostEnabled.Text       = ENABLED;
        m_BluetoothBoostEnabled.SetFgColor(ToColor(GREEN_YELLOW));
    }
    else
    {
        m_BluetoothBoostEnabled.Text       = DISABLED;       // TORIAEZU
        m_BluetoothBoostEnabled.SetFgColor(ToColor(RED));
    }
    if(g_BluetoothAfhEnabled)
    {
        m_BluetoothAfhEnabled.Text       = ENABLED;
        m_BluetoothAfhEnabled.SetFgColor(ToColor(GREEN_YELLOW));
    }
    else
    {
        m_BluetoothAfhEnabled.Text       = DISABLED;       // TORIAEZU
        m_BluetoothAfhEnabled.SetFgColor(ToColor(RED));
    }

    m_Page.Show(Display::GetInstance());

    if(!m_Selected)
    {
        isHoldStart     = isHoldStart   || pad.IsHold(Button::START);
        isHoldSelect    = isHoldSelect  || pad.IsHold(Button::SELECT);
        isHoldZl        = isHoldZl      || pad.IsHold(Button::ZL);
        isHoldZr        = isHoldZr      || pad.IsHold(Button::ZR);

        if( pad.IsTrigger(Button::A))
        {
            pad.Clear();
            m_Selected = true;
        }
        else if(isHoldStart && isHoldSelect && isHoldZl && isHoldZr)
        {
            Exit();
        }
    }
    else
    {
        if(pad.IsTrigger(Button::START))
        {
            m_Selected = false;
        }
        else
        {
            m_Selector.GetSelectingItem()->function();
            m_Selected = false;
        }
    }

    if( (nn::os::GetSystemTick() - tick).ToTimeSpan() >= nn::TimeSpan::FromMilliSeconds(1000) )
    {
        isHoldStart = false;
        isHoldSelect = false;
        isHoldZr = false;
        isHoldZl = false;
        tick = nn::os::GetSystemTick();
    }

    Display::GetInstance().DisableConsoleOutput();
    if(isTriggered)
    {
        Display::GetInstance().EnableConsoleOutput();
    }
}

extern "C" void nnMain()
{
    NN_LOG("Bluetooth Setting Tool Now Running\n");

    Initialize();

    Display::EnableDisplayOutput();

    BluetoothSettingToolApp app;
    app.Execute();

    Finalize();

    return;
}

