﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Log.h>
#include "BluetoothSettingTool_Display.h"

using namespace std;

namespace BluetoothSettingTool {

/*
  Drawer
*/

Display* Display::m_pDisplay = nullptr;
bool Display::m_EnableConsoleOutput = false;
bool Display::m_EnableDisplayOutput = false;
const int RenderWidth = 1280;
const int RenderHeight = 720;

// const size_t TextRenderer::VISIBLE_POOL_MEMORY_SIZE = 32 * 1024 * 1024;
// const size_t TextRenderer::INVISIBLE_POOL_MEMORY_SIZE = 40 * 1024 * 1024;
const size_t TextRenderer::VISIBLE_POOL_MEMORY_SIZE = 128 * 1024 * 1024;
const size_t TextRenderer::INVISIBLE_POOL_MEMORY_SIZE = 128 * 1024 * 1024;

TextRenderer::TextRenderer() :
m_Cs(false, 0),
m_pVisiblePoolMemory(nullptr),
m_pInvisiblePoolMemory(nullptr),
m_pMemoryPoolStart(nullptr),
m_MemoryPoolOffset(0),
m_pInvisibleMemoryPoolStart(nullptr),
m_InvisibleMemoryPoolOffset(0),
m_SamplerDescriptorBaseIndex(0),
m_TextureDescriptorBaseIndex(0),
m_pMemory(nullptr),
m_pMemoryHeap(nullptr),
m_pDebugFontHeap(nullptr)
{
    m_FontWidth = 0;
    m_FontHeight = 0;
}

TextRenderer::~TextRenderer()
{}

//------------------------------------------------------------------------------
//  デバイスを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    m_Device.Initialize(info);
}

//------------------------------------------------------------------------------
//  メモリプールを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pVisiblePoolMemory = malloc(VISIBLE_POOL_MEMORY_SIZE + alignment);

    m_pMemoryPoolStart = nn::util::BytePtr(m_pVisiblePoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pMemoryPoolStart, nn::util::align_down(VISIBLE_POOL_MEMORY_SIZE,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_MemoryPool.Initialize(&m_Device, info);

    m_MemoryPoolOffset = 0;
}

void TextRenderer::InitializeInvisibleMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible
        | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pInvisiblePoolMemory = malloc(INVISIBLE_POOL_MEMORY_SIZE + alignment);

    m_pInvisibleMemoryPoolStart = nn::util::BytePtr(m_pInvisiblePoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pInvisibleMemoryPoolStart, nn::util::align_down(INVISIBLE_POOL_MEMORY_SIZE,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_InvisibleMemoryPool.Initialize(&m_Device, info);

    m_InvisibleMemoryPoolOffset = 0;
}

//------------------------------------------------------------------------------
//  スワップチェーンを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(RenderWidth);
    info.SetHeight(RenderHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);
    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);
        m_InvisibleMemoryPoolOffset = nn::util::align_up(m_InvisibleMemoryPoolOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));
        m_SwapChain.Initialize(&m_Device, info, &m_InvisibleMemoryPool, m_InvisibleMemoryPoolOffset, size);
        m_InvisibleMemoryPoolOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, NULL, 0, 0);
    }
}

//------------------------------------------------------------------------------
//  キューを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    m_Queue.Initialize(&m_Device, info);
}

//------------------------------------------------------------------------------
//  コマンドバッファを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    m_CommandBuffer.Initialize(&m_Device, info);
}

//
//------------------------------------------------------------------------------
//  ビューポートシザーを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast< float >(RenderWidth));
        viewportInfo.SetHeight(static_cast< float >(RenderHeight));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(RenderWidth);
        scissorInfo.SetHeight(RenderHeight);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    m_ViewportScissor.Initialize(&m_Device, info);
}

//------------------------------------------------------------------------------
//  サンプラディスクリプタプールの初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(m_SamplerDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_SamplerDescriptorPool.Initialize(&m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  テクスチャディスクリプタプールの初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(m_TextureDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_TextureDescriptorPool.Initialize(&m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  レイヤを初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &m_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &m_pLayer, m_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( m_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの初期化
//------------------------------------------------------------------------------
void TextRenderer::InitializeGfxObjects()
{
    //m_pMemoryHeap.Reset(malloc(1024 * 1024 * 32));
    m_pMemoryHeap.Reset(malloc(1024 * 1024 * 64));
    m_pMemory = m_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &m_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &m_SamplerDescriptorBaseIndex);
#endif

    InitializeMemoryPool();
    InitializeInvisibleMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    //NN_ASSERT(m_pMemoryHeap.Distance(m_pMemory.Get()) < 1024 * 1024 * 32);
    NN_ASSERT(m_pMemoryHeap.Distance(m_pMemory.Get()) < 1024 * 1024 * 64);
    NN_ASSERT(m_MemoryPoolOffset < VISIBLE_POOL_MEMORY_SIZE);
    NN_ASSERT(m_InvisibleMemoryPoolOffset < INVISIBLE_POOL_MEMORY_SIZE);
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの終了処理
//------------------------------------------------------------------------------
void TextRenderer::FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);

    m_ViewportScissor.Finalize(&m_Device);
    m_CommandBuffer.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_InvisibleMemoryPool.Finalize(&m_Device);
    m_MemoryPool.Finalize(&m_Device);
    m_Device.Finalize();

    free(m_pMemoryHeap.Get());
    free(m_pVisiblePoolMemory);
    free(m_pInvisiblePoolMemory);
}

void TextRenderer::Initialize()
{
    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();

    // デバッグフォント初期化
    const int charCountMax = 4 * 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(&m_Device, info);
    m_pDebugFontHeap.Reset(new uint8_t[debugFontHeapSize]);

    m_Writer.Initialize(
        &m_Device,
        info,
        m_pDebugFontHeap.Get(),
        debugFontHeapSize,
        nullptr,
        0,
        0
        );

    m_Writer.SetDisplayWidth(RenderWidth);
    m_Writer.SetDisplayHeight(RenderHeight);
    m_Writer.SetTextureDescriptor(&m_TextureDescriptorPool, m_TextureDescriptorBaseIndex);
    m_Writer.SetSamplerDescriptor(&m_SamplerDescriptorPool, m_SamplerDescriptorBaseIndex);

    m_Width = RenderWidth;
    m_Height = RenderHeight;

    // 等幅フォントにする
    m_Writer.SetFixedWidthEnabled(true);
    m_Writer.SetFixedWidth(DEFAULT_FIXED_WIDTH);
    m_FixedWidth = m_Writer.GetFixedWidth();

    m_Writer.SetFontSize(DEFAULT_FONT_WIDTH, DEFAULT_FONT_HEIGHT);
    m_FontWidth = m_Writer.GetFontWidth();
    m_FontHeight = m_Writer.GetFontHeight();
}

void TextRenderer::Finalize()
{
    // デバッグフォント終了
    m_Writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(m_pDebugFontHeap.Get());
    m_pDebugFontHeap.Reset(nullptr);

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( m_pLayer );
    nn::vi::CloseDisplay( m_pDisplay );
    nn::vi::Finalize();
}

void TextRenderer::SetColor(Color color)
{
    m_Writer.SetTextColor(ToUnormColor(color));
}

void TextRenderer::SetScale(float x, float y)
{
    m_ScaleX = x;
    m_ScaleY = y;
    m_Writer.SetScale(m_ScaleX, m_ScaleY);

    m_FontWidth = m_Writer.GetFontWidth();
    m_FontHeight = m_Writer.GetFontHeight();
}

void TextRenderer::SetFixedWidth(float width)
{
    m_FixedWidth = width;
    m_Writer.SetFixedWidth(m_FixedWidth);
}

void TextRenderer::SetFontSize(float width, float height)
{
    m_FontWidth = width;
    m_FontHeight = height;
    m_Writer.SetFontSize(m_FontWidth, m_FontHeight);
}

void TextRenderer::SetCursor(float x, float y)
{
    m_Writer.SetCursor(x, y);
}

void TextRenderer::DrawText(float x, float y, const string& text)
{
    if( text.size() == 0 )
    {
        return;
    }

    SetCursor(x, y);
    m_Writer.Print(text.c_str());
}

void TextRenderer::VPrint(const char *format, va_list formatArg)
{
    //m_Cs.Lock();
    static char buffer[1024];
    vsprintf(buffer, format, formatArg);
    m_Writer.Print(buffer);
    //m_Cs.Unlock();
}

void TextRenderer::Print(float x, float y, const char* format, ...)
{
    SetCursor(x, y);

    va_list list;
    va_start(list, format);
    VPrint(format, list);
    va_end(list);
}

void TextRenderer::Print(const char* format, ...)
{
    va_list list;
    va_start(list, format);
    VPrint(format, list);
    va_end(list);
}

void TextRenderer::SwapBuffers()
{
    //m_Cs.Lock();
    // コマンド生成
    m_CommandBuffer.Reset();
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
                                            nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));
    m_CommandBuffer.AddCommandMemory(&m_MemoryPool, m_MemoryPoolOffset, 1024 * 1024);
    m_pMemory.AlignUp(256);
    m_CommandBuffer.AddControlMemory(m_pMemory.Get(), 256);
    m_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = m_SwapChain.AcquireNextScanBufferView();
        m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
        m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);
        //m_CommandBuffer.ClearColor(pTarget, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
        m_CommandBuffer.ClearColor(pTarget, 0.0f, 0.0f, 0.0f, 1.0f, NULL);
        m_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);
        m_CommandBuffer.SetViewportScissorState(&m_ViewportScissor);

        // デバッグフォント用のコマンド生成
        m_Writer.Draw(&m_CommandBuffer);
    }
    m_CommandBuffer.End();

    // コマンドの実行
    m_Queue.ExecuteCommand(&m_CommandBuffer, NULL);

    // 結果の表示
    m_Queue.Present(&m_SwapChain, 1);
    m_Queue.Sync();

#ifdef NN_BUILD_CONFIM_OS_WIN
    nn::hws::ProcessMessage();
#endif

    //m_Cs.Unlock();
}


/*
  Display
*/

Display& Display::GetInstance()
{
    if( m_pDisplay == nullptr )
    {
        m_pDisplay = new Display();

        if( m_pDisplay->m_EnableDisplayOutput && m_pDisplay->m_pRenderSystem == nullptr )
        {
            m_pDisplay->m_pRenderSystem = new TextRenderer();
            m_pDisplay->m_pRenderSystem->Initialize();
            m_pDisplay->m_pRenderSystem->SetColor(ToColor(WHITE));
        }
    }

    return *m_pDisplay;
}

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/

Display::Display()
{
    m_pRenderSystem = nullptr;
}

Display::~Display()
{
    if( m_pDisplay )
    {
        if( m_pRenderSystem )
        {
            m_pRenderSystem->Finalize();
            delete m_pRenderSystem;
        }

        delete m_pDisplay;
    }
}



}    // WlanTest
