﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv.h>
#include <glv_font.h>

#include "Base/BcatTestApp_Sequence.h"
#include "Base/BcatTestApp_Glv.h"
#include "Base/BcatTestApp_Hid.h"
#include "Base/BcatTestApp_Utility.h"
#include "Common/BcatTestApp_Menu.h"

namespace app
{
namespace
{
//----------------------------------------------------------------
// メニューの内部コンソール用プリンタ
//
void DrawMenuConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    app::Menu* p = reinterpret_cast<app::Menu*>( arg );
    if ( ! p )
    {
        return;
    }

    app::ColorSet cset( app::GetConsoleColorR(*attr),
                        app::GetConsoleColorG(*attr),
                        app::GetConsoleColorB(*attr) );
    app::SetGlvColor( cset );

    glv::draw::text( string,
                     p->GetPosition().GetLeft() + x * p->GetFontSize() + + 10,
                     p->GetPosition().GetTop() + y * (p->GetFontSize() + p->GetLinesDiff()) + 10,
                     p->GetFontSize() );
}
void DrawMenuConsoleString16( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    app::Menu* p = reinterpret_cast<app::Menu*>( arg );
    if ( ! p )
    {
        return;
    }

    app::ColorSet cset( app::GetConsoleColorR(*attr),
                        app::GetConsoleColorG(*attr),
                        app::GetConsoleColorB(*attr) );
    app::SetGlvColor( cset );

    glv::draw::text( string,
                     p->GetPosition().GetLeft() + x * p->GetFontSize() + + 10,
                     p->GetPosition().GetTop() + y * (p->GetFontSize() + p->GetLinesDiff()) + 10,
                     p->GetFontSize() );
}
} //namespace

//----------------------------------------------------------------
// メニューの描画コールバック
//
void Menu::DrawMenuConsole( void* arg ) NN_NOEXCEPT
{
    app::Menu* p = reinterpret_cast<app::Menu*>( arg );
    if ( ! p )
    {
        return;
    }

    // 非アクティブ中は描画しないかの判定
    if ( ! p->m_IsDrawInactive && ! app::sequence::IsDrawCallbackActive() )
    {
        return;
    }

    // フレーム
    app::DrawFrameRectangle( p->m_Position, p->m_ColorBack, p->m_ColorFrame, p->m_FrameWidth );

    // メニュー項目後ろの四角
    int cx = p->m_Position.GetLeft() + (p->m_MenuPositionX + 1)  * p->GetFontSize() + 10;
    int cy = p->m_Position.GetTop() + (p->m_MenuPositionY + p->m_MenuCursor) * (p->GetFontSize() + p->GetLinesDiff()) + 10;
    app::DrawRectangle( cx, cy, p->m_Position.GetRight() - 10, cy + p->GetFontSize() + 4, p->m_ColorSelect );

    // 描画コールバック
    if ( p->m_Callback )
    {
        (p->m_Callback)( p->m_pConsole, p->m_CallbackArg );
    }
    if ( p->m_Callback16 )
    {
        (p->m_Callback16)( p->m_pConsole16, p->m_CallbackArg );
    }

    // 内部コンソールの描画
    if ( p->m_pConsole )
    {
        (p->m_pConsole)->Display();
    }
    if ( p->m_pConsole16 )
    {
        (p->m_pConsole16)->Display();
    }
}

//----------------------------------------------------------------
// メニューの内部コンソール作成
//
void Menu::CreateConsole( ConsoleSize consoleSize,
                          app::RectanglePosition pos, int consoleSizeX, int consoleSizeY,
                          int menuPositionX, int menuPositionY,
                          int fontSize, int linesDiff, int priority, int tag ) NN_NOEXCEPT
{
    // まだコンソールを作っていない確認
    NN_ASSERT( ! m_pConsole && ! m_pConsole16 );

    m_Position = pos;
    m_MenuPositionX = menuPositionX;
    m_MenuPositionY = menuPositionY;
    m_FontSize = fontSize;
    m_LinesDiff = linesDiff;
    m_Tag = tag;

    // コンソール用の領域を作成
    switch( consoleSize )
    {
        case app::ConsoleSize_Char:
            {
                m_pConsole = new app::FixedProportionalConsole<char>;
                int32_t bufferSize = APP_CONSOLE_BUFFER_SIZE( consoleSizeX + 20, consoleSizeY ); // 20 byte ほど余裕をみる
                m_ConsoleBuffer = new char[ bufferSize ];

                // コンソールに領域と描画関数をセット
                m_pConsole->SetBuffer( m_ConsoleBuffer, bufferSize, consoleSizeX, consoleSizeY );
                m_pConsole->SetPrinter( DrawMenuConsoleString, this );
            }
            break;
        case app::ConsoleSize_Char16_t:
            {
                m_pConsole16 = new app::FixedProportionalConsole<char16_t>;
                int32_t bufferSize = APP_CONSOLE_BUFFER_SIZE_CHAR16( consoleSizeX + 20, consoleSizeY ); // 20 byte ほど余裕をみる
                m_ConsoleBuffer = new char[ bufferSize ];

                // コンソールに領域と描画関数をセット
                m_pConsole16->SetBuffer( m_ConsoleBuffer, bufferSize, consoleSizeX, consoleSizeY );
                m_pConsole16->SetPrinter( DrawMenuConsoleString16, this );

            }
            break;
        default:
            break;
    }

    // 描画コールバック設定
    app::sequence::AddDrawCallback( DrawMenuConsole, this, priority, tag );
}

//----------------------------------------------------------------
// メニューの内部コンソール用領域を解放
//
void Menu::DestroyConsole() NN_NOEXCEPT
{
    if ( m_ConsoleBuffer )
    {
        delete[] m_ConsoleBuffer;
        delete m_pConsole;
        delete m_pConsole16;
        m_ConsoleBuffer = nullptr;
        m_pConsole = nullptr;
        m_pConsole16 = nullptr;
    }

    // タグを指定して特定の描画コールバックだけ削除
    app::sequence::RemoveDrawCallbackWithTag( DrawMenuConsole, m_Tag );
}
//----------------------------------------------------------------
// コンソール取得
//
app::FixedProportionalConsole<char>* Menu::GetConsole() NN_NOEXCEPT
{
    return m_pConsole;
}
app::FixedProportionalConsole<char16_t>* Menu::GetConsole16() NN_NOEXCEPT
{
    return m_pConsole16;
}

//----------------------------------------------------------------
// コンソール消去
//
void Menu::ClearConsole() NN_NOEXCEPT
{
    if ( m_pConsole )
    {
        m_pConsole->Clear();
    }
    if ( m_pConsole16 )
    {
        m_pConsole16->Clear();
    }
}

//----------------------------------------------------------------
// メニューの描画コールバックの設定
//
void Menu::SetCallback( DrawCallback callback, void* arg ) NN_NOEXCEPT
{
    m_Callback = callback;
    m_CallbackArg = arg;
}
void Menu::SetCallback16( DrawCallback16 callback, void* arg ) NN_NOEXCEPT
{
    m_Callback16 = callback;
    m_CallbackArg = arg;
}
//----------------------------------------------------------------
// コンソールカラーなどの設定
//
void Menu::SetBackColor( int colorBack, int colorFrame, float frameWidth ) NN_NOEXCEPT
{
    m_ColorBack = colorBack;
    m_ColorFrame = colorFrame;
    m_FrameWidth = frameWidth;
}
//----------------------------------------------------------------
// コンソールカラーなどの設定
//
void Menu::SetSelectColor( int colorSelect ) NN_NOEXCEPT
{
    m_ColorSelect = colorSelect;
}
//----------------------------------------------------------------
// アイテムパラメータ設定
//
void Menu::SetItemParameter( int itemNum, int firstPos ) NN_NOEXCEPT
{
    m_MenuItemMax = itemNum;
    m_MenuCursor = firstPos;
    DrawCursor();
}
//----------------------------------------------------------------
// hid イベント設定
//
void Menu::SetHidEvent( const glv::HidEvents* pEvent ) NN_NOEXCEPT
{
    m_pHidEvent = pEvent;
}

//----------------------------------------------------------------
// カーソル位置設定
//
void Menu::SetCursor( int n ) NN_NOEXCEPT
{
    if ( n >= 0 && n < m_MenuItemMax )
    {
        m_MenuCursor = n;
    }
}
//----------------------------------------------------------------
// カーソル位置取得
//
int Menu::GetCursor() NN_NOEXCEPT
{
    return m_MenuCursor;
}
//----------------------------------------------------------------
// カーソルのコンソールプリント
//
void Menu::DrawCursor() NN_NOEXCEPT
{
    if ( m_pConsole )
    {
        m_pConsole->PrintfEx( m_MenuPositionX, m_MenuPositionY + m_MenuCursor, app::ConsoleColor_Yellow, ">" );
    }
    if ( m_pConsole16 )
    {
        m_pConsole16->PrintfEx( m_MenuPositionX, m_MenuPositionY + m_MenuCursor, app::ConsoleColor_Yellow, u">" );
    }
}
//----------------------------------------------------------------
// カーソルのコンソールからの消去
//
void Menu::ClearCursor() NN_NOEXCEPT
{
    if ( m_pConsole )
    {
        m_pConsole->ClearPartial( m_MenuPositionX, m_MenuPositionY + m_MenuCursor );
    }
    if ( m_pConsole16 )
    {
        m_pConsole16->ClearPartial( m_MenuPositionX, m_MenuPositionY + m_MenuCursor );
    }
}

//----------------------------------------------------------------
// 更新
//
void Menu::Update() NN_NOEXCEPT
{
    app::Pad pad( *m_pHidEvent );

    // 上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        if ( m_MenuItemMax > 0 )
        {
            ClearCursor();
            SetCursor( app::RotateWithinRange( m_MenuCursor, -1, 0, m_MenuItemMax - 1 ) );
            DrawCursor();
            m_IsUpdated = true;
        }
    }
    // 下
    else if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        if ( m_MenuItemMax > 0 )
        {
            ClearCursor();
            SetCursor( app::RotateWithinRange( m_MenuCursor, 1, 0, m_MenuItemMax - 1 ) );
            DrawCursor();
            m_IsUpdated = true;
        }
    }
}
//----------------------------------------------------------------
// 更新情報取得
//
bool Menu::IsUpdated() NN_NOEXCEPT
{
    return m_IsUpdated;
}
//----------------------------------------------------------------
// 更新情報クリア
//
void Menu::ClearUpdated() NN_NOEXCEPT
{
    m_IsUpdated = false;
}

//----------------------------------------------------------------
// ボタンチェック(ok)
//
int Menu::CheckButtonOk() NN_NOEXCEPT
{
    app::Pad pad( *m_pHidEvent );
    if ( pad.IsButtonDownA() )
    {
        m_IsUpdated = true;
        return m_MenuCursor;
    }
    return -1;
}
//----------------------------------------------------------------
// ボタンチェック(cancel)
//
bool Menu::CheckButtonCancel() NN_NOEXCEPT
{
    app::Pad pad( *m_pHidEvent );
    return ( pad.IsButtonDownB() );
}

} //namespece app
