﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv_font.h>

#include "Base/BcatTestApp_Glv.h"
#include "Base/BcatTestApp_Sequence.h"
#include "Common/BcatTestApp_Console.h"
#include "Common/BcatTestApp_BaseDisplay.h"

namespace app
{
namespace
{
    // コンソールバッファ(システムコンソール用)
    const int SystemConsoleWidth = 128;
    const int SystemConsoleHeight = 50;
    const int SystemConsoleViewHeightSmall = 12;
    const int SystemConsoleViewHeightBig = 39;
    const int SystemConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( SystemConsoleWidth, SystemConsoleHeight );
    char g_SystemConsoleBuffer[ SystemConsoleBufferSize ];
    const int SystemConsoleLeft      = 0;
    const int SystemConsoleRight     = 1000;
    const int SystemConsoleTopWide   = 0;
    const int SystemConsoleTopNarrow = 490;
    const int SystemConsoleBottom    = 720;

    // コンソールバッファ(スクリーンコンソール用)
    const int ScreenConsoleWidth = 100;
    const int ScreenConsoleHeight = 33;
    const int ScreenConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ScreenConsoleWidth, ScreenConsoleHeight );
    char g_ScreenConsoleBuffer2[ ScreenConsoleBufferSize ];

    // コンソールバッファ(ヘルプ用)
    const int HelpConsoleWidth = 300;
    const int HelpConsoleHeight = 1;
    const int HelpConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE_CHAR16( HelpConsoleWidth, HelpConsoleHeight );
    char g_HelpConsoleBuffer3[ HelpConsoleBufferSize ];

    // コンソール
    // システムコンソール
    app::ScrollConsole<char> g_SystemConsole;
    // スクリーンコンソール
    app::FixedProportionalConsole<char> g_ScreenConsole;
    // ヘルプコンソール
    app::FixedConsole<char16_t> g_HelpConsole;

    // システムコンソールの状態
    bool g_IsSystemConsoleDisplayed = false;
    bool g_IsSystemConsoleWide = false;

} //namespace

//----------------------------------------------------------------
// コンソール取得
//
// システムコンソール
app::ScrollConsole<char>& GetSystemConsole() NN_NOEXCEPT
{
    return g_SystemConsole;
}
// スクリーンコンソール
app::FixedProportionalConsole<char>& GetScreenConsole() NN_NOEXCEPT
{
    return g_ScreenConsole;
}
// ヘルプコンソール
app::FixedConsole<char16_t>& GetHelpConsole() NN_NOEXCEPT
{
    return g_HelpConsole;
}

//----------------------------------------------------------------
// コンソールプリンタ
// システムコンソールが使う
void DrawConsoleString_FontSize16( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    glv::draw::color( app::GetConsoleColorR(*attr),
                      app::GetConsoleColorG(*attr),
                      app::GetConsoleColorB(*attr) );
    glv::draw::lineWidth( 1 );

    if ( g_IsSystemConsoleWide )
    {
        glv::draw::text( string, 10 + x * 16 , y * 18 + 20, 16 );
    }
    else
    {
        glv::draw::text( string, 10 + x * 16 , y * 18 + 500, 16 );
    }
}

//----------------------------------------------------------------
// コンソールプリンタ
// スクリーンコンソールが使う
void DrawConsoleString_FontSize24( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    glv::draw::color( app::GetConsoleColorR(*attr),
                      app::GetConsoleColorG(*attr),
                      app::GetConsoleColorB(*attr) );
    glv::draw::lineWidth( 1 );
    glv::draw::text( string, 10 + x * 24 , y * (24 + 4), 24 );
}

//----------------------------------------------------------------
// コンソールプリンタ
// ヘルプコンソールが使う
void DrawConsoleString_ForHelp( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    //app::DrawProportionalConsoleGeneric( 10, 680, 4, 24, 16, 24, x, y, string, attr, stringLen, arg );
    if ( IsEnglish() )
    {
        app::DrawProportionalConsoleGeneric16( 10, 680, 20, 12, 20, x, y, string, attr, stringLen, arg );
    }
    else
    {
        app::DrawProportionalConsoleGeneric16( 10, 680, 24, 16, 24, x, y, string, attr, stringLen, arg );
    }
}

//----------------------------------------------------------------
// システムコンソールの描画コールバック
void DrawSystemConsoles( void* arg ) NN_NOEXCEPT
{
    if ( g_IsSystemConsoleDisplayed )
    {
        // バックを描画
        if ( g_IsSystemConsoleWide )
        {
            app::DrawRectangle( SystemConsoleLeft, SystemConsoleTopWide, SystemConsoleRight, SystemConsoleBottom, DrawColorSet_SystemConsoleBack );
        }
        else
        {
            app::DrawRectangle( SystemConsoleLeft, SystemConsoleTopNarrow, SystemConsoleRight, SystemConsoleBottom, DrawColorSet_SystemConsoleBack );
        }

        // コンソールの描画
        g_SystemConsole.Display();
    }
}

// スクリーンコンソール/ヘルプコンソールの描画コールバック
void DrawScreenConsoles( void* arg ) NN_NOEXCEPT
{
    g_ScreenConsole.Display();
    g_HelpConsole.Display();
}

//----------------------------------------------------------------
// システムコンソールの外部からの状態切替
//
void SwitchSystemConsoleStatus( bool sw1, bool sw2 ) NN_NOEXCEPT
{
    if ( sw1 )
    {
        g_IsSystemConsoleDisplayed = ! g_IsSystemConsoleDisplayed;
    }

    if ( g_IsSystemConsoleDisplayed )
    {
        if ( sw2 )
        {
            if ( g_IsSystemConsoleWide )
            {
                g_SystemConsole.SetViewHeight( SystemConsoleViewHeightSmall );
                g_SystemConsole.ScrollBottom();
            }
            else
            {
                g_SystemConsole.SetViewHeight( SystemConsoleViewHeightBig );
                g_SystemConsole.ScrollTop();
                g_SystemConsole.ScrollBottom();
            }
            g_IsSystemConsoleWide = ! g_IsSystemConsoleWide;
        }
    }
}

//----------------------------------------------------------------
// コンソール初期化
void InitializeConsole() NN_NOEXCEPT
{
    g_SystemConsole.SetBuffer( g_SystemConsoleBuffer, SystemConsoleBufferSize, SystemConsoleWidth, SystemConsoleHeight );
    g_SystemConsole.SetPrinter( DrawConsoleString_FontSize16 );
    g_SystemConsole.SetViewHeight( SystemConsoleViewHeightSmall );
    g_IsSystemConsoleDisplayed = false;
    g_IsSystemConsoleWide = false;

    g_ScreenConsole.SetBuffer( g_ScreenConsoleBuffer2, ScreenConsoleBufferSize, ScreenConsoleWidth, ScreenConsoleHeight );
    g_ScreenConsole.SetPrinter( DrawConsoleString_FontSize24 );

    g_HelpConsole.SetBuffer( g_HelpConsoleBuffer3, HelpConsoleBufferSize, HelpConsoleWidth, HelpConsoleHeight );
    g_HelpConsole.SetPrinter( DrawConsoleString_ForHelp );

    app::sequence::AddDrawCallback( DrawSystemConsoles, nullptr, app::DrawPriority_SystemConsole );
    app::sequence::AddDrawCallback( DrawScreenConsoles, nullptr, app::DrawPriority_ScreenConsole );
}

//----------------------------------------------------------------
// エラーコード(result)のシステムコンソールへの出力
void PrintErrorCode( nn::Result& result ) NN_NOEXCEPT
{
    if ( result.IsSuccess() )
    {
        GetSystemConsole().Printf("Success\n" );
    }
    else
    {
        GetSystemConsole().Printf("Error: %08x, %03d-%04d\n",
                                  result.GetInnerValueForDebug(),
                                  result.GetModule(), result.GetDescription());
    }
}

//----------------------------------------------------------------
// 固定コンソール用の、汎用プリンタ(char)
void DrawFixedConsoleGeneric( int sx, int sy, int lsy, int fontSize, int fx, int fy, int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char tmpStr[2];
    tmpStr[1] = '\0';

    for( int i=0; i<stringLen; i++ )
    {
        char ch = *string++;
        int at = *attr++;
        if ( ! ch )
        {
            continue;
        }
        if ( ch > 0x80 )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        float curX = sx + (x + i) * fx + (fx - w) / 2.F;
        float curY = sy + y * (fy + lsy);

        glv::draw::text( tmpStr, curX, curY, fontSize );
    }
}

//----------------------------------------------------------------
// 固定コンソール用の、汎用プリンタ(char)
void DrawFixedConsoleGeneric( int sx, int sy, float fontSize, float fx, float fy, int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char tmpStr[2];
    tmpStr[1] = '\0';

    for( int i=0; i<stringLen; i++ )
    {
        char ch = *string++;
        int at = *attr++;
        if ( ! ch )
        {
            continue;
        }
        if ( ch > 0x80 )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        float curX = sx + (x + i) * fx + (fx - w) / 2.F;
        float curY = sy + y * fy;

        glv::draw::text( tmpStr, curX, curY, fontSize );
    }
}

//----------------------------------------------------------------
// 固定コンソール用の、汎用プリンタ(char16_t)
void DrawFixedConsoleGeneric16( int sx, int sy, float fontSize, float fx, float fy, int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char16_t tmpStr[2];
    tmpStr[1] = u'\0';

    for( int i=0; i<stringLen; i++ )
    {
        char16_t ch = *string++;
        int at = *attr++;
        if ( ! ch )
        {
            continue;
        }
        //NN_LOG("ch=%x \n", ch);
        if ( ch > 0xffff )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        float curX = sx + (x + i) * fx + (fx - w) / 2.F;
        float curY = sy + y * fy;

        glv::draw::text( tmpStr, curX, curY, fontSize );
    }
}

//----------------------------------------------------------------
// プロポーショナル表示コンソール用の、汎用プリンタ
void DrawProportionalConsoleGeneric( int sx, int sy, int lsy, int fontSize, int fx, int fy, int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    float curX = sx + x*fx;
    float curY = sy + y*(fy + lsy);

    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char tmpStr[2];
    tmpStr[1] = '\0';

    for( int i=0; i<stringLen; i++ )
    {
        char ch = *string++;
        int at = *attr++;

        if ( ! ch )
        {
            curX += fx;
            continue;
        }
        if ( ch > 0x80 )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        glv::draw::text( tmpStr, curX, curY, fontSize );
        curX += w;
    }
}

//----------------------------------------------------------------
// プロポーショナル表示コンソール用の、汎用プリンタ(char)
void DrawProportionalConsoleGeneric( int sx, int sy, float fontSize, float fx, float fy, int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    float curX = sx + x * fx;
    float curY = sy + y * fy;

    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char tmpStr[2];
    tmpStr[1] = '\0';

    for( int i=0; i<stringLen; i++ )
    {
        char ch = *string++;
        int at = *attr++;

        if ( ! ch )
        {
            curX += fx;
            continue;
        }
        if ( ch > 0x80 )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        glv::draw::text( tmpStr, curX, curY, fontSize );
        curX += w;
    }
}

//----------------------------------------------------------------
// プロポーショナル表示コンソール用の、汎用プリンタ(char16_t)
void DrawProportionalConsoleGeneric16( int sx, int sy, float fontSize, float fx, float fy, int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    float curX = sx + x * fx;
    float curY = sy + y * fy;

    // フォント幅を調べるため
    glv::Font f;
    f.size(fontSize);
    f.lineSpacing(1.5f);
    f.tabSpaces(4);

    char16_t tmpStr[2];
    tmpStr[1] = u'\0';

    for( int i=0; i<stringLen; i++ )
    {
        char16_t ch = *string++;
        int at = *attr++;

        if ( ! ch )
        {
            curX += fx;
            continue;
        }
        if ( ch > 0xffff )
        {
            ch = '?';
        }
        tmpStr[0] = ch;

        float w;
        float h;
        f.getBounds( w, h, tmpStr );

        glv::draw::color( app::GetConsoleColorR(at),
                          app::GetConsoleColorG(at),
                          app::GetConsoleColorB(at) );

        glv::draw::text( tmpStr, curX, curY, fontSize );
        curX += w;
    }
}

} //namespece app
