﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv.h>
#include <nn/nifm.h>

#include <nn/util/util_FormatString.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

#include <nn/nsd/nsd_ApiForMenu.h>

#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_Types.h>
#include <nn/bcat/bcat_Api.h>
#if defined( APP_BCAT_SYSTEM_DEBUG_TOOL )
#include <nn/bcat/detail/bcat_TypesDetail.h>
#include <nn/bcat/bcat_ApiAdmin.h>
#endif

#include "Base/BcatTestApp_ConsoleCore.h"
#include "Base/BcatTestApp_Glv.h"
#include "Base/BcatTestApp_Hid.h"
#include "Base/BcatTestApp_Sequence.h"
#include "Common/BcatTestApp_Console.h"
#include "Common/BcatTestApp_BaseDisplay.h"


namespace app
{
    const int DrawPriority_BaseFrame = 10;
    const RectanglePosition Position_BaseFrame = { 2, 2, 1278, 718 };
    const int DrawColorSet_BaseFrame = app::ColorSet_Black;

namespace
{
    bool g_IsConnected = false; // 接続状態
    bool g_IsMountedCacheStorage = false; // マウント状態

    nn::ApplicationId g_CurrentApplication = { 0 }; // 現在のアプリケーション
    bool g_IsCurrentApplicationAvailable = false; // 現在のアプリケーションが決定されているか

    bool g_IsEnglish = false; // 言語(true で英語)

} //namespace

//----------------------------------------------------------------
// 言語
//
bool IsEnglish() NN_NOEXCEPT
{
    return g_IsEnglish;
}
void SetEnglish( bool isEnglish ) NN_NOEXCEPT
{
    g_IsEnglish = isEnglish;
}

//----------------------------------------------------------------
// 接続状態
//
bool IsConnected() NN_NOEXCEPT
{
    return nn::nifm::IsNetworkAvailable();
}

//----------------------------------------------------------------
// マウント状態
bool IsMountedCacheStorage() NN_NOEXCEPT
{
    return g_IsMountedCacheStorage;
}

//----------------------------------------------------------------
// ネットワークの接続状態チェック
//
void CheckNetworkConnected() NN_NOEXCEPT
{
    // 現在接続状態であれば、nifm に問い合わせて
    // 切断されていれば状態を変化させる
    if ( g_IsConnected && ! IsConnected() )
    {
        g_IsConnected = false;
        app::GetSystemConsole().Printf("Network is disconnected for some reason.\n" );
        app::GetSystemConsole().Printf("nn::nifm::IsNetworkAvailable: false\n" );
    }
}

//----------------------------------------------------------------
// ネットワークの接続状態
void SetNetworkConnectionStatus( bool isConnected ) NN_NOEXCEPT
{
    g_IsConnected = isConnected;
}

//----------------------------------------------------------------
// 現在のアプリ
//
// セット
void SetCurrentApplication( nn::ApplicationId& app ) NN_NOEXCEPT
{
    g_IsCurrentApplicationAvailable = true;
    g_CurrentApplication = app;
}
// 取得
nn::ApplicationId& GetCurrentApplication() NN_NOEXCEPT
{
    return g_CurrentApplication;
}
// 設定されているか
bool IsCurrentApplicationAvailable() NN_NOEXCEPT
{
    return g_IsCurrentApplicationAvailable;
}
// 設定クリア
void ResetCurrentApplication() NN_NOEXCEPT
{
    g_IsCurrentApplicationAvailable = false;
    g_CurrentApplication.value = 0;
}

//----------------------------------------------------------------
// アンマウント処理
//
void UnmountCacheStorage() NN_NOEXCEPT
{
    if ( g_IsMountedCacheStorage )
    {
        nn::bcat::UnmountDeliveryCacheStorage(); // 必ず成功

        app::GetSystemConsole().Printf("nn::bcat::UnmountDeliveryCacheStorage()\n");
        g_IsMountedCacheStorage = false;
    }
}

//----------------------------------------------------------------
// マウント処理
//
nn::Result MountCacheStorage() NN_NOEXCEPT
{
    // アプリ選択されていないなら戻る
    if ( ! IsCurrentApplicationAvailable() )
    {
        NN_RESULT_SUCCESS;
    }

    // 現在マウント状態
    if ( g_IsMountedCacheStorage )
    {
        // アンマウント
        UnmountCacheStorage();
    }

    // マウントする
#if defined( APP_BCAT_TEST_APP )
    app::GetSystemConsole().Printf("nn::bcat::MountDeliveryCacheStorage() : " );
    nn::Result result = nn::bcat::MountDeliveryCacheStorage();
#elif defined( APP_BCAT_SYSTEM_DEBUG_TOOL )
    app::GetSystemConsole().Printf("nn::bcat::MountDeliveryCacheStorage( %llx ) : ", GetCurrentApplication().value );
    nn::Result result = nn::bcat::MountDeliveryCacheStorage( GetCurrentApplication() );
#else
    Error // エラー
#endif

    app::PrintErrorCode( result );
    if ( result.IsSuccess() )
    {
        g_IsMountedCacheStorage = true;
    }
    return result;
}

//----------------------------------------------------------------
// 時間の取得
//
bool GetCurrentTime( int* pYear, int* pMonth, int* pDay, int* pHour, int* pMinute, int* pSecond ) NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    nn::Result result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&posixTime);
    if ( result.IsFailure() )
    {
        return false;
    }

    nn::time::CalendarTime calendarTime = {};
    nn::time::CalendarAdditionalInfo additionalInfo = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &calendarTime, &additionalInfo, posixTime ) );

    if ( pYear )
    {
        *pYear = calendarTime.year;
    }
    if ( pMonth )
    {
        *pMonth = calendarTime.month;
    }
    if ( pDay )
    {
        *pDay = calendarTime.day;
    }
    if ( pHour )
    {
        *pHour = calendarTime.hour;
    }
    if ( pMinute )
    {
        *pMinute = calendarTime.minute;
    }
    if ( pSecond )
    {
        *pSecond = calendarTime.second;
    }
    return true;
}

//----------------------------------------------------------------
// 時間を取得しコンソールにプリントする
//
void GetCurrentTime() NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    nn::Result result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&posixTime);
    if ( result.IsFailure() )
    {

        char buf[64];
        nn::util::SNPrintf( buf, sizeof(buf), "Time: N/A (%08x, %03d-%04d)\n",
                            result.GetInnerValueForDebug(),
                            result.GetModule(), result.GetDescription());
        GetScreenConsole().PrintfEx( 35, 0, app::ConsoleColor_Red, buf );
        return;
    }

    nn::time::CalendarTime calendarTime = {};
    nn::time::CalendarAdditionalInfo additionalInfo = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &calendarTime, &additionalInfo, posixTime ) );

    char buf[64];
    nn::util::SNPrintf( buf, sizeof(buf), "%04d/%02d/%02d %02d:%02d:%02d %s",
                        calendarTime.year, calendarTime.month, calendarTime.day,
                        calendarTime.hour, calendarTime.minute, calendarTime.second,
                        additionalInfo.timeZone.standardTimeName );

    GetScreenConsole().PrintfEx( 35, 0, app::ConsoleColor_Cyan, buf );
}

//----------------------------------------------------------------
// サービスディスカバリの環境識別子を取得しコンソールにプリントする
//
void GetCurrentEnvironmentIdentifier() NN_NOEXCEPT
{
    static bool s_IsPrintEnvironmentIdentifier = false;
    if ( ! s_IsPrintEnvironmentIdentifier )
    {
        s_IsPrintEnvironmentIdentifier = true;

        nn::nsd::EnvironmentIdentifier env; // (最大8文字)
        GetEnvironmentIdentifier( &env );

        GetScreenConsole().PrintfEx( 28, 0, app::ConsoleColor_BrightCyan, "Env:[%s]", env.value );
    }
}

//================================================================================
//----------------------------------------------------------------
// ベース表示描画
//
void DrawBaseFrame( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrame( Position_BaseFrame, DrawColorSet_BaseFrame );

    // アプリケーションID
    app::SetGlvColor( app::ColorSet_White );
    glv::draw::text( "ApplicationID:", 10 + 31 * 24, 1 * 28, 24 );
    if ( IsCurrentApplicationAvailable() )
    {
        app::SetGlvColor( app::ColorSet_Yellow );
        char buf[20];
        nn::util::SNPrintf( buf, sizeof(buf), "0x%016llx", GetCurrentApplication().value );
        glv::draw::text( buf, 10 + 39 * 24, 1 * 28, 24 );
    }
    else
    {
        app::SetGlvColor( app::ColorSet_Red );
        glv::draw::text( "N/A", 10 + 39 * 24, 1 * 28, 24 );
    }
}

//----------------------------------------------------------------
// ベース表示の毎フレームタスク
//
void ExecBaseDisplay( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 時刻取得
    GetCurrentTime();

    // ネットワークの接続状態チェック
    CheckNetworkConnected();

    // 環境識別子取得
    GetCurrentEnvironmentIdentifier();

    // 状態
    app::FixedProportionalConsole<char>& c = GetScreenConsole();
    c.PrintfEx( 1,1, app::ConsoleColor_White, "Network:" );
    if ( IsConnected() )
    {
        c.PrintfEx( 6,1, app::ConsoleColor_Green, "OK" );
    }
    else
    {
        c.PrintfEx( 6,1, app::ConsoleColor_Red, "N/A" );
    }

    c.PrintfEx( 9,1, app::ConsoleColor_White, "Mount:" );
    if ( g_IsMountedCacheStorage )
    {
        c.PrintfEx( 13,1, app::ConsoleColor_Green, "OK" );
    }
    else
    {
        c.PrintfEx( 13,1, app::ConsoleColor_Red, "N/A" );
    }

    // L/R ボタンによってシステムコンソール切替
    app::Pad pad( events );
    SwitchSystemConsoleStatus( pad.IsButtonDownR(), pad.IsButtonDownL() );
}

//================================================================================
//----------------------------------------------------------------
// ベース情報表示のセットアップ
//
void SetUpBaseDisplay() NN_NOEXCEPT
{
    static bool isConsoleInitialized = false;
    if ( ! isConsoleInitialized )
    {
        // コンソール初期化
        app::InitializeConsole();

        // 常駐タスク登録
        app::sequence::AddPreTaskCallback( ExecBaseDisplay, nullptr, 100 );
        app::sequence::AddDrawCallback( DrawBaseFrame, nullptr, app::DrawPriority_BaseFrame );

        app::sequence::EnableDraw();
        isConsoleInitialized = true;
    }
}

} // namespace app
