﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_SaveAllFile.h"
#include "BcatTestApp_FileSystem.h"
#include <nn/bcat/bcat_DeliveryCacheDirectory.h>
#include <nn/bcat/bcat_DeliveryCacheFile.h>

namespace app
{
void ExecSaveAllFile_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecSaveAllFile           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecSaveAllFile_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawSaveAllFile( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecSaveAllFileGroup = {
    ExecSaveAllFile_Initialize,
    ExecSaveAllFile,
    ExecSaveAllFile_Finalize,
    nullptr,

    DrawSaveAllFile,
    nullptr,
    DrawPriority_SaveAllFile,
    0
};

namespace
{
    // 処理経過コンソール
    const int ProgressConsoleWidth = 100;
    const int ProgressConsoleHeight = 500;
    const int ProgressConsoleViewHeight = 18;
    const int ProgressConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE_CHAR16( ProgressConsoleWidth, ProgressConsoleHeight );
    char* g_pProgressConsoleBuffer;
    app::ScrollConsole<char16_t> g_ProgressConsole;

    // コンソールの表示オフセット
    const int ConsoleOffsetX = 10;
    const int ConsoleOffsetY = 10;

    // 処理スレッド
    nn::os::ThreadType g_Thread;
    static const size_t ThreadStackSize = 32 * 1024;
    char* g_AllocatedStack;
    void* g_pStack;

    // ディレクトリやファイル情報読み込みバッファ
    nn::bcat::DirectoryName* g_pDirectoryName;
    nn::bcat::DeliveryCacheDirectoryEntry* g_pFileEntry;

    // 処理数
    int g_CopiedFileNum;       // 処理したファイル数
    int g_SizeSkippedFileNum;  // サイズが大きいのでスキップしたファイル数
    int g_HashSkippedFileNum;  // ハッシュが等しいのでスキップしたファイル数
    int g_ErrorFileNum;        // ファイルセーブのエラーの発生数
    int g_CreatedDirectoryNum; // ディレクトリの作成数
    int g_ErrorDirectoryNum;   // ディレクトリ作成のエラー発生数

    // セーブターゲット(SaveData か SDカードか)
    app::SaveAllFileTarget g_SaveTarget;
    // セーブタイプ(上書きか差分か)
    app::SaveAllFileType g_SaveType;
    // 保存ディレクトリ名(SDカード上)
    char* g_pSdDirectoryName;

    // 処理の終了フラグ
    bool g_IsFinished;

    // スレッド処理開始フラグ
    bool g_IsThreadStarted;

    bool g_IsHelpChanged;

    // キーリピート
    int g_RepeatCountUp;
    int g_RepeatCountDown;
    int g_RepeatCountLeft;
    int g_RepeatCountRight;

    char DummyDirectoryName[] = "DummyDirectory";
}

namespace
{
//----------------------------------------------------------------
// プロポーショナルフォントのコンソールプリンタ
//
void DrawPropConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    const float FontSize = 24.0F;
    const float GridSizeX = 16.0F;
    const float GridSizeY = 28.0F;
    DrawProportionalConsoleGeneric16( Position_SaveAllFile.l + ConsoleOffsetX,
                                      Position_SaveAllFile.t + ConsoleOffsetY,
                                      FontSize, GridSizeX, GridSizeY,
                                      x, y, string, attr, stringLen, arg );
}
} //namespace

//----------------------------------------------------------------
// セーブファイル用描画コールバック
//
void DrawSaveAllFile( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrameRectangle( Position_SaveAllFile, app::ColorSet_DarkBlue, app::ColorSet_Blue );

    g_ProgressConsole.Display();

    // スクロールバー
    int LineNum;
    int ViewTop;
    int ViewLines;
    g_ProgressConsole.GetInfo( &LineNum, &ViewTop, &ViewLines );
    app::DrawScrollBar( LineNum, ProgressConsoleViewHeight, ViewTop, ScrollBarDefaultViewWidth, Position_SaveAllFileScrollBar );
}


//----------------------------------------------------------------
// データ配信キャッシュの各ディレクトリ処理
//
void ProcessEachDirectory( nn::bcat::DirectoryName* pDirName ) NN_NOEXCEPT
{
    nn::bcat::DeliveryCacheDirectory dir;
    dir.Open( *pDirName );
    NN_UTIL_SCOPE_EXIT
    {
        dir.Close();
    };

    int outCount;
    nn::Result result = dir.Read( &outCount, g_pFileEntry, nn::bcat::DeliveryCacheFileCountMaxPerDirectory );
    if ( result.IsFailure() )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_Red, u"ファイル一覧の取得に失敗しました\n" );
        return;
    }

    if ( g_SaveTarget == app::SaveAllFileTarget_SdCard )
    {
        bool isDirectoryCreated = false;
        app::SetRootDirectoryName( g_pSdDirectoryName );
        SyncFileToSdCard( pDirName, nullptr, false, &isDirectoryCreated ); // ファイルを nullptr にするとディレクトリを同期

        if ( isDirectoryCreated )
        {
            g_CreatedDirectoryNum ++;
        }
    }

    // ディレクトリ内の各ファイル処理
    for( int i=0; i<outCount; i++ )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_White, u"ファイル: " );
        PrintEntryToConsole( pDirName, &g_pFileEntry[i], app::ConsoleColor_White, true );

        bool isHashCompare = (g_SaveType == SaveAllFileType_Difference );
        bool isDirectoryCreated = false;

        if ( g_SaveTarget == app::SaveAllFileTarget_SdCard )
        {
            app::SetRootDirectoryName( g_pSdDirectoryName );
        }

        SyncFileResult sfResult = ( g_SaveTarget == app::SaveAllFileTarget_SaveData )?
            SyncFileToSaveData( pDirName, &g_pFileEntry[i], isHashCompare, &isDirectoryCreated ) :
            SyncFileToSdCard( pDirName, &g_pFileEntry[i], isHashCompare, &isDirectoryCreated );

        if ( isDirectoryCreated )
        {
            g_CreatedDirectoryNum ++;
        }

        switch( sfResult )
        {
            case SyncFileResult_Copied:
                g_CopiedFileNum ++;
                break;
            case SyncFileResult_SizeSkipped:
                g_SizeSkippedFileNum ++;
                break;
            case SyncFileResult_HashSkipped:
                g_HashSkippedFileNum ++;
                break;
            case SyncFileResult_ErrorOccurred:
                {
                    SyncFileError r = app::GetLastSyncFileError();
                    if ( r == SyncFileError_SyncSaveDataDirectory )
                    {
                        g_ErrorDirectoryNum ++;
                    }
                    else
                    {
                        g_ErrorFileNum ++;
                    }
                }
                break;
            default:
                break;
        }
    }
}

//----------------------------------------------------------------
// セーブファイル保存の実処理スレッドエントリ関数
//
void SaveAllFileThreadFunc( void* arg ) NN_NOEXCEPT
{
    // コンソールセット
    app::SetConsoleForProcessFileSystem( &g_ProgressConsole );

    // データ配信キャッシュからディレクトリ情報取得
    int outCount;
    nn::Result result = nn::bcat::EnumerateDeliveryCacheDirectory( &outCount, g_pDirectoryName, nn::bcat::DeliveryCacheDirectoryCountMax );
    app::PrintErrorCode( result );

    if ( g_SaveTarget == app::SaveAllFileTarget_SdCard )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_Cyan, u"SD カードに保存します\n" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_Cyan, u"フォルダ：" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_Cyan, app::ConvertToChar16_t( "%s\n", g_pSdDirectoryName ) );
    }
    else
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_Cyan, u"セーブデータに保存します\n" );
    }

    // 各ディレクトリ処理
    for( int i = 0; i<outCount; i++ )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_White, u"ディレクトリ: " );
        PrintDirectoryToConsole( &g_pDirectoryName[i], app::ConsoleColor_White, true, true );
        ProcessEachDirectory( &g_pDirectoryName[i] );
    }

    g_ProgressConsole.Printf( u"すべてのデータの処理が終わりました。\n" );
    g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightCyan, u"保存したファイル :" );
    g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightCyan, app::ConvertToChar16_t( "%d\n", g_CopiedFileNum ) );
    if ( g_SizeSkippedFileNum > 0 )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightGreen, u"サイズが大きくスキップしたファイル :" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightGreen, app::ConvertToChar16_t( "%d\n", g_SizeSkippedFileNum ) );
    }
    if ( g_HashSkippedFileNum > 0 )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightGreen, u"更新がなくスキップしたファイル :" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightGreen, app::ConvertToChar16_t( "%d\n", g_HashSkippedFileNum ) );
    }
    if ( g_ErrorFileNum > 0 )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightRed, u"保存エラーファイル :" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightRed, app::ConvertToChar16_t( "%d\n", g_ErrorFileNum ) );
    }
    if ( g_CreatedDirectoryNum > 0 )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightCyan, u"ディレクトリ作成 :" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightCyan, app::ConvertToChar16_t( "%d\n", g_CreatedDirectoryNum ) );
    }
    if ( g_ErrorDirectoryNum > 0 )
    {
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightRed, u"ディレクトリ作成エラー :" );
        g_ProgressConsole.PrintfEx( app::ConsoleColor_BrightRed, app::ConvertToChar16_t( "%d\n", g_ErrorDirectoryNum ) );
    }

    g_IsFinished = true;
}

//----------------------------------------------------------------
// セーブファイル保存コンソールスクロールのためのカーソルキー処理
//
bool CheckSaveAllFileCursor( app::Pad &pad ) NN_NOEXCEPT
{
    int LineNum;
    int ViewTop;
    int ViewLines;
    g_ProgressConsole.GetInfo( &LineNum, &ViewTop, &ViewLines );

    int prev = ViewTop;

    // 上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        g_RepeatCountUp = pad.IsButtonRepeatUp()? (g_RepeatCountUp + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountUp, ProgressConsoleHeight ), ProgressConsoleHeight / 20 );
        g_ProgressConsole.Scroll( -d );
    }
    // 下
    else if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        g_RepeatCountDown = pad.IsButtonRepeatDown()? (g_RepeatCountDown + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountDown, ProgressConsoleHeight ), ProgressConsoleHeight / 20 );
        g_ProgressConsole.Scroll( d );
    }
    // 左
    else if ( pad.IsButtonDownLeft() || pad.IsButtonRepeatLeft() )
    {
        g_RepeatCountLeft = pad.IsButtonRepeatLeft()? (g_RepeatCountLeft + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountLeft, ProgressConsoleHeight ) * ProgressConsoleViewHeight, ProgressConsoleHeight / 20 );
        g_ProgressConsole.Scroll( -d );
    }
    // 右
    else if ( pad.IsButtonDownRight() || pad.IsButtonRepeatRight() )
    {
        g_RepeatCountRight = pad.IsButtonRepeatRight()? (g_RepeatCountRight + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountRight, ProgressConsoleHeight ) * ProgressConsoleViewHeight, ProgressConsoleHeight / 20 );
        g_ProgressConsole.Scroll( d );
    }

    g_ProgressConsole.GetInfo( &LineNum, &ViewTop, &ViewLines );
    return ( prev != ViewTop );
}

//----------------------------------------------------------------
// セーブファイル保存のキー説明
// 処理前から処理中
void PrintSaveAllFileHelp_ForProcessing() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @2[B]@7...中断" );
}

//----------------------------------------------------------------
// セーブファイル保存のキー説明
// 処理終了後
void PrintSaveAllFileHelp_ForFinished() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @4[A]@7...終了  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
// セーブファイル保存の実行コールバック(開始処理)
//
void ExecSaveAllFile_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // パラメータ受け取り
    SaveAllFileParam* p = reinterpret_cast<SaveAllFileParam*>(arg);
    g_SaveType = p->saveType;
    g_SaveTarget = p->saveTarget;
    g_pSdDirectoryName = p->pDirectoryName;
    if ( g_pSdDirectoryName == nullptr || g_pSdDirectoryName[0] == '\0' )
    {
        g_pSdDirectoryName = DummyDirectoryName;
    }

    // コンソール作成
    g_pProgressConsoleBuffer = new char[ ProgressConsoleBufferSize ];
    g_ProgressConsole.SetBuffer( g_pProgressConsoleBuffer, ProgressConsoleBufferSize, ProgressConsoleWidth, ProgressConsoleHeight );
    g_ProgressConsole.SetPrinter( DrawPropConsoleString );
    g_ProgressConsole.SetViewHeight( ProgressConsoleViewHeight );

    // 作業バッファ確保
    g_pDirectoryName = new nn::bcat::DirectoryName[ nn::bcat::DeliveryCacheDirectoryCountMax ];
    g_pFileEntry = new nn::bcat::DeliveryCacheDirectoryEntry[ nn::bcat::DeliveryCacheFileCountMaxPerDirectory ];

    g_IsFinished = false;
    g_IsThreadStarted = false;

    // 処理スレッド用スタック
    g_AllocatedStack = new char[ ThreadStackSize + nn::os::ThreadStackAlignment ];
    g_pStack = reinterpret_cast<void*>( (reinterpret_cast<uintptr_t>(g_AllocatedStack) + (nn::os::ThreadStackAlignment - 1)) & ~(nn::os::ThreadStackAlignment - 1) );

    // キー説明
    PrintSaveAllFileHelp_ForProcessing();

    // ユーザ選択
    if ( g_SaveTarget == app::SaveAllFileTarget_SaveData && ! IsUserValid() )
    {
        SelectUser();
        if ( ! IsUserValid() )
        {
            app::SetInvalidUserDialog( true );
            return;
        }
    }

    g_IsHelpChanged = false;
    g_CopiedFileNum = 0;
    g_SizeSkippedFileNum = 0;
    g_HashSkippedFileNum = 0;
    g_ErrorFileNum = 0;
    g_CreatedDirectoryNum = 0;
    g_ErrorDirectoryNum = 0;

    g_RepeatCountUp = 0;
    g_RepeatCountDown = 0;
    g_RepeatCountLeft = 0;
    g_RepeatCountRight = 0;

    // 処理スレッドを開始
    g_IsThreadStarted = true;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_Thread, SaveAllFileThreadFunc, nullptr, g_pStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
    nn::os::SetThreadNamePointer( &g_Thread, "BcatTest_SaveAllFile" );
    nn::os::StartThread( &g_Thread );
}
//----------------------------------------------------------------
// セーブファイル保存の実行コールバック(終了処理)
//
void ExecSaveAllFile_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( g_IsThreadStarted )
    {
        nn::os::WaitThread( &g_Thread );
        nn::os::DestroyThread( &g_Thread );
    }

    // 各領域を破棄
    delete[] g_AllocatedStack;
    delete[] g_pFileEntry;
    delete[] g_pDirectoryName;
    delete[] g_pProgressConsoleBuffer;
}
//----------------------------------------------------------------
// セーブファイル保存の実行コールバック
//
void ExecSaveAllFile( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // B ボタンで中断 (まだ途中実装)
    if ( pad.IsButtonDownB() )
    {
        app::sequence::Return();
    }

    // 完了後は A ボタンでも戻れる
    if ( g_IsFinished )
    {
        if ( ! g_IsHelpChanged )
        {
            g_ProgressConsole.PrintfEx( app::ConsoleColor_Cyan, u"A ボタンを押してください\n" );

            // キー説明の変更
            PrintSaveAllFileHelp_ForFinished();
            g_IsHelpChanged = true;
        }
        if( pad.IsButtonDownA() )
        {
            app::sequence::Return();
        }
    }

    CheckSaveAllFileCursor( pad );
}

} //namespace app
