﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_TopMenu.h"
#include "BcatTestApp_CacheProgress.h"

namespace app
{
void ExecOtherMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecOtherMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecOtherMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawOtherMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecOtherMenuGroup = {
    ExecOtherMenu_Initialize,
    ExecOtherMenu,
    ExecOtherMenu_Finalize,
    nullptr,

    DrawOtherMenu,
    nullptr,
    DrawPriority_OtherMenu,
    0
};

//----------------------------------------------------------------
namespace
{
    app::Menu g_Menu;
    enum
    {
        MenuIndex_RemoveUser = 0,
        MenuIndex_RequestSyncDeliveryCache,
        MenuIndex_RequestSyncDeliveryCacheDir1,
        MenuIndex_RequestSyncDeliveryCacheDir2,
        MenuIndex_RequestSyncDeliveryCacheDir3,
        MenuIndex_CancelSyncDeliveryCacheRequest,
        MenuIndex_Back,
    };
    int g_InitialMenuIndex = MenuIndex_Back;
    const int MenuMax = MenuIndex_Back + 1;
} // namespace

//----------------------------------------------------------------
// その他メニュー用描画コールバック
//
void DrawOtherMenu( void* arg ) NN_NOEXCEPT
{
}

//----------------------------------------------------------------
// その他メニューのメニュー文字列描画
//
namespace
{
void DrawOtherMenuItems() NN_NOEXCEPT
{
    app::FixedProportionalConsole<char16_t>* p = g_Menu.GetConsole16();
    if ( p )
    {
        app::SetGlvColor( app::ColorSet_White );
        p->PrintfEx( 1, 0,
                     u"-- その他 --" );
        p->PrintfEx( 2, 2,
                     app::IsUserValid() ? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ユーザの選択を解除する" );
        p->PrintfEx( 2, 3,
                     ( ! app::IsMountedCacheStorage() && app::IsConnected() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュの同期を要求する" );
        p->PrintfEx( 2, 4,
                     ( ! app::IsMountedCacheStorage() && app::IsConnected() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュの同期を要求する (dir1)" );
        p->PrintfEx( 2, 5,
                     ( ! app::IsMountedCacheStorage() && app::IsConnected() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュの同期を要求する (dir2)" );
        p->PrintfEx( 2, 6,
                     ( ! app::IsMountedCacheStorage() && app::IsConnected() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュの同期を要求する (dir3)" );
        p->PrintfEx( 2, 7,
                     ( ! app::IsMountedCacheStorage() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュの同期要求をキャンセルする" );
        p->PrintfEx( 2, 8,
                     app::ConsoleColor_White,
                     u"戻る" );
    }
}

//----------------------------------------------------------------
// その他メニューのキー説明
void PrintOtherMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
// ユーザ選択の解除
//
void RemoveUser() NN_NOEXCEPT
{
    if ( ! app::IsUserValid() )
    {
        return;
    }

    app::ForgetUser();

    const char16_t* pMessage[] = { u"ユーザ選択を解除しました。" };
    app::SetNoticeDialog( pMessage, 1 );
}
//----------------------------------------------------------------
// データ配信キャッシュ即時同期要求
//
void RequestSync() NN_NOEXCEPT
{
    if ( app::IsMountedCacheStorage() || ! app::IsConnected() )
    {
        return;
    }

    nn::bcat::DeliveryCacheProgress* pProgress = new nn::bcat::DeliveryCacheProgress;
    NN_ASSERT_NOT_NULL( pProgress );

    // データ配信キャッシュ同期の要求
    app::GetSystemConsole().Printf( "nn::bcat::RequestSyncDeliveryCache() :" );
    nn::Result result = nn::bcat::RequestSyncDeliveryCache( pProgress );
    app::PrintErrorCode( result );

    if ( result.IsSuccess() )
    {
        const char16_t* pMessage[] = { u"同期の要求を行いました" };
        app::SetNoticeDialog( pMessage, 1 );
        app::RegisterProgress( pProgress );
    }
    else
    {
        app::SetErrorDialog( u"同期の要求に失敗しました", result );
        delete pProgress;
    }
}
//----------------------------------------------------------------
// データ配信キャッシュ即時同期要求（ディレクトリ指定）
//
void RequestSync(const char* dirName) NN_NOEXCEPT
{
    if ( app::IsMountedCacheStorage() || ! app::IsConnected() )
    {
        return;
    }

    nn::bcat::DeliveryCacheProgress* pProgress = new nn::bcat::DeliveryCacheProgress;
    NN_ASSERT_NOT_NULL( pProgress );

    // データ配信キャッシュ同期の要求
    app::GetSystemConsole().Printf( "nn::bcat::RequestSyncDeliveryCache() :" );
    nn::Result result = nn::bcat::RequestSyncDeliveryCache( pProgress, dirName );
    app::PrintErrorCode( result );

    if ( result.IsSuccess() )
    {
        const char16_t* pMessage[] = { u"同期の要求を行いました" };
        app::SetNoticeDialog( pMessage, 1 );
        app::RegisterProgress( pProgress );
    }
    else
    {
        app::SetErrorDialog( u"同期の要求に失敗しました", result );
        delete pProgress;
    }
}
//----------------------------------------------------------------
// データ配信キャッシュ即時同期要求のキャンセル
//
void CancelSyncRequest() NN_NOEXCEPT
{
    if ( app::IsMountedCacheStorage() )
    {
        return;
    }

    // データ配信キャッシュ即時同期要求のキャンセル
    app::GetSystemConsole().Printf( "nn::bcat::CancelSyncDeliveryCacheRequest()\n" );
    nn::bcat::CancelSyncDeliveryCacheRequest();

    const char16_t* pMessage[] = { u"同期要求をキャンセルました" };
    app::SetNoticeDialog( pMessage, 1 );
}

} // namespace

//----------------------------------------------------------------
// その他メニュー(開始処理)
//
void ExecOtherMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // メニュー作成
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_OtherMenu, 80, 6, 1, 2, 24, 4, DrawPriority_OtherMenu + 1, 0 );
    g_Menu.SetBackColor( app::DrawColorSet_OtherMenuBack, app::DrawColorSet_OtherMenuFrame, app::DrawFrameWidth );
    g_Menu.SetItemParameter( MenuMax, g_InitialMenuIndex );

    DrawOtherMenuItems();

    // キー説明
    PrintOtherMenuHelp();
}
//----------------------------------------------------------------
// その他メニュー(終了処理)
//
void ExecOtherMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}

//----------------------------------------------------------------
// その他メニュー
//
void ExecOtherMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    switch( g_Menu.CheckButtonOk() )
    {
        case MenuIndex_RemoveUser:
            {
                // ユーザ選択の解除
                RemoveUser();
                return;
            }
            break;
        case MenuIndex_RequestSyncDeliveryCache:
            {
                // 即時同期要求
                RequestSync();
                return;
            }
            break;
        case MenuIndex_RequestSyncDeliveryCacheDir1:
            {
                // 即時同期要求（ディレクトリ指定）
                RequestSync("dir1");
                return;
            }
            break;
        case MenuIndex_RequestSyncDeliveryCacheDir2:
            {
                // 即時同期要求（ディレクトリ指定）
                RequestSync("dir2");
                return;
            }
            break;
        case MenuIndex_RequestSyncDeliveryCacheDir3:
            {
                // 即時同期要求（ディレクトリ指定）
                RequestSync("dir3");
                return;
            }
            break;
        case MenuIndex_CancelSyncDeliveryCacheRequest:
            {
                // 即時同期要求のキャンセル
                CancelSyncRequest();
                return;
            }
            break;
        case MenuIndex_Back:
            {
                app::sequence::JumpTo( ExecTopMenuGroup );
                return;
            }
            break;
        default:
            break;
    }

    // B ボタンで戻る
    if ( g_Menu.CheckButtonCancel() )
    {
        app::sequence::JumpTo( ExecTopMenuGroup );
    }

    if ( g_Menu.IsUpdated() )
    {
        DrawOtherMenuItems();
        g_Menu.ClearUpdated();
    }
}
} //namespace app


