﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv.h>

#include "BcatTestApp_Sequence.h"

namespace app
{
namespace
{
//----------------
// 描画コールバックセット
struct DrawCallbackInfo
{
    app::DrawCallback drawCallback;
    void* drawCallbackArg;
    int priority;
    int tag;
    uint64_t count;

    ExecCallbackGroup* pCallbackGroup; // 設定された callbackgroup を覚えておく
};

// 描画コールバックリスト
std::list<DrawCallbackInfo*> g_DrawCallbackInfoList;

// 現在の描画コールバック
DrawCallbackInfo* g_CurrentDrawCallback = nullptr;
bool g_IsDrawAvailable = false;

//----------------
// タスクコールバックセット
std::list<TaskCallbackInfo*> g_PreExecTaskInfoList;
std::list<TaskCallbackInfo*> g_PostExecTaskInfoList;

//----------------
// 実行コールバックセット
struct ExecCallbackInfo
{
    ExecCallbackGroup callbackGroup;

    void* returnArg;
    int sequence;
    uint64_t count;
    bool isInitialized;

    ExecCallbackInfo* prev;
    //ExecCallbackInfo* next;

    ExecCallback fromCallCallback;
    void* fromCallCallbackArg;

    ExecCallbackInfo() NN_NOEXCEPT :
        returnArg( nullptr ),
        sequence( 0 ),
        count( 0LL ),
        isInitialized( false ),
        fromCallCallback( nullptr ),
        fromCallCallbackArg( nullptr )
    {
    }
};

// 実行コールバックリスト先頭
ExecCallbackInfo* g_ExecCallbackTop = nullptr;
ExecCallbackInfo* g_CurrentExecCallback = nullptr;
ExecCallbackInfo* g_NextExecCallback = nullptr;

int g_CurrentSequence;
void* g_CurrentReturnArg;
const glv::HidEvents* g_CurrentHidEvents;

ExecCallbackInfo g_UpdateExecCallbackInfo;
bool g_IsUpdateExecCallback;

int g_DrawSkipCount = 0;
int g_ExecContinueCount = 0;

} //namespace

namespace sequence
{
//================================================================
//----------------------------------------------------------------
// 描画コールバック群の呼び出し
//
void InvokeDrawCallbacks( void* ) NN_NOEXCEPT
{
    if ( ! g_IsDrawAvailable )
    {
        return;
    }

    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        g_CurrentDrawCallback = *iter;
        //NN_LOG("(%d,%x) ", (*iter)->priority, (*iter)->drawCallback );
        ( (*iter)->drawCallback )( (*iter)->drawCallbackArg );
        (*iter)->count ++;
    }
    g_CurrentDrawCallback = nullptr;
    //NN_LOG("\n");
}

void EnableDraw() NN_NOEXCEPT
{
    g_IsDrawAvailable = true;
}

//----------------------------------------------------------------
// 描画コールバック登録
//
void AddDrawCallbackCore( DrawCallback callback, void* arg, int priority, int tag, ExecCallbackGroup* pCallbackGroup ) NN_NOEXCEPT
{
    if ( callback == nullptr )
    {
        return;
    }

    DrawCallbackInfo* pNewItem = new DrawCallbackInfo();
    pNewItem->drawCallback = callback;
    pNewItem->drawCallbackArg = arg;
    pNewItem->priority = priority;
    pNewItem->tag = tag;
    pNewItem->count = 0LL;

    // 設定した ExecCallbackGroup を覚えておく
    pNewItem->pCallbackGroup = pCallbackGroup;

    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        if ( (*iter)->priority > priority )
        {
            g_DrawCallbackInfoList.insert( iter, pNewItem );
            return;
        }
    }
    g_DrawCallbackInfoList.push_back( pNewItem );
}

void AddDrawCallback( DrawCallback callback, void* arg, int priority, int tag ) NN_NOEXCEPT
{
    AddDrawCallbackCore( callback, arg, priority, tag, g_CurrentExecCallback ? &(g_CurrentExecCallback->callbackGroup): nullptr );
}

//----------------------------------------------------------------
// 描画コールバック削除
//
void RemoveDrawCallback( DrawCallback callback ) NN_NOEXCEPT
{
    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        if ( (*iter)->drawCallback == callback )
        {
            DrawCallbackInfo* p = *iter;
            g_DrawCallbackInfoList.erase(iter);
            delete p;
            return;
        }
    }
}

//----------------------------------------------------------------
// 描画コールバック削除
//
void RemoveDrawCallbackWithTag( DrawCallback callback, int tag ) NN_NOEXCEPT
{
    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        if ( (*iter)->drawCallback == callback && (*iter)->tag == tag )
        {
            DrawCallbackInfo* p = *iter;
            g_DrawCallbackInfoList.erase(iter);
            delete p;
            return;
        }
    }
}
//----------------------------------------------------------------
// 描画コールバック存在確認
//
bool IsExistDrawCallback( DrawCallback callback ) NN_NOEXCEPT
{
    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        if ( (*iter)->drawCallback == callback )
        {
            return true;
        }
    }
    return false;
}
//----------------------------------------------------------------
// 描画コールバック存在確認
//
bool IsExistDrawCallback( DrawCallback callback, int tag ) NN_NOEXCEPT
{
    for( auto iter = g_DrawCallbackInfoList.begin(); iter != g_DrawCallbackInfoList.end(); iter++ )
    {
        if ( (*iter)->drawCallback == callback && (*iter)->tag == tag )
        {
            return true;
        }
    }
    return false;
}

//----------------------------------------------------------------
// 描画フレーム取得
//
uint64_t GetDrawFrameCount() NN_NOEXCEPT
{
    return g_CurrentDrawCallback? g_CurrentDrawCallback->count: 0LL;
}
//----------------------------------------------------------------
// 描画フレームスキップ設定
void SetDrawFrameSkip( int count ) NN_NOEXCEPT
{
    g_DrawSkipCount = count;
}

//----------------------------------------------------------------
// アクティブか
// (設定した ExecCallback がカレントか)
bool IsDrawCallbackActive() NN_NOEXCEPT
{
    return ( g_CurrentDrawCallback && g_CurrentExecCallback ) ?
        (g_CurrentDrawCallback->pCallbackGroup == &(g_CurrentExecCallback->callbackGroup) ) :
        false;
}

//================================================================
// タスクコールバック群の呼び出し
void InvokeTaskCallbacks( std::list<TaskCallbackInfo*>& taskList, const glv::HidEvents& g ) NN_NOEXCEPT
{
    for( auto iter = taskList.begin(); iter != taskList.end(); iter++ )
    {
        ( (*iter)->callback )( g, (*iter)->callbackArg );
    }
}
//----------------------------------------------------------------
// タスク登録
void AddTaskCallback( std::list<TaskCallbackInfo*>& taskList, ExecCallback callback, void* arg, int priority ) NN_NOEXCEPT
{
    if ( callback == nullptr )
    {
        return;
    }
    TaskCallbackInfo* pNewItem = new TaskCallbackInfo();
    pNewItem->callback = callback;
    pNewItem->callbackArg = arg;
    pNewItem->priority = priority;

    for( auto iter = taskList.begin(); iter != taskList.end(); iter++ )
    {
        if ( (*iter)->priority > priority )
        {
            taskList.insert( iter, pNewItem );
            return;
        }
    }
    taskList.push_back( pNewItem );
}
void AddPreTaskCallback( ExecCallback callback, void* arg, int priority ) NN_NOEXCEPT
{
    AddTaskCallback( g_PreExecTaskInfoList, callback, arg, priority );
}
void AddPostTaskCallback( ExecCallback callback, void* arg, int priority ) NN_NOEXCEPT
{
    AddTaskCallback( g_PostExecTaskInfoList, callback, arg, priority );
}
//----------------------------------------------------------------
// タスク削除
void RemoveTaskCallback( std::list<TaskCallbackInfo*>& taskList, ExecCallback callback ) NN_NOEXCEPT
{
    for( auto iter = taskList.begin(); iter != taskList.end(); iter++ )
    {
        if ( (*iter)->callback == callback )
        {
            TaskCallbackInfo* p = *iter;
            taskList.erase(iter);
            delete p;
            return;
        }
    }
}
void RemovePreTaskCallback( ExecCallback callback ) NN_NOEXCEPT
{
    RemoveTaskCallback( g_PreExecTaskInfoList, callback );
}
void RemovePostTaskCallback( ExecCallback callback ) NN_NOEXCEPT
{
    RemoveTaskCallback( g_PostExecTaskInfoList, callback );
}
//----------------------------------------------------------------
// タスク存在確認
bool IsExistTaskCallback( std::list<TaskCallbackInfo*>& taskList, ExecCallback callback ) NN_NOEXCEPT
{
    for( auto iter = taskList.begin(); iter != taskList.end(); iter++ )
    {
        if ( (*iter)->callback == callback )
        {
            return true;
        }
    }
    return false;
}
bool IsExistPreTaskCallback( ExecCallback callback ) NN_NOEXCEPT
{
    return IsExistTaskCallback( g_PreExecTaskInfoList, callback );
}
bool IsExistPostTaskCallback( ExecCallback callback ) NN_NOEXCEPT
{
    return IsExistTaskCallback( g_PostExecTaskInfoList, callback );
}

//================================================================
//----------------------------------------------------------------
// 実行コールバックリスト先頭
void InvokeExecCallback( const glv::HidEvents& g, void* ) NN_NOEXCEPT
{
     g_CurrentHidEvents = &g;

     InvokeTaskCallbacks( g_PreExecTaskInfoList, g );

    if ( g_CurrentExecCallback )
    {
        g_ExecContinueCount = 1;
        while( --g_ExecContinueCount >= 0 )
        {
            g_CurrentSequence = g_CurrentExecCallback->sequence;
            g_CurrentReturnArg = g_CurrentExecCallback->returnArg;
            g_IsUpdateExecCallback = false;
            g_NextExecCallback = g_CurrentExecCallback;

            ExecCallbackGroup &group = g_CurrentExecCallback->callbackGroup;
            // 初回
            if ( ! g_CurrentExecCallback->isInitialized )
            {
                if ( group.execInitializeCallback )
                {
                    (group.execInitializeCallback)( *g_CurrentHidEvents, group.execCallbackArg );
                }
                g_CurrentExecCallback->isInitialized = true;

                // 描画コールバック設定
                if ( g_IsUpdateExecCallback == false && group.drawAutoCallback )
                {
                    app::sequence::AddDrawCallbackCore( group.drawAutoCallback, group.drawAutoCallbackArg, group.drawAutoCallbackPriority, group.drawAutoCallbackTag, &group );
                }
            }
            if ( g_IsUpdateExecCallback == false && g_CurrentExecCallback->fromCallCallback )
            {
                (g_CurrentExecCallback->fromCallCallback)( *g_CurrentHidEvents, g_CurrentExecCallback->fromCallCallbackArg );
                g_CurrentExecCallback->fromCallCallback = nullptr;
                g_CurrentExecCallback->fromCallCallbackArg = nullptr;
            }

            // コールバック呼び出し
            //NN_LOG(":::call  callback=%x arg=%x seq=%d\n", g_CurrentExecCallback->execCallback, g_CurrentExecCallback->execCallbackArg, g_CurrentExecCallback->sequence);
            if ( g_IsUpdateExecCallback == false && group.execCallback )
            {
                (group.execCallback)( *g_CurrentHidEvents, group.execCallbackArg );
            }

            g_CurrentExecCallback->count ++;

            // シーケンスの更新がある場合
            if ( g_IsUpdateExecCallback )
            {
                g_CurrentExecCallback = g_NextExecCallback;
                if ( g_CurrentExecCallback )
                {
                    g_CurrentExecCallback->callbackGroup = g_UpdateExecCallbackInfo.callbackGroup;

                    g_CurrentExecCallback->returnArg = g_UpdateExecCallbackInfo.returnArg;
                    g_CurrentExecCallback->sequence = g_UpdateExecCallbackInfo.sequence;
                    g_CurrentExecCallback->isInitialized = g_UpdateExecCallbackInfo.isInitialized;
                    g_UpdateExecCallbackInfo.count = g_UpdateExecCallbackInfo.isInitialized;
                }
            }
        }
    }

    InvokeTaskCallbacks( g_PostExecTaskInfoList, g );
}

//----------------------------------------------------------------
// 実行コールバック先頭の設定
void SetEntry( ExecCallbackGroup callbackGroup, int sequence ) NN_NOEXCEPT
{
    if ( g_ExecCallbackTop )
    {
        return;
    }

    ExecCallbackInfo* pNewItem = new ExecCallbackInfo();
    pNewItem->callbackGroup = callbackGroup;

    pNewItem->sequence = sequence;
    pNewItem->returnArg = nullptr;
    pNewItem->prev = nullptr;
    //pNewItem->next = nullptr;
    pNewItem->count = 0LL;

    g_ExecCallbackTop = pNewItem;
    g_CurrentExecCallback = pNewItem;
}

void SetEntry( ExecCallback callback, void* arg, int sequence ) NN_NOEXCEPT
{
    ExecCallbackGroup group =
        {
            nullptr, callback, nullptr, arg,
            nullptr, nullptr, 0, 0
        };
    SetEntry( group, sequence );
}

void JumpTo( ExecCallbackGroup callbackGroup, int sequence ) NN_NOEXCEPT
{
    if ( g_IsUpdateExecCallback )
    {
        return;
    }

    if ( ! g_CurrentExecCallback )
    {
        ExecCallbackInfo* pNewItem = new ExecCallbackInfo();
        g_NextExecCallback = pNewItem;

        pNewItem->prev = nullptr;
        //pNewItem->next = nullptr;

        g_ExecCallbackTop = pNewItem;
    }

    g_IsUpdateExecCallback = true;
    g_UpdateExecCallbackInfo.callbackGroup = callbackGroup;
    g_UpdateExecCallbackInfo.returnArg = nullptr;
    g_UpdateExecCallbackInfo.sequence = sequence;
    g_UpdateExecCallbackInfo.isInitialized = false;
    g_UpdateExecCallbackInfo.count = 0LL;

    // 終了コールバック
    ExecCallbackGroup &group = g_CurrentExecCallback->callbackGroup;
    if ( group.drawAutoCallback )
    {
        RemoveDrawCallbackWithTag( group.drawAutoCallback, group.drawAutoCallbackTag );
    }
    if ( group.execFinalizeCallback )
    {
        (group.execFinalizeCallback)( *g_CurrentHidEvents, group.execCallbackArg );
    }
}

void JumpTo( ExecCallbackGroup callbackGroup, void* arg, int sequence ) NN_NOEXCEPT
{
    ExecCallbackGroup group = callbackGroup;
    group.execCallbackArg = arg;
    JumpTo( group, sequence );
}

void JumpTo( ExecCallback callback, void* arg, int sequence ) NN_NOEXCEPT
{
    ExecCallbackGroup group =
        {
            nullptr, callback, nullptr, arg,
            nullptr, nullptr, 0, 0
        };
    JumpTo( group, sequence );
}

void SlideTo( ExecCallback callback ) NN_NOEXCEPT
{
    g_CurrentExecCallback->callbackGroup.execCallback = callback;
}

void Call( ExecCallbackGroup callbackGroup, int sequence ) NN_NOEXCEPT
{
    if ( g_IsUpdateExecCallback )
    {
        return;
    }

    ExecCallbackInfo* pNewItem = new ExecCallbackInfo();
    g_NextExecCallback = pNewItem;

    if ( g_CurrentExecCallback )
    {
        //g_CurrentExecCallback->next = pNewItem;
        g_CurrentExecCallback->fromCallCallback = nullptr;
        g_CurrentExecCallback->fromCallCallbackArg = nullptr;
    }
    else
    {
        g_ExecCallbackTop = pNewItem;
    }

    pNewItem->prev = g_CurrentExecCallback;
    //pNewItem->next = g_CurrentExecCallback;
    pNewItem->count = 0LL;

    g_IsUpdateExecCallback = true;
    g_UpdateExecCallbackInfo.callbackGroup = callbackGroup;
    g_UpdateExecCallbackInfo.returnArg = nullptr;
    g_UpdateExecCallbackInfo.sequence = sequence;
    g_UpdateExecCallbackInfo.isInitialized = false;
}

void Call( ExecCallbackGroup callbackGroup, void* arg, int sequence ) NN_NOEXCEPT
{
    ExecCallbackGroup group = callbackGroup;
    group.execCallbackArg = arg;
    Call( group, sequence );
}

void Call( ExecCallback callback, void* arg, int sequence ) NN_NOEXCEPT
{
    ExecCallbackGroup group =
        {
            nullptr, callback, nullptr, arg,
            nullptr, nullptr, 0, 0
        };
    Call( group, sequence );
}

void SetFromCall( ExecCallback callback, void* arg ) NN_NOEXCEPT
{
    g_CurrentExecCallback->fromCallCallback = callback;
    g_CurrentExecCallback->fromCallCallbackArg = arg;
}

void Return( void* arg ) NN_NOEXCEPT
{
    if ( g_IsUpdateExecCallback )
    {
        return;
    }

    ExecCallbackInfo* p = g_CurrentExecCallback->prev;

    // 終了コールバック
    ExecCallbackGroup &group = g_CurrentExecCallback->callbackGroup;
    if ( group.drawAutoCallback )
    {
        RemoveDrawCallbackWithTag( group.drawAutoCallback, group.drawAutoCallbackTag );
    }
    if ( group.execFinalizeCallback )
    {
        (group.execFinalizeCallback)( *g_CurrentHidEvents, group.execCallbackArg );
    }

    delete g_CurrentExecCallback;
    if ( p )
    {
        //p->next = nullptr;
    }
    else
    {
        g_ExecCallbackTop = nullptr;
    }

    g_IsUpdateExecCallback = true;
    g_NextExecCallback = p;
    if ( p )
    {
        g_UpdateExecCallbackInfo.callbackGroup = p->callbackGroup;
        g_UpdateExecCallbackInfo.returnArg = arg;
        g_UpdateExecCallbackInfo.sequence = p->sequence;
        g_UpdateExecCallbackInfo.isInitialized = true;
    }
}

//----------------------------------------------------------------
// シーケンス設定
void SetSequence( int sequence ) NN_NOEXCEPT
{
    if ( g_CurrentExecCallback )
    {
        g_CurrentExecCallback->sequence = sequence;
    }
}

int GetSequence() NN_NOEXCEPT
{
    return g_CurrentSequence;
}

void* GetReturnArg() NN_NOEXCEPT
{
    return g_CurrentReturnArg;
}

//----------------------------------------------------------------
uint64_t GetFrameCount() NN_NOEXCEPT
{
    return ( g_CurrentExecCallback )? g_CurrentExecCallback->count: 0LL;
}

//----------------------------------------------------------------
// 実行フレーム続行設定
void SetFrameContinue( int count ) NN_NOEXCEPT
{
    g_CurrentExecCallback += count;
}

} //namespace sequence
} //namespace app

