﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace app {

    enum
    {
        ColorSet_Black       = 0x000000ff,
        ColorSet_Blue        = 0x0000ffff,
        ColorSet_Red         = 0xff0000ff,
        ColorSet_Magenta     = 0xff00ffff,
        ColorSet_Green       = 0x00ff00ff,
        ColorSet_Cyan        = 0x00ffffff,
        ColorSet_Yellow      = 0xffff00ff,
        ColorSet_White       = 0xffffffff,
        ColorSet_Glay        = 0x404040ff,
        ColorSet_DarkBlue    = 0x000080ff,
        ColorSet_DarkRed     = 0x800000ff,
        ColorSet_DarkMagenta = 0x800080ff,
        ColorSet_DarkGreen   = 0x008000ff,
        ColorSet_DarkCyan    = 0x008080ff,
        ColorSet_DarkYellow  = 0x808000ff,
        ColorSet_DarkWhite   = 0x808080ff,

        ColorSet_TransBlack       = 0x00000080,
        ColorSet_TransBlue        = 0x0000ff80,
        ColorSet_TrasnRed         = 0xff000080,
        ColorSet_TrasnMagenta     = 0xff00ff80,
        ColorSet_TransGreen       = 0x00ff0080,
        ColorSet_TransCyan        = 0x00ffff80,
        ColorSet_TransYellow      = 0xffff0080,
        ColorSet_TransWhite       = 0xffffff80,
        ColorSet_TransGlay        = 0x40404080,
        ColorSet_TransDarkBlue    = 0x00008080,
        ColorSet_TransDarkRed     = 0x80000080,
        ColorSet_TransDarkMagenta = 0x80008080,
        ColorSet_TransDarkGreen   = 0x00800080,
        ColorSet_TransDarkCyan    = 0x00808080,
        ColorSet_TransDarkYellow  = 0x80800080,
        ColorSet_TransDarkWhite   = 0x80808080,
    };

    // 汎用のスクロールバーの表示色と背景色
    const int DrawColorSet_ScrollBarView = app::ColorSet_Yellow;
    const int DrawColorSet_ScrollBarBack = app::ColorSet_DarkGreen;

struct ColorSet
{
    float valueR;
    float valueG;
    float valueB;
    float valueA;

    ColorSet() NN_NOEXCEPT
    {
        SetRgba( 0.F, 0.F, 0.F, 0.F );
    }
    explicit ColorSet( int rgba ) NN_NOEXCEPT
    {
        SetIntRgba( (rgba >> 24) & 0xff, (rgba >> 16) & 0xff, (rgba >> 8) & 0xff, rgba & 0xff );
    }
    ColorSet( int r, int g, int b ) NN_NOEXCEPT
    {
        SetIntRgb( r, g, b );
    }
    ColorSet( int r, int g, int b, int a ) NN_NOEXCEPT
    {
        SetIntRgba( r, g, b, a );
    }
    ColorSet( float r, float g, float b ) NN_NOEXCEPT
    {
        SetRgb( r, g, b );
    }
    ColorSet( float r, float g, float b, float a ) NN_NOEXCEPT
    {
        SetRgba( r, g, b, a );
    }

    int GetIntR() NN_NOEXCEPT
    {
        return static_cast<int>( valueR * 255 );
    }
    int GetIntG() NN_NOEXCEPT
    {
        return static_cast<int>( valueG * 255 );
    }
    int GetIntB() NN_NOEXCEPT
    {
        return static_cast<int>( valueB * 255 );
    }
    int GetIntA() NN_NOEXCEPT
    {
        return static_cast<int>( valueA * 255 );
    }
    float GetR() NN_NOEXCEPT
    {
        return valueR;
    }
    float GetG() NN_NOEXCEPT
    {
        return valueG;
    }
    float GetB() NN_NOEXCEPT
    {
        return valueB;
    }
    float GetA() NN_NOEXCEPT
    {
        return valueA;
    }
    void SetIntRgb( int r, int g, int b ) NN_NOEXCEPT
    {
        SetIntRgba( r, g, b, 255 );
    }
    void SetIntRgba( int r, int g, int b, int a ) NN_NOEXCEPT
    {
        NN_ASSERT( 0 <= r && r <= 255 &&
                   0 <= g && g <= 255 &&
                   0 <= b && b <= 255 &&
                   0 <= a && a <= 255 );

        valueR = r / 255.F;
        valueG = g / 255.F;
        valueB = b / 255.F;
        valueA = a / 255.F;
    }
    void SetRgb( float r, float g, float b ) NN_NOEXCEPT
    {
        SetRgba( r, g, b, 1.F );
    }
    void SetRgba( float r, float g, float b, float a ) NN_NOEXCEPT
    {
        NN_ASSERT( 0.F <= r && r <= 1.F &&
                   0.F <= g && g <= 1.F &&
                   0.F <= b && b <= 1.F &&
                   0.F <= a && a <= 1.F );

        valueR = r;
        valueG = g;
        valueB = b;
        valueA = a;
    }
};

struct RectanglePosition
{
    int l;
    int t;
    int r;
    int b;

    RectanglePosition()
    {
    }

    RectanglePosition( int posL, int posT, int posR, int posB ) NN_NOEXCEPT :
        l( posL ),
        t( posT ),
        r( posR ),
        b( posB )
    {
    }

    RectanglePosition( const RectanglePosition& pos ) NN_NOEXCEPT :
        l( pos.l ),
        t( pos.t ),
        r( pos.r ),
        b( pos.b )
    {
    }

    int GetLeft() NN_NOEXCEPT
    {
        return l;
    }
    int GetTop() NN_NOEXCEPT
    {
        return t;
    }
    int GetRight() NN_NOEXCEPT
    {
        return r;
    }
    int GetBottom() NN_NOEXCEPT
    {
        return b;
    }
    void SetLeft( int posL ) NN_NOEXCEPT
    {
        l = posL;
    }
    void SetTop( int posT ) NN_NOEXCEPT
    {
        t = posT;
    }
    void SetRight( int posR ) NN_NOEXCEPT
    {
        r = posR;
    }
    void SetBottom( int posB ) NN_NOEXCEPT
    {
        b = posB;
    }
};

static const float ScrollBarDefaultViewWidth = 30.0F;

void SetGlvColor( ColorSet cset ) NN_NOEXCEPT;
void SetGlvColor( int c ) NN_NOEXCEPT;

void DrawFrame( int x1, int y1, int x2, int y2, ColorSet cset, float frameWidth = 1.0f ) NN_NOEXCEPT;
void DrawFrame( int x1, int y1, int x2, int y2, int c, float frameWidth = 1.0f  ) NN_NOEXCEPT;
void DrawFrame( RectanglePosition pos, ColorSet cset, float frameWidth = 1.0f  ) NN_NOEXCEPT;
void DrawFrame( RectanglePosition pos, int c, float frameWidth = 1.0f  ) NN_NOEXCEPT;

void DrawRectangle( int x1, int y1, int x2, int y2, ColorSet cset ) NN_NOEXCEPT;
void DrawRectangle( int x1, int y1, int x2, int y2, int c ) NN_NOEXCEPT;
void DrawRectangle( RectanglePosition pos, ColorSet cset ) NN_NOEXCEPT;
void DrawRectangle( RectanglePosition pos, int c ) NN_NOEXCEPT;

void DrawFrameRectangle( int x1, int y1, int x2, int y2, ColorSet cset1, ColorSet cset2, float frameWidth = 1.0f  ) NN_NOEXCEPT;
void DrawFrameRectangle( int x1, int y1, int x2, int y2, int c1, int c2, float frameWidth = 1.0f  ) NN_NOEXCEPT;
void DrawFrameRectangle( RectanglePosition pos, ColorSet cset1, ColorSet cset2, float frameWidth = 1.0f  ) NN_NOEXCEPT;
void DrawFrameRectangle( RectanglePosition pos, int c1, int c2, float frameWidth = 1.0f  ) NN_NOEXCEPT;

// 汎用スクロールバー
void DrawScrollBar( int64_t wholeSize, int64_t displaySize, int64_t currentPosition,
                    float minWidth, RectanglePosition scrollBarPosition ) NN_NOEXCEPT;

// 文字列の描画サイズ取得
void GetStringBounds( const char* str, int fontSize, float spacing, float* pWidth, float* pHeight ) NN_NOEXCEPT;

// 文字列描画
void SetGlvTextParameter( unsigned fontSize, float lineSpacing = 1.5F, float tabSpaces = 4.0F ) NN_NOEXCEPT;
void DrawGlvText( float x, float y, const char* s ) NN_NOEXCEPT;
void DrawGlvText( float x, float y, const char* s, unsigned fontSize, float lineSpacing, float tabSpaces ) NN_NOEXCEPT;
void DrawGlvText( float x, float y, const char16_t* s ) NN_NOEXCEPT;
void DrawGlvText( float x, float y, const char16_t* s, unsigned fontSize, float lineSpacing, float tabSpaces ) NN_NOEXCEPT;

} //namespace app
