﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nn/util/util_ScopeExit.h>

#include "BcatTestApp_ConsoleCore.h"
#include "BcatTestApp_Utility.h"

namespace app
{
namespace
{
    const size_t PrintBufferSize = 1024;
    static char g_PrintBuffer[ PrintBufferSize ];
} //namespace

float GetConsoleColorR( int attr ) NN_NOEXCEPT
{
    return (attr & 2)?
        (attr & 8? 0.6f: 1.f):
        (attr & 16? 0.4f: 0.f);
}
float GetConsoleColorG( int attr ) NN_NOEXCEPT
{
    return (attr & 4)?
        (attr & 8? 0.6f: 1.f):
        (attr & 16? 0.4f: 0.f);
}
float GetConsoleColorB( int attr ) NN_NOEXCEPT
{
    return (attr & 1)?
        (attr & 8? 0.6f: 1.f):
        (attr & 16? 0.4f: 0.f);
}

template <typename T>
Console<T>::Console( void* pBuffer, int bufferSize, int width, int viewHeight ) NN_NOEXCEPT
    : m_Mutex(true)
{
    m_Printer = nullptr;
    SetBuffer( pBuffer, bufferSize, width, viewHeight );
    m_bgR = m_bgG = m_bgB = 0.f;
}

template <typename T>
void Console<T>::SetBuffer( void* pBuffer, int bufferSize, int width, int viewHeight ) NN_NOEXCEPT
{
    m_pBuffer = reinterpret_cast<char*>( pBuffer );
    m_BufferSize = bufferSize;
    m_Width = width;
    m_ViewHeight = viewHeight;
    m_BufLine = m_BufferSize / ((m_Width + sizeof(T)) * (sizeof(T) + 1));

    Clear();
    m_IsInitialized = true;
}

template <typename T>
void Console<T>::Clear() NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    m_DispHeight = 0;
    m_X = 0;
    m_Y = 0;
    m_TopIndex = 0;
    m_DispIndex = 0;
    m_Attr = app::ConsoleConsole_Default;

    ClearLine(m_Y);
}

template <typename T>
void Console<T>::ClearLine( int y ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    T* pChar = GetPtr(0,y);
    char* pAttr = GetAttrPtr(0,y);

    for( int n=0; n<m_Width; n++ )
    {
        *pChar++ = 0;
        *pAttr++ = 0;
    }
}

template <typename T>
T* Console<T>::GetPtr( int x, int y ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    return reinterpret_cast<T*>( m_pBuffer + (m_Width + sizeof(T)) * (sizeof(T) + 1) * y + x * sizeof(T) );
}

template <typename T>
char* Console<T>::GetAttrPtr( int x, int y ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    return m_pBuffer + (m_Width + sizeof(T)) * ( (sizeof(T) + 1) * y + sizeof(T) ) + x;
}

template <typename T>
void Console<T>::PutChar( int x, int y, T c ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    *GetPtr( x, y ) = c;
}

template <typename T>
void Console<T>::PutAttr( int x, int y, char attr ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    *GetAttrPtr( x, y ) = attr;
}

template <typename T>
void Console<T>::PutChar( T c, char attr ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    PutChar(m_X, m_Y, c);
    PutAttr(m_X, m_Y, attr);
}

template <typename T>
int Console<T>::YNext( int y ) NN_NOEXCEPT
{
    int y0 = y + 1;
    return (y0 < m_BufLine)? y0: 0;
}

template <typename T>
int Console<T>::YIncrement( int& y ) NN_NOEXCEPT
{
    y = YNext( y );
    return y;
}

template <typename T>
int Console<T>::YPrev( int y ) NN_NOEXCEPT
{
    int y0 = y - 1;
    return (y0 < 0)? m_BufLine - 1 : y0;
}

template <typename T>
int Console<T>::YDecrement( int& y ) NN_NOEXCEPT
{
    y = YPrev( y );
    return y;
}

template <typename T>
int Console<T>::YDiff( int y1, int y2 ) NN_NOEXCEPT
{
    return ( y1 >= y2 )? ( y1 - y2 ): ( y1 + (m_BufLine - y2) );
}

template <typename T>
int Console<T>::YSub( int& y, int n ) NN_NOEXCEPT
{
    y -= n;
    while( y < 0 )
    {
        y += m_BufLine;
    }
    return y;
}

template <typename T>
int Console<T>::YAdd( int& y, int n ) NN_NOEXCEPT
{
    y += n;
    while( y > m_BufLine )
    {
        y -= m_BufLine;
    }
    return y;
}

template <typename T>
void Console<T>::SetAttribute( char attr ) NN_NOEXCEPT
{
    m_Attr = attr;
}

template <typename T>
char Console<T>::GetAttribute() NN_NOEXCEPT
{
    return m_Attr;
}

template <typename T>
void Console<T>::ReturnLeft() NN_NOEXCEPT
{
    m_X = 0;
}

template <typename T>
void Console<T>::PutString( const T* s ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    int attr  = m_Attr;
    int attr0 = m_Attr;

    while(*s)
    {
        T c = static_cast<T>(*s++);

        if ( c == '\n' )
        {
            m_X = m_Width;
        }
        else if ( c == '@' )
        {
            T next_c = *s;

            if ( next_c == '@' )
            {
                PutChar('@', attr);
                m_X ++;
                PutChar(0, 0);
                s++;
            }
            if ( '0'<=next_c && next_c<='9' )
            {
                attr = next_c - '0';
            }
            else if ( 'A'<=next_c && next_c<='F' )
            {
                attr = next_c - 'A' + 10;
            }
            else if ( 'a'<=next_c && next_c<='f' )
            {
                attr = next_c - 'a' + 10;
            }
            else if ( next_c == 'r' || next_c == 'R' )
            {
                attr = attr0;
            }
            s++;
            continue;
        }
        else
        {
            PutChar(c, attr);
            m_X ++;
            PutChar(0, 0);
        }

        if ( m_X < m_Width )
        {
            continue;
        }

        m_X = 0;
        YIncrement(m_Y);
        ArrangeY_inPutString();
    }
}
template <>
void Console<char16_t>::PutString( const char16_t* s ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    int attr  = m_Attr;
    int attr0 = m_Attr;

    while(*s)
    {
        char16_t c = static_cast<char16_t>(*s++);

        if ( c == u'\n' )
        {
            m_X = m_Width;
        }
        else if ( c == u'@' )
        {
            char16_t next_c = *s;

            if ( next_c == u'@' )
            {
                PutChar(u'@', attr);
                m_X ++;
                PutChar(0, 0);
                s++;
            }
            if ( u'0'<=next_c && next_c<=u'9' )
            {
                attr = next_c - u'0';
            }
            else if ( u'A'<=next_c && next_c<=u'F' )
            {
                attr = next_c - u'A' + 10;
            }
            else if ( u'a'<=next_c && next_c<=u'f' )
            {
                attr = next_c - u'a' + 10;
            }
            else if ( next_c == u'r' || next_c == u'R' )
            {
                attr = attr0;
            }
            s++;
            continue;
        }
        else
        {
            PutChar(c, attr);
            m_X ++;
            if ( GetType() != ConsoleType_Fixed )
            {
                PutChar(0, 0);
            }
        }

        if ( m_X < m_Width )
        {
            continue;
        }

        m_X = 0;
        YIncrement(m_Y);
        ArrangeY_inPutString();
    }
}

template <typename T>
int Console<T>::Scroll( int ) NN_NOEXCEPT
{
    return 0;
}

template <typename T>
const T* Console<T>::Printf( const T* format, ... ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    va_list args;
    va_start( args, format );

    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    PutString( reinterpret_cast<T*>(g_PrintBuffer) );
    va_end( args );
    return reinterpret_cast<T*>(g_PrintBuffer);
}
template <>
const char16_t* Console<char16_t>::Printf( const char16_t* format, ... ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    PutString( format );
    return format;
}

template <typename T>
void Console<T>::SetPrinter( Printer f, void* arg ) NN_NOEXCEPT
{
    Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Unlock();
    };

    m_Printer = f;
    m_PrinterArg = arg;
}

//================================================================================
// スクロールコンソール
//
template <typename T>
const T* ScrollConsole<T>::PrintfEx( int attr, const T* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    char attr0 = Console<T>::m_Attr;
    Inherited::SetAttribute( attr );

    va_list args;
    va_start( args, format );

    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    Inherited::PutString( g_PrintBuffer );

    Inherited::SetAttribute( attr0 );
    va_end( args );
    return g_PrintBuffer;
}
template <>
const char16_t* ScrollConsole<char16_t>::PrintfEx( int attr, const char16_t* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    char attr0 = Console<char16_t>::m_Attr;
    Inherited::SetAttribute( attr );

    Inherited::PutString( format );

    Inherited::SetAttribute( attr0 );
    return format;
}

template <typename T>
void ScrollConsole<T>::GetInfo( int* pBufferLines, int *pDisplayPosition, int* pDisplayLines ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    *pBufferLines = Inherited::YDiff( Inherited::m_Y, Inherited::m_TopIndex ) + 1;

    int disp = Inherited::YDiff( Inherited::m_Y, Inherited::m_DispIndex ) + 1;
    *pDisplayLines = (disp > Inherited::m_ViewHeight)? Inherited::m_ViewHeight: disp;

    *pDisplayPosition = Inherited::YDiff( Inherited::m_DispIndex, Inherited::m_TopIndex );
}

template <typename T>
void ScrollConsole<T>::ArrangeY_inPutString() NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    // 表示位置の調整
    if ( Inherited::YDiff( Inherited::m_Y, Inherited::m_DispIndex ) >= Inherited::m_ViewHeight )
    {
        Inherited::YIncrement( Inherited::m_DispIndex );
    }

    // 現在行がリングバッファ先頭に来た場合ずらす
    if ( Inherited::m_Y == Inherited::m_TopIndex )
    {
        // 表示位置も先頭だった場合、ずらす
        if ( Inherited::m_TopIndex == Inherited::m_DispIndex )
        {
            Inherited::YIncrement( Inherited::m_DispIndex );
        }
        Inherited::YIncrement(Inherited::m_TopIndex);
    }

    Inherited::ClearLine(Inherited::m_Y);
}

template <typename T>
int ScrollConsole<T>::Scroll( int lines ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( lines == 0 )
    {
        return 0;
    }

    int disp = Inherited::YDiff( Inherited::m_Y, Inherited::m_DispIndex ) + 1;    // 現在行から表示行までの距離+1 (表示している行数)
    int bufs = Inherited::YDiff( Inherited::m_DispIndex, Inherited::m_TopIndex ); // 表示行からバッファトップまでの距離

    if ( lines < 0 )
    {
        lines = ( -lines > bufs )? -bufs: lines;
        Inherited::YSub( Inherited::m_DispIndex, -lines );
    }
    else
    {
        // 表示行が画面行より小さい場合
        if ( disp < Inherited::m_ViewHeight )
        {
            lines = 0;
        }
        else
        {
            lines = ( lines > disp - Inherited::m_ViewHeight )? disp - Inherited::m_ViewHeight: lines;
        }
        Inherited::YAdd( Inherited::m_DispIndex, lines );
    }

    return lines;
}

template <typename T>
void ScrollConsole<T>::Display() NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( Inherited::m_Printer == nullptr )
    {
        return;
    }

    int y = Inherited::m_DispIndex;
    for( int n=0; n<Inherited::m_ViewHeight; n++ )
    {
        Inherited::m_Printer( 0, n, Inherited::GetPtr(0, y), Inherited::GetAttrPtr(0, y), Inherited::m_Width, Inherited::m_PrinterArg );
        if ( y == Inherited::m_Y )
        {
            break;
        }
        Inherited::YIncrement( y );
        if ( y == Inherited::m_TopIndex )
        {
            break;
        }
    }
}

//================================================================================
// 固定コンソール
//
template <typename T>
const T* FixedConsole<T>::PrintfEx( int x, int y, const T* format, ... ) NN_NOEXCEPT
{
    SetPosition( x, y );

    va_list args;
    va_start( args, format );

    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    Inherited::PutString( reinterpret_cast<T*>(g_PrintBuffer) );
    va_end( args );
    return reinterpret_cast<T*>(g_PrintBuffer);
}
template <>
const char16_t* FixedConsole<char16_t>::PrintfEx( int x, int y, const char16_t* format, ... ) NN_NOEXCEPT
{
    SetPosition( x, y );
    Inherited::PutString( format );
    return format;
}

template <typename T>
const T* FixedConsole<T>::PrintfEx( int x, int y, int attr, const T* format, ... ) NN_NOEXCEPT
{
    char attr0 = Inherited::m_Attr;
    Inherited::SetAttribute( attr );
    SetPosition( x, y );

    va_list args;
    va_start( args, format );

    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    Inherited::PutString( reinterpret_cast<T*>(g_PrintBuffer) );

    Inherited::SetAttribute( attr0 );
    va_end( args );
    return reinterpret_cast<T*>(g_PrintBuffer);
}
template <>
const char16_t* FixedConsole<char16_t>::PrintfEx( int x, int y, int attr, const char16_t* format, ... ) NN_NOEXCEPT
{
    char attr0 = Inherited::m_Attr;
    Inherited::SetAttribute( attr );
    SetPosition( x, y );

    Inherited::PutString( format );

    Inherited::SetAttribute( attr0 );
    return format;
}

template <typename T>
const T* FixedConsole<T>::PrintfEx( int attr, const T* format, ... ) NN_NOEXCEPT
{
    char attr0 = Inherited::m_Attr;
    Inherited::SetAttribute( attr );

    va_list args;
    va_start( args, format );

    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    Inherited::PutString( reinterpret_cast<T*>(g_PrintBuffer) );

    Inherited::SetAttribute( attr0 );
    va_end( args );
    return reinterpret_cast<T*>(g_PrintBuffer);
}
template <>
const char16_t* FixedConsole<char16_t>::PrintfEx( int attr, const char16_t* format, ... ) NN_NOEXCEPT
{
    char attr0 = Inherited::m_Attr;
    Inherited::SetAttribute( attr );

    Inherited::PutString( format );

    Inherited::SetAttribute( attr0 );
    return format;
}

template <typename T>
void FixedConsole<T>::Clear() NN_NOEXCEPT
{
    Inherited::m_X = 0;
    Inherited::m_Y = 0;
    Inherited::m_TopIndex = 0;
    Inherited::m_Attr = app::ConsoleConsole_Default;

    ClearScreen();
}

template <typename T>
void FixedConsole<T>::ClearLine( int y ) NN_NOEXCEPT
{
    if (y<0 || y>=Inherited::m_BufLine )
    {
        return;
    }

    T* pChar = Inherited::GetPtr(0,y);
    char* pAttr = Inherited::GetAttrPtr(0,y);

    for( int x=0; x<Inherited::m_Width; x++ )
    {
        *pChar++ = 0;
        *pAttr++ = 0;
    }
}

template <typename T>
void FixedConsole<T>::ClearScreen() NN_NOEXCEPT
{
    for( int y=0; y<Inherited::m_BufLine; y++ )
    {
        ClearLine(y);
    }
}

template <typename T>
void FixedConsole<T>::SetPosition( int x, int y ) NN_NOEXCEPT
{
    if ( x<0 || x>=Inherited::m_Width || y<0 || y>=Inherited::m_BufLine )
    {
        return ;
    }
    Inherited::m_X = x;
    Inherited::m_Y = y;
}

template <typename T>
void FixedConsole<T>::Display() NN_NOEXCEPT
{
    if ( Inherited::m_Printer == nullptr )
    {
        return;
    }

    for( int y=0; y<Inherited::m_BufLine; y++ )
    {
        Inherited::m_Printer( 0, y, Inherited::GetPtr(0,y), Inherited::GetAttrPtr(0,y), Inherited::m_Width, Inherited::m_PrinterArg );
    }
}

template <typename T>
void FixedConsole<T>::FillLineAttr( int y, int attr ) NN_NOEXCEPT
{
    for( int x=0; x<Inherited::m_Width; x++ )
    {
        Inherited::PutAttr( x, y, attr );
    }
}

template <typename T>
void FixedConsole<T>::ArrangeY_inPutString() NN_NOEXCEPT
{
    // 現在行が表示行を超えた場合の考慮
    if ( Inherited::m_Y >= Inherited::m_ViewHeight )
    {
        Inherited::m_Y = 0;
    }
}

template class FixedConsole<char>;
template class FixedConsole<char16_t>;

//================================================================================
// 固定コンソール(プロポーショナル)
//
template <typename T>
const T* FixedProportionalConsole<T>::Printf( const T* format, ... ) NN_NOEXCEPT
{
    return nullptr;
}

template <typename T>
const T* FixedProportionalConsole<T>::AddString( int x, int y, int attr, const T* buffer ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( ! buffer )
    {
        return nullptr;
    }
    T* pChar = Inherited::GetPtr(0, y);
    char* pAttr = Inherited::GetAttrPtr(0, y);

    T* p = pChar;
    char* pA = pAttr;

    int num = (*p << 8) | *(p + 1);
    p += 2;
    pA += 2;
    for( int i=0; i<num; i++ )
    {
        int xx = (*p << 8) | *(p + 1);
        p += 2;
        pA += 2;
        int len = (*p << 8) | *(p + 1);
        p += 2;
        pA += 2;

        if ( x == xx )
        {
            return nullptr;
        }

        p += len + 1;
        pA += len + 1;
    }

    // この時点で p, pA の部分に書き込める
    int len = ( sizeof(T) == 1 )? strlen( reinterpret_cast<const char*>(buffer) ): app::StrLen16( reinterpret_cast<const char16_t*>(buffer) );
    if ( (p - pChar) + 4 + len > Inherited::m_Width )
    {
        return nullptr;
    }

    // 情報
    *p = x >> 8;
    *(p + 1) = x & 0xff;
    *(p + 2) = len >> 8;
    *(p + 3) = len & 0xff;

    p += 4;
    pA += 4;

    const T* pBuffer = buffer;
    for( int i=0; i<len; i++ )
    {
        *p++ = *pBuffer++;
        *pA++ = attr;
    }

    // 個数調整
    num ++;
    *pChar = (num >> 8);
    *(pChar + 1) = (num & 0xff);

    return buffer;
}

template <typename T>
const T* FixedProportionalConsole<T>::PrintfEx( int x, int y, const T* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( y<0 || y >= Inherited::m_BufLine )
    {
        return nullptr;
    }

    if ( IsStringExist( x, y ) )
    {
        ClearPartial( x, y );
    }

    va_list args;
    va_start( args, format );
    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    if ( ! AddString( x, y, Inherited::m_Attr, g_PrintBuffer ) )
    {
        return nullptr;
    }
    va_end( args );
    return g_PrintBuffer;
}
template <>
const char16_t* FixedProportionalConsole<char16_t>::PrintfEx( int x, int y, const char16_t* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( y<0 || y >= Inherited::m_BufLine )
    {
        return nullptr;
    }

    if ( IsStringExist( x, y ) )
    {
        ClearPartial( x, y );
    }

    if ( ! AddString( x, y, Inherited::m_Attr, format ) )
    {
        return nullptr;
    }
    return format;
}

template <typename T>
const T* FixedProportionalConsole<T>::PrintfEx( int x, int y, int attr, const T* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( y<0 || y >= Inherited::m_BufLine )
    {
        return nullptr;
    }

    if ( IsStringExist( x, y ) )
    {
        ClearPartial( x, y );
    }

    va_list args;
    va_start( args, format );
    vsnprintf( g_PrintBuffer, sizeof(g_PrintBuffer), format, args );
    if ( ! AddString( x, y, attr, g_PrintBuffer ) )
    {
        return nullptr;
    }
    va_end( args );
    return g_PrintBuffer;
}
template <>
const char16_t* FixedProportionalConsole<char16_t>::PrintfEx( int x, int y, int attr, const char16_t* format, ... ) NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( y<0 || y >= Inherited::m_BufLine )
    {
        return nullptr;
    }

    if ( IsStringExist( x, y ) )
    {
        ClearPartial( x, y );
    }

    if ( ! AddString( x, y, attr, format ) )
    {
        return nullptr;
    }
    return format;
}

template <typename T>
const T* FixedProportionalConsole<T>::PrintfEx( int attr, const T* format, ... ) NN_NOEXCEPT
{
    return nullptr;
}

template <typename T>
void FixedProportionalConsole<T>::Clear() NN_NOEXCEPT
{
    Inherited::m_X = 0;
    Inherited::m_Y = 0;
    Inherited::m_TopIndex = 0;
    Inherited::m_Attr = app::ConsoleConsole_Default;

    ClearScreen();
}

template <typename T>
void FixedProportionalConsole<T>::ClearLine( int y ) NN_NOEXCEPT
{
    if (y<0 || y>=Inherited::m_BufLine )
    {
        return;
    }

    T* pChar = Inherited::GetPtr(0,y);
    char* pAttr = Inherited::GetAttrPtr(0,y);

    for( int x=0; x<Inherited::m_Width; x++ )
    {
        *pChar++ = 0;
        *pAttr++ = 0;
    }
}

template <typename T>
void FixedProportionalConsole<T>::ClearScreen() NN_NOEXCEPT
{
    for( int y=0; y<Inherited::m_BufLine; y++ )
    {
        ClearLine(y);
    }
}

template <typename T>
void FixedProportionalConsole<T>::SetPosition( int x, int y ) NN_NOEXCEPT
{
    if ( x<0 || x>=Inherited::m_Width || y<0 || y>=Inherited::m_BufLine )
    {
        return;
    }
    Inherited::m_X = x;
    Inherited::m_Y = y;
}

template <typename T>
void FixedProportionalConsole<T>::Display() NN_NOEXCEPT
{
    Inherited::Lock();
    NN_UTIL_SCOPE_EXIT
    {
        Inherited::Unlock();
    };

    if ( Inherited::m_Printer == nullptr )
    {
        return;
    }

    for( int y=0; y<Inherited::m_BufLine; y++ )
    {
        T* pChar = Inherited::GetPtr(0,y);
        char* pAttr = Inherited::GetAttrPtr(0,y);
        T* p = pChar;

        int num = (*p << 8) | *(p + 1);
        p += 2;
        for( int i=0; i<num; i++ )
        {
            int x = (*p << 8) | *(p + 1);
            p += 2;
            int len = (*p << 8) | *(p + 1);
            p += 2;

            Inherited::m_Printer( x, y, p, pAttr + (p - pChar), len, Inherited::m_PrinterArg );
            p += len + 1;
        }
    }
}

template <typename T>
void FixedProportionalConsole<T>::FillLineAttr( int y, int attr ) NN_NOEXCEPT
{
    for( int x=0; x<Inherited::m_Width; x++ )
    {
        Inherited::PutAttr( x, y, attr );
    }
}

template <typename T>
void FixedProportionalConsole<T>::ArrangeY_inPutString() NN_NOEXCEPT
{
    // 現在行が表示行を超えた場合の考慮
    if ( Inherited::m_Y >= Inherited::m_ViewHeight )
    {
        Inherited::m_Y = 0;
    }
}

template <typename T>
bool FixedProportionalConsole<T>::IsStringExist( int x, int y ) NN_NOEXCEPT
{
    if (y<0 || y>=Inherited::m_BufLine )
    {
        return false;
    }

    T* pChar = Inherited::GetPtr(0,y);
    T* p = pChar;

    int num = (*p << 8) | *(p + 1);
    p += 2;
    for( int i=0; i<num; i++ )
    {
        int xx = (*p << 8) | *(p + 1);
        p += 2;
        int len = (*p << 8) | *(p + 1);
        p += 2;

        if ( xx == x )
        {
            return true;
        }
        p += len + 1;
    }
    return false;
}

template <typename T>
void FixedProportionalConsole<T>::ClearPartial( int x, int y ) NN_NOEXCEPT
{
    if (y<0 || y>=Inherited::m_BufLine )
    {
        return;
    }

    T* pChar = Inherited::GetPtr(0,y);
    char* pAttr = Inherited::GetAttrPtr(0,y);
    T* p = pChar;
    char* pA = pAttr;

    int num = (*p << 8) | *(p + 1);
    p += 2;
    pA += 2;
    for( int i=0; i < num; i++ )
    {
        int xx = (*p << 8) | *(p + 1);
        p += 2;
        pA += 2;
        int len = (*p << 8) | *(p + 1);
        p += 2;
        pA += 2;

        if ( x != xx )
        {
            p += len + 1;
            pA += len + 1;
            continue;
        }

        // 発見した。まずデータを消す
        for( int j=0; j<len; j++ )
        {
            *(p + j) = 0;
            *(pA + j) = 0;
        }
        // 後ろをずらす
        if ( i + 1 < num )
        {
            T* pNext = p + len + 1;
            char* pANext = pA + len + 1;
            p -= 4;
            pA -= 4;

            while( pNext - pChar < Inherited::m_Width )
            {
                *p++ = *pNext++;
                *pA++ = *pANext++;
            }
        }
        // 個数調整
        num --;
        *pChar = (num >> 8);
        *(pChar + 1) = (num & 0xff);
        break;
    }
}

//================================================================================
// 実体化

template class Console<char>;
template class Console<char16_t>;

template class ScrollConsole<char>;
template class ScrollConsole<char16_t>;

template class FixedProportionalConsole<char>;
template class FixedProportionalConsole<char16_t>;

} // namespace app

