﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <string>
#include <list>
#include <vector>
#include <unordered_map>
#include <cerrno>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_ApplicationId.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/htc/htc_Api.h>
#include "BackupSaveData.h"

// #define ENABLE_SAVE_DATA_INFO_LOG
// #define LOG_COPY_FILE_NAME

namespace {
    std::string exportPath;
    std::string mountHostPath;
    const std::string userExportDirName = "/user/";
#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    const std::string systemExportDirName = "/system/";
#endif
    const nn::ApplicationId pseudoApplicationId = { 0x0000000000000000 };
    std::unordered_map<uint64_t, uint8_t> TitleSaveDataNumMap;
#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    std::unordered_map<uint64_t, uint8_t> SystemSaveDataNumMap;
#endif
#if !defined(USE_MALLOC)
    const size_t MemoryHeapSize = 12 * 1024 * 1024;
    nn::lmem::HeapHandle HeapHandleForSaveData;
    uint8_t HeapBuffer[MemoryHeapSize];
#endif
#if defined(SKIP_BACKUP_DEVMENU_SAVEDATA)
    uint64_t DevMenuApplicationIds[] = { 0x0100000000002040ULL, 0x0100000000002042ULL,
                        0x0100000000002064ULL, 0x0100000000002065ULL, 0x0100000000002073ULL };
#endif
#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    const nn::Bit64 SystemSaveDataId0Mask = 0xFFFFFFFFFFFFF000ULL;
    const nn::Bit64 SystemSaveDataId0Base = 0x8000000000001000ULL;
    uint64_t SkipSystemSaveDataIds[] = { 0x8000000000001050ULL };
#endif

#if defined(ENABLE_SAVE_DATA_INFO_LOG)
    void DumpSaveDataInfo(nn::fs::SaveDataType type, nn::ncm::ApplicationId appId, nn::fs::UserId userId, nn::Bit64 ownerId, int64_t saveDataSize, size_t saveDataAvailableSize, size_t saveDataJournalSize, uint32_t saveDataFlags, uint8_t saveDataSpaceId, uint16_t cacheIndex)
    {
        NN_LOG("data info : %d\n"
               "            %016lx\n"
               "            %016lx %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %08x\n"
               "            %02x\n"
               "            %04x\n"
            ,(int)type, appId.value, userId._data[0], userId._data[1], ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags, saveDataSpaceId, cacheIndex);
    }

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    void DumpSaveDataInfo(nn::fs::SystemSaveDataId systemSaveDataId, nn::fs::UserId userId, nn::Bit64 ownerId, int64_t saveDataSize, size_t saveDataAvailableSize, size_t saveDataJournalSize, uint32_t saveDataFlags)
    {
        NN_LOG("data info : %016lx\n"
               "            %016lx %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %016lx\n"
               "            %08x\n"
            ,systemSaveDataId, userId._data[0], userId._data[1], ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
    }
#endif // defined(NN_BACKUPSAVEDATA_SYSTEM)
#else
#define DumpSaveDataInfo(...)
#endif // defined(ENABLE_SAVE_DATA_INFO_LOG)

    std::string GetSaveDataTypeString(nn::fs::SaveDataType type)
    {
        switch(type)
        {
            case nn::fs::SaveDataType::System:
                return "System";
            case nn::fs::SaveDataType::Account:
                return "Account";
            case nn::fs::SaveDataType::Bcat:
                return "Bcat";
            case nn::fs::SaveDataType::Device:
                return "Device";
            case nn::fs::SaveDataType::Cache:
                return "Cache";
            default:
                return "";
        }
    }

    nn::fs::SaveDataType GetSaveDataType(std::string typeString)
    {
        if(typeString == "System")
        {
            return nn::fs::SaveDataType::System;
        }
        else if(typeString ==  "Account")
        {
            return nn::fs::SaveDataType::Account;
        }
        else if(typeString ==  "Bcat")
        {
            return nn::fs::SaveDataType::Bcat;
        }
        else if(typeString ==  "Device")
        {
            return nn::fs::SaveDataType::Device;
        }
        else if(typeString ==  "Cache")
        {
            return nn::fs::SaveDataType::Cache;
        }
        else
        {
            return nn::fs::SaveDataType::Account;
        }
    }

    std::string GetUint64String(uint64_t value)
    {
        char valueString[32];
        memset(valueString, 0, sizeof(valueString));
        sprintf(valueString, "%016lx", value);
        return valueString;
    }

    std::string GetInt64String(int64_t value)
    {
        return GetUint64String(value);
    }

    std::string GetUint32String(uint32_t value)
    {
        char valueString[16];
        memset(valueString, 0, sizeof(valueString));
        sprintf(valueString, "%08x", value);
        return valueString;
    }

    std::string GetInt32String(int32_t value)
    {
        return GetUint32String(value);
    }

    std::string GetAppIdString(uint64_t titleId)
    {
        return GetUint64String(titleId);
    }

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    std::string GetSystemSaveDataIdString(nn::fs::SystemSaveDataId systemSaveDataId)
    {
        return GetAppIdString(systemSaveDataId);
    }
#endif

    bool CheckTitleIdString(std::string titleString)
    {
        errno = 0;
        std::strtoull(titleString.c_str(), NULL, 16);  // only for check xdigit
        if(titleString.length() == 16 && !errno)
        {
            return true;
        }
        return false;
    }

    bool CheckBackupTitileId(uint64_t titleId)
    {
        if(titleId == pseudoApplicationId.value)
        {
            return false;
        }
#if defined(SKIP_BACKUP_DEVMENU_SAVEDATA)
        for(uint64_t skipId : DevMenuApplicationIds)
        {
            if(titleId == skipId)
            {
                return false;
            }
        }
#endif
        return true;
    }

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    bool CheckBackupSystemSaveDataId(nn::fs::SystemSaveDataId saveDataId)
    {
        if((saveDataId & SystemSaveDataId0Mask) == SystemSaveDataId0Base)
        {
            for(uint64_t skipId : SkipSystemSaveDataIds)
            {
                if(saveDataId == skipId)
                {
                    return false;
                }
            }
            return true;
        }
        return false;
    }
#endif

    bool CheckSkipCopyFileName(std::string fileName)
    {
        if(fileName == SaveDataInfoFileName)
        {
            return true;
        }
        return false;
    }

    std::string GetSaveDataCountString(uint8_t count)
    {
        char countString[32];
        memset(countString, 0, sizeof(countString));
        sprintf(countString, "%04x", count);
        return countString;
    }

    ::nn::Result GetTitleSaveDataCount(uint64_t titleId, uint8_t& count)
    {
        if(TitleSaveDataNumMap.count(titleId) == 0)
        {
            count = 1;
            TitleSaveDataNumMap[titleId] = count;
            return nn::ResultSuccess();
        }
        count = TitleSaveDataNumMap[titleId] + 1;
        TitleSaveDataNumMap[titleId] = count;
        return nn::ResultSuccess();
    }

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    ::nn::Result GetSystemSaveDataCount(uint64_t systemSaveDataId, uint8_t& count)
    {
        if(SystemSaveDataNumMap.count(systemSaveDataId) == 0)
        {
            count = 1;
            SystemSaveDataNumMap[systemSaveDataId] = count;
            return nn::ResultSuccess();
        }
        count = SystemSaveDataNumMap[systemSaveDataId] + 1;
        SystemSaveDataNumMap[systemSaveDataId] = count;
        return nn::ResultSuccess();
    }
#endif

    ::nn::Result ConvertTitileIdsStringToList(std::list<uint64_t> *titleIdList, std::string titleIds)
    {
        if(titleIds.empty())
        {
            titleIdList->clear();
            return nn::ResultSuccess();
        }

        std::string tidsString = titleIds;
        for(size_t c = tidsString.find_first_of(" "); c != std::string::npos; c = tidsString.find_first_of(" ")){
            tidsString.erase(c,1);
        }

        std::stringstream ss(tidsString);
        std::string s;
        titleIdList->clear();
        while (std::getline(ss, s, ',')) {
            uint64_t titleId = std::stoull(s, nullptr, 16);
            titleIdList->push_back(titleId);
        }
        return nn::ResultSuccess();
    }

    bool CheckTitleIdFromList(std::list<uint64_t> &titleIdList, uint64_t titleId)
    {
        std::list<uint64_t>::iterator it = titleIdList.begin();
        while( it != titleIdList.end() )
        {
            if(*it == titleId)
            {
                return true;
            }
            ++it;
        }
        return false;
    }

    ::nn::Result GetSaveDataNumMap(nn::fs::SaveDataIterator &iter, std::unordered_map<uint64_t, int> &titleIdSaveDataNumMap)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
            if(titleIdSaveDataNumMap.count(info.applicationId.value) > 0)
            {
                titleIdSaveDataNumMap[info.applicationId.value]++;
            }
        }
        return nn::ResultSuccess();
    }

    ::nn::Result CheckUserSaveDataExists(std::list<uint64_t> &titleIdList)
    {
        nn::Result result;
        std::unordered_map<uint64_t, int> titleIdSaveDataNumMap;

        std::list<uint64_t>::iterator it = titleIdList.begin();
        while( it != titleIdList.end() )
        {
            titleIdSaveDataNumMap[*it] = 0;
            ++it;
        }

        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        std::unique_ptr<nn::fs::SaveDataIterator> itersd;
        RETURN_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
        RETURN_IF_FAILED(GetSaveDataNumMap(*iter, titleIdSaveDataNumMap));
        {
            nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
            if(nn::fs::OpenSaveDataIterator(&itersd, nn::fs::SaveDataSpaceId::SdUser).IsSuccess())
            {
                RETURN_IF_FAILED(GetSaveDataNumMap(*itersd, titleIdSaveDataNumMap));
            }
        }

        result = nn::ResultSuccess();

        std::unordered_map<uint64_t, int>::iterator it2 = titleIdSaveDataNumMap.begin();
        while( it2 != titleIdSaveDataNumMap.end() )
        {
            if( it2->second == 0 )
            {
                NN_LOG("[ERROR] title %llx does't have save data.\n", it2->first);
                result = ResultApplicationError();
            }
            ++it2;
        }

        return result;
    }

    std::string GetStorageName(std::string path)
    {
        int pos = path.find(":");
        if(pos == std::string::npos)
        {
            return "";
        }
        return path.substr(0, pos);
    }

    bool CheckExistAccount(nn::fs::UserId userId)
    {
        return CheckDevkitAccountExist(userId);
    }

    bool CheckParsedSaveDataInfo(nn::fs::SaveDataType saveDataType, uint8_t saveDataSpaceId, int32_t cacheIndex)
    {
        if(saveDataType == nn::fs::SaveDataType::Cache)
        {
            if(cacheIndex < 0)
            {
                ERROR_S("The cache index is not set in cache storage save info.");
                return false;
            }
        }
        return true;
    }
}

static ::nn::Result DeletePartialTitleSaveData(std::list<uint64_t> &titleIdList);
static ::nn::Result ParseAccountFromConfigFiles(std::string baseDirectoryPath, uint64_t titleId);

::nn::Result InitializeDirectories()
{
    const int bufferSize = 1024;
    char buffer[bufferSize];
    size_t size;
    nn::Result result;

    exportPath = "host:/BackupSaveData";

    nn::htc::Initialize();
    RETURN_IF_FAILED(nn::htc::GetEnvironmentVariable(&size, buffer, bufferSize, "TEMP"));
    nn::htc::Finalize();
    mountHostPath = buffer;

    return nn::ResultSuccess();
}

::nn::Result InitilizeHeap()
{
    nn::Result result;

#if !defined(USE_MALLOC)
    HeapHandleForSaveData = nn::lmem::CreateExpHeap(HeapBuffer,
                                                    MemoryHeapSize,
                                                    nn::lmem::CreationOption_DebugFill);
                                                    // nn::lmem::CreationOption_NoOption);
    if(HeapHandleForSaveData == nullptr)
    {
        ERROR_S("Create heap for save data FAILED.");
        return ResultApplicationError();
    }
#endif

    return nn::ResultSuccess();
}

void FinalizeHeap()
{
#if !defined(USE_MALLOC)
    if(HeapHandleForSaveData != nullptr)
    {
        nn::lmem::DestroyExpHeap(HeapHandleForSaveData);
    }
#endif
}

static void* SaveDataHeapAllocate(size_t size)
{
#if !defined(USE_MALLOC)
    return nn::lmem::AllocateFromExpHeap(HeapHandleForSaveData, size, 64);
#else
    return std::malloc(size);
#endif
}

static void SaveDataHeapFree(void* addr)
{
#if !defined(USE_MALLOC)
    nn::lmem::FreeToExpHeap(HeapHandleForSaveData, addr);
#else
    std::free(addr);
#endif
}

static ::nn::Result CreateExportDirectory(bool isPartial = false)
{
    std::string tmpdir;
    nn::Result result;

    RETURN_IF_FAILED(nn::fs::MountHost("host", mountHostPath.c_str()));

    result = nn::fs::DeleteDirectoryRecursively(exportPath.c_str());
    if(!result.IsSuccess() && !nn::fs::ResultPathNotFound().Includes(result))
    {
        ERROR_LOG("nn::fs::DeleteDirectoryRecursively() failed (%llx)\n", result.GetInnerValueForDebug());
        goto quit;
    }

    QUIT_IF_FAILED(nn::fs::CreateDirectory(exportPath.c_str()));
    QUIT_IF_FAILED(CreateConfigFile(exportPath, isPartial));

    result = nn::ResultSuccess();
quit:
    nn::fs::Unmount("host");

    return result;
}

bool IsExistFile(std::string path)
{
    // MEMO: mountしていることが前提
    nn::Result result;
    nn::fs::DirectoryEntryType dummyEntry;

    result = nn::fs::GetEntryType(&dummyEntry, path.c_str());
    if(!result.IsSuccess())
    {
        return false;
    }
    return true;
}

static ::nn::Result CopySaveDataDirectory(std::string destPath, std::string srcPath, bool isDestPathSaveData, int64_t destSaveDataJournalSize = 0)
{
    //TODO: ヒープ割り当てをラベルquitにして QUIT_IF_FAILED を使う
    NN_LOG("copy %s to %s\n", srcPath.c_str(), destPath.c_str());
    nn::Result result = nn::ResultSuccess();
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    bool isCommit = isDestPathSaveData;
    std::string saveDataStorageName = GetStorageName(destPath);

    if(isDestPathSaveData)
    {
        if(destSaveDataJournalSize == 0)
        {
            NN_LOG("[ERROR] To copy savedata, must set destSaveDataJournalSize.");
            return ResultApplicationError();
        }
        if(destSaveDataJournalSize < 0x4000)
        {
            NN_LOG("[WARN] destSaveDataJournalSize is too small. (%llx)", destSaveDataJournalSize);
        }
    }

    // Copy current directory.
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, srcPath.c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryNum = 0;
    int64_t readNum = 0;
    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        nn::fs::FileHandle frHandle, fwHandle;
        int64_t fileSize;
        if(CheckSkipCopyFileName(dEntryArea[i].name))
        {
            continue;
        }
#if defined(LOG_COPY_FILE_NAME)
        NN_LOG("  copy file : %s\n", dEntryArea[i].name);
#endif
        std::string srcName = dEntryArea[i].name;
        std::string destName = isDestPathSaveData ? DecodeFromBackupFileName(dEntryArea[i].name) : EncodeToBackupFileName(dEntryArea[i].name);
        if(destName == "")
        {
            return ResultApplicationError();
        }
        std::string rPath = srcPath + '/' + srcName;
        std::string path = destPath + '/' + destName;
        RETURN_IF_FAILED(nn::fs::OpenFile(&frHandle, rPath.c_str(), nn::fs::OpenMode_Read));
        RETURN_IF_FAILED(nn::fs::GetFileSize(&fileSize, frHandle));
        if(IsExistFile(path))
        {
            NN_LOG("  skipped %s\n", dEntryArea[i].name);
            continue;
        }
        RETURN_IF_FAILED(nn::fs::CreateFile(path.c_str(), fileSize));
        if(isCommit)
        {
            RETURN_IF_FAILED(nn::fs::CommitSaveData(saveDataStorageName.c_str()));
        }
        const int64_t bufferSize = 4096;
        int64_t journalSize = isDestPathSaveData ? destSaveDataJournalSize : fileSize;
        char buffer[bufferSize];
        for(int64_t joffset=0; joffset<fileSize; joffset+=journalSize)
        {
            int64_t jSize = std::min(fileSize - joffset, journalSize);
            RETURN_IF_FAILED(nn::fs::OpenFile(&fwHandle, path.c_str(), nn::fs::OpenMode_Write));
            for(int64_t boffset=0; boffset<jSize; boffset+=bufferSize)
            {
                int64_t offset = joffset + boffset;
                int64_t size = std::min(jSize - boffset, bufferSize);
                RETURN_IF_FAILED(nn::fs::ReadFile(frHandle, offset, buffer, static_cast<int>(size)));
                RETURN_IF_FAILED(nn::fs::WriteFile(fwHandle, offset, buffer, static_cast<int>(size), nn::fs::WriteOption()));
            }
            RETURN_IF_FAILED(nn::fs::FlushFile(fwHandle));
            nn::fs::CloseFile(fwHandle);
            if(isCommit)
            {
                RETURN_IF_FAILED(nn::fs::CommitSaveData(saveDataStorageName.c_str()));
            }
        }
        nn::fs::CloseFile(frHandle);
        if(isCommit)
        {
            RETURN_IF_FAILED(nn::fs::CommitSaveData(saveDataStorageName.c_str()));
        }
    }
    SaveDataHeapFree(reinterpret_cast<void*>(dEntryArea));

    // Call self function recursively.
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, srcPath.c_str(), nn::fs::OpenDirectoryMode_Directory));
    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        std::string rPath = srcPath + '/' + dEntryArea[i].name;
        std::string dPath = destPath + '/' + dEntryArea[i].name;
        result = nn::fs::CreateDirectory(dPath.c_str());
        if(!result.IsSuccess() && !nn::fs::ResultPathAlreadyExists().Includes(result))
        {
            return result;
        }
        if(isCommit)
        {
            RETURN_IF_FAILED(nn::fs::CommitSaveData(saveDataStorageName.c_str()));
        }
        RETURN_IF_FAILED(CopySaveDataDirectory(dPath, rPath, isDestPathSaveData, destSaveDataJournalSize));
    }

    SaveDataHeapFree(reinterpret_cast<void*>(dEntryArea));

    return result;
}   // NOLINT(impl/function_size)


static ::nn::Result MountSaveDataUser(nn::fs::SaveDataType safeDataType, std::string mountName, nn::ncm::ApplicationId appId, nn::fs::UserId userId, int cacheIndex)
{
    switch(safeDataType)
    {
    case nn::fs::SaveDataType::System:
        NN_LOG("error: Mount System is invalid.\n");
        return ResultApplicationError();
    case nn::fs::SaveDataType::Account:
        return nn::fs::MountSaveData(mountName.c_str(), appId, userId);
    case nn::fs::SaveDataType::Bcat:
        return nn::fs::MountBcatSaveData(mountName.c_str(), appId);
    case nn::fs::SaveDataType::Device:
#if defined(NN_BACKUPSAVEDATA_DEVICE)
        return nn::fs::MountDeviceSaveData(mountName.c_str(), appId);
#else
        NN_LOG("error: Cannot mount type %d.\n", nn::fs::SaveDataType::Device);
        return ResultApplicationError();
#endif
    case nn::fs::SaveDataType::Cache:
        return nn::fs::MountCacheStorage(mountName.c_str(), appId, cacheIndex);
    default:
        return ResultApplicationError();
    }
}


static ::nn::Result ExportSaveData(const nn::fs::SaveDataInfo& saveInfo)
{
    nn::Result result;

#if !defined(NN_BACKUPSAVEDATA_DEVICE) && defined(SKIP_BACKUP_DEVICE_SAVEDATA)
    if(saveInfo.saveDataType == nn::fs::SaveDataType::Device)
    {
#if defined(SKIP_BACKUP_DEVICE_SAVEDATA)
        // MEMO: 過去のソフトと動作を一致させるために存在
        //       現在は不要
        NN_LOG("info: skipped save data type %d. (titleId = %016lx)\n", nn::fs::SaveDataType::Device, saveInfo.applicationId.value);
#endif
        return nn::ResultSuccess();
    }
#endif

    uint8_t titleSaveDataCount;
    GetTitleSaveDataCount(saveInfo.applicationId.value, titleSaveDataCount);

    nn::Bit64 ownerId;
    uint32_t saveDataFlags;
    int64_t saveDataAvailableSize;
    int64_t saveDataJournalSize;
    RETURN_IF_FAILED(nn::fs::GetSaveDataOwnerId(&ownerId, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataFlags(&saveDataFlags, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataAvailableSize(&saveDataAvailableSize, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataJournalSize(&saveDataJournalSize, saveInfo.saveDataSpaceId, saveInfo.saveDataId));

    nn::ncm::ApplicationId appId;
    appId.value = saveInfo.applicationId.value;
    int cacheIndex = saveInfo.index;
    RETURN_IF_FAILED(MountSaveDataUser(saveInfo.saveDataType, "save", appId, saveInfo.saveDataUserId, cacheIndex));

    std::string hostExportPathBase = exportPath + userExportDirName + GetAppIdString(saveInfo.applicationId.value);
    NN_LOG("create %s\n", hostExportPathBase.c_str());
    result = nn::fs::CreateDirectory(hostExportPathBase.c_str());
    if(!result.IsSuccess() && !nn::fs::ResultPathAlreadyExists().Includes(result))
    {
        NN_LOG("error: failed to crate directory %s (%08x)\n",
                hostExportPathBase.c_str(), result.GetInnerValueForDebug());
        goto quit;
    }
    hostExportPathBase += "/" + GetSaveDataCountString(titleSaveDataCount);
    QUIT_IF_FAILED(nn::fs::CreateDirectory(hostExportPathBase.c_str()));
    DumpSaveDataInfo(saveInfo.saveDataType, saveInfo.applicationId, saveInfo.saveDataUserId, ownerId, saveInfo.saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags, saveInfo.saveDataSpaceId, cacheIndex);
    QUIT_IF_FAILED(CreateSaveDataInfoFile(hostExportPathBase,
                                            GetSaveDataTypeString(saveInfo.saveDataType),
                                            GetUint64String(saveInfo.saveDataUserId._data[0]),
                                            GetUint64String(saveInfo.saveDataUserId._data[1]),
                                            GetUint64String(saveInfo.applicationId.value),
                                            GetUint64String(saveInfo.saveDataSize),
                                            GetUint64String(ownerId),
                                            GetUint32String(saveDataFlags),
                                            GetInt64String(saveDataAvailableSize),
                                            GetInt64String(saveDataJournalSize),
                                            GetUint32String((uint32_t)saveInfo.saveDataSpaceId),
                                            saveInfo.saveDataType == nn::fs::SaveDataType::Cache ? GetInt32String(saveInfo.index) : ""));
    QUIT_IF_FAILED(CopySaveDataDirectory(hostExportPathBase, "save:/", false));

    result = nn::ResultSuccess();
quit:
    nn::fs::Unmount("save");

    return nn::ResultSuccess();
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
// static ::nn::Result ExportSaveDataSystem(nn::fs::SaveDataId saveDataId, nn::fs::SystemSaveDataId systemSaveDataId, nn::fs::UserId userId)
static ::nn::Result ExportSaveDataSystem(const nn::fs::SaveDataInfo& saveInfo)
{
    nn::Result result;

    uint8_t systemSaveDataCount;
    GetSystemSaveDataCount(saveInfo.systemSaveDataId, systemSaveDataCount);

    nn::Bit64 ownerId;
    uint32_t saveDataFlags;
    int64_t saveDataAvailableSize;
    int64_t saveDataJournalSize;
    RETURN_IF_FAILED(nn::fs::GetSaveDataOwnerId(&ownerId, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataFlags(&saveDataFlags, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataAvailableSize(&saveDataAvailableSize, saveInfo.saveDataSpaceId, saveInfo.saveDataId));
    RETURN_IF_FAILED(nn::fs::GetSaveDataJournalSize(&saveDataJournalSize, saveInfo.saveDataSpaceId, saveInfo.saveDataId));

    RETURN_IF_FAILED(nn::fs::MountSystemSaveData("ssave", saveInfo.systemSaveDataId, saveInfo.saveDataUserId));

    std::string hostExportPathBase = exportPath + systemExportDirName + GetSystemSaveDataIdString(saveInfo.systemSaveDataId);
    NN_LOG("create %s\n", hostExportPathBase.c_str());
    result = nn::fs::CreateDirectory(hostExportPathBase.c_str());
    if(!result.IsSuccess() && !nn::fs::ResultPathAlreadyExists().Includes(result))
    {
        NN_LOG("error: failed to crate directory %s (%08x)\n",
                hostExportPathBase.c_str(), result.GetInnerValueForDebug());
        goto quit;
    }
    hostExportPathBase += "/" + GetSaveDataCountString(systemSaveDataCount);
    QUIT_IF_FAILED(nn::fs::CreateDirectory(hostExportPathBase.c_str()));
    DumpSaveDataInfo(saveInfo.systemSaveDataId, saveInfo.saveDataUserId, ownerId, saveInfo.saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
    QUIT_IF_FAILED(CreateSaveDataInfoFile(hostExportPathBase,
                                            GetSaveDataTypeString(saveInfo.saveDataType),
                                            GetUint64String(saveInfo.saveDataUserId._data[0]),
                                            GetUint64String(saveInfo.saveDataUserId._data[1]),
                                            GetUint64String(saveInfo.applicationId.value),
                                            GetUint64String(saveInfo.saveDataSize),
                                            GetUint64String(ownerId),
                                            GetUint32String(saveDataFlags),
                                            GetInt64String(saveDataAvailableSize),
                                            GetInt64String(saveDataJournalSize),
                                            "", ""));
    QUIT_IF_FAILED(CopySaveDataDirectory(hostExportPathBase, "ssave:/", false));

    result = nn::ResultSuccess();
quit:
    nn::fs::Unmount("ssave");

    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result DeleteSaveDataTitle(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId)
{
    NN_LOG("delete savedata (%016lx, %016lx)\n", saveDataId, saveDataSpaceId);
    return nn::fs::DeleteSaveData(saveDataSpaceId, saveDataId);
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
static ::nn::Result DeleteSaveDataSystem(nn::fs::SaveDataId saveDataId)
{
    NN_LOG("delete system savedata %016lx\n", saveDataId);
    return nn::fs::DeleteSaveData(saveDataId);
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result GetDirectorySizeRecursively(size_t& getsize, std::string dirPath)
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    int64_t totalFileSize = 0;

    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, dirPath.c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryNum = 0;
    int64_t readNum = 0;
    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        if(CheckSkipCopyFileName(dEntryArea[i].name))
        {
            continue;
        }
        nn::fs::FileHandle frHandle;
        std::string rPath = dirPath + '/' + dEntryArea[i].name;
        RETURN_IF_FAILED(nn::fs::OpenFile(&frHandle, rPath.c_str(), nn::fs::OpenMode_Read));
        RETURN_IF_FAILED(nn::fs::GetFileSize(&totalFileSize, frHandle));
        nn::fs::CloseFile(frHandle);
    }
    SaveDataHeapFree(reinterpret_cast<void*>(dEntryArea));

    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, dirPath.c_str(), nn::fs::OpenDirectoryMode_Directory));
    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        std::string nextDirPath = dirPath + '/' + dEntryArea[i].name;
        RETURN_IF_FAILED(GetDirectorySizeRecursively(getsize, nextDirPath));
    }
    SaveDataHeapFree(reinterpret_cast<void*>(dEntryArea));

    getsize += static_cast<size_t>(totalFileSize);

    return nn::ResultSuccess();
}

static ::nn::Result GetTitleSaveDataSize(size_t& getsize, uint64_t titleId, std::string exportNumberName)
{
    nn::Result result;
    std::string hostTitlePathBase = exportPath + userExportDirName + GetAppIdString(titleId) + '/' + exportNumberName;
    size_t hostSaveDataSizeSum = 0;

    QUIT_IF_FAILED(GetDirectorySizeRecursively(hostSaveDataSizeSum, hostTitlePathBase));

    // TODIRAEZU: ディレクトリサイズはファイルサイズの高々半分より小さいはず
    //            そして4Mbyte単位で切り上げる
    size_t returnSize;
    returnSize = (hostSaveDataSizeSum * 3) >> 1;
    returnSize = ((returnSize >> 22) + 1) << 22;
    NN_LOG("SaveDataSize(%016llx) = %016llx\n", titleId, returnSize);
    getsize = returnSize;
    return nn::ResultSuccess();

quit:
    return result;
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
static ::nn::Result GetSystemSaveDataSize(size_t& getsize, uint64_t systemSaveDataId, std::string exportNumberName)
{
    nn::Result result;
    std::string hostTitlePathBase = exportPath + systemExportDirName + GetSystemSaveDataIdString(systemSaveDataId) + '/' + exportNumberName;
    size_t hostSaveDataSizeSum = 0;

    QUIT_IF_FAILED(GetDirectorySizeRecursively(hostSaveDataSizeSum, hostTitlePathBase));

    // TODIRAEZU: ディレクトリサイズはファイルサイズの高々半分より小さいはず
    //            そして4Mbyte単位で切り上げる
    size_t returnSize;
    returnSize = (hostSaveDataSizeSum * 3) >> 1;
    returnSize = ((returnSize >> 22) + 1) << 22;
    NN_LOG("SaveDataSize(%016llx) = %016llx\n", systemSaveDataId, returnSize);
    getsize = returnSize;
    return nn::ResultSuccess();

quit:
    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result CreateTitleSaveData(nn::fs::SaveDataType saveDataType, uint64_t titleId, nn::fs::UserId userId, nn::Bit64 ownerId, int64_t saveDataSize, int64_t saveDataAvailableSize, int64_t saveDataJournalSize, uint32_t saveDataFlags, uint8_t saveDataSpaceId, int cacheIndex)
{
    if(saveDataSize == 0x0LL && saveDataAvailableSize == 0x0ULL)
    {
        return ResultApplicationError();
    }

    nn::Result result;

    // TODIRAEZU: ジャーナルサイズは設定サイズと同じにする
    //            セーブデータサイズは設定サイズの倍にする
    //            OwnerId は ApplicationId と同じにする
    nn::ncm::ApplicationId appId;
    appId.value = titleId;
    nn::fs::UserId createSaveDataUserId = userId;
    DumpSaveDataInfo(saveDataType, appId, createSaveDataUserId, ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags, saveDataSpaceId, cacheIndex);
    if(saveDataAvailableSize != 0x0ULL)
    {
        switch(saveDataType)
        {
            case nn::fs::SaveDataType::System:
                NN_LOG("error: Cannot use System type in title.\n");
                return ResultApplicationError();
            case nn::fs::SaveDataType::Account:
                return nn::fs::CreateSaveData(appId, createSaveDataUserId, ownerId, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
            case nn::fs::SaveDataType::Bcat:
                return nn::fs::CreateBcatSaveData(appId, saveDataAvailableSize);
            case nn::fs::SaveDataType::Device:
                return nn::fs::CreateDeviceSaveData(appId, ownerId, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
            case nn::fs::SaveDataType::Cache:
                return nn::fs::CreateCacheStorage(appId, (nn::fs::SaveDataSpaceId)saveDataSpaceId, ownerId, cacheIndex, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
            default:
                return ResultApplicationError();
        }
    }
    else
    {
        NN_LOG("info: use save data size from SaveDataInfo.saveDataSize.\n");
        switch(saveDataType)
        {
            case nn::fs::SaveDataType::System:
                NN_LOG("error: Cannot use System type in title.\n");
                return ResultApplicationError();
            case nn::fs::SaveDataType::Account:
                return nn::fs::CreateSaveData(appId, createSaveDataUserId, ownerId, saveDataSize, saveDataSize, saveDataFlags);
            case nn::fs::SaveDataType::Bcat:
                return nn::fs::CreateBcatSaveData(appId, saveDataSize);
            case nn::fs::SaveDataType::Device:
                return nn::fs::CreateDeviceSaveData(appId, ownerId, saveDataSize, saveDataSize, saveDataFlags);
            default:
                return ResultApplicationError();
        }
    }
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
static ::nn::Result CreateSaveDataForSystem(uint64_t systemSaveDataId, nn::fs::UserId userId, nn::Bit64 ownerId, int64_t saveDataSize, int64_t saveDataAvailableSize, int64_t saveDataJournalSize, uint32_t saveDataFlags)
{
    if(saveDataSize == 0x0LL && saveDataAvailableSize == 0x0ULL)
    {
        return ResultApplicationError();
    }

    nn::Result result;

    // TODIRAEZU: ジャーナルサイズは設定サイズと同じにする
    //            セーブデータサイズは設定サイズの倍にする
    //            OwnerId は ApplicationId と同じにする
    nn::fs::SystemSaveDataId ssdId = systemSaveDataId;
    nn::fs::UserId createSaveDataUserId = userId;
    DumpSaveDataInfo(ssdId, createSaveDataUserId, ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
    if(saveDataAvailableSize != 0x0ULL)
    {
        return nn::fs::CreateSystemSaveData(ssdId, createSaveDataUserId, ownerId, saveDataAvailableSize, saveDataJournalSize, saveDataFlags);
    }
    else
    {
        NN_LOG("info: use save data size from SaveDataInfo.saveDataSize.\n");
        return nn::fs::CreateSystemSaveData(ssdId, createSaveDataUserId, ownerId, saveDataSize, saveDataSize, saveDataFlags);
    }
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result ImportSaveData(uint64_t titleId, bool isReplaceAccount = false)
{
    nn::Result result;

    //TODO: タイトルが存在するかの確認を行ってなければ警告を出力

    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string hostImportPathBase = exportPath + userExportDirName + GetAppIdString(titleId);

    // タイトルディレクトリ以下にある各セーブデータをそれぞれインポートする
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, hostImportPathBase.c_str(), nn::fs::OpenDirectoryMode_Directory));
    int64_t entryNum = 0;
    int64_t readNum = 0;

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        std::string hostPath = hostImportPathBase + "/" + dEntryArea[i].name;
        uint64_t saveDataUserIdH, saveDataUserIdL, applicationId_dummy, saveDataSize;
        nn::Bit64 ownerId;
        uint32_t saveDataFlags;
        int64_t saveDataAvailableSize, saveDataJournalSize;
        std::string saveDataTypeString;
        uint8_t saveDataSpaceId;
        int32_t cacheIndex;
        RETURN_IF_FAILED(ParseSaveDataInfoFile(hostPath, saveDataTypeString, saveDataUserIdH, saveDataUserIdL, applicationId_dummy, saveDataSize, ownerId, saveDataFlags,
                                                (uint64_t&)saveDataAvailableSize, (uint64_t&)saveDataJournalSize, (uint32_t&)saveDataSpaceId, cacheIndex));
        nn::fs::UserId userIdFromInfoFile = {{saveDataUserIdH, saveDataUserIdL}};
        nn::fs::UserId userId = isReplaceAccount ? ConvertAccount(userIdFromInfoFile) : userIdFromInfoFile;
        nn::fs::SaveDataType saveDataType = GetSaveDataType(saveDataTypeString);
        // NN_LOG("save type : %s (%d)\n", saveDataTypeString.c_str(), (int)saveDataType);
        if(saveDataSize == 0x0ULL)
        {
            NN_LOG("info: calculate save data size.\n");
            RETURN_IF_FAILED(GetTitleSaveDataSize(saveDataSize, titleId, dEntryArea[i].name));
        }
#if !defined(NN_BACKUPSAVEDATA_DEVICE) && defined(SKIP_BACKUP_DEVICE_SAVEDATA)
        if(saveDataType == nn::fs::SaveDataType::Device)
        {
            NN_LOG("info: Skip backup save data type %d.\n", nn::fs::SaveDataType::Device);
            continue;
        }
#endif
        if(saveDataType == nn::fs::SaveDataType::Account &&
           isReplaceAccount && userId == nn::fs::InvalidUserId)
        {
            continue;
        }
        if(!CheckParsedSaveDataInfo(saveDataType, saveDataSpaceId, cacheIndex))
        {
            return ResultNotFound();
        }
        RETURN_IF_FAILED(CreateTitleSaveData(saveDataType, titleId, userId, ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags, saveDataSpaceId, cacheIndex));
        nn::ncm::ApplicationId appId = {titleId};
        RETURN_IF_FAILED(MountSaveDataUser(saveDataType, "save", appId, userId, cacheIndex));
        RETURN_IF_FAILED(CopySaveDataDirectory("save:/", hostPath, true, saveDataJournalSize));
        RETURN_IF_FAILED(nn::fs::CommitSaveData("save"));
       //TODO: quit ラベルをつけてエラー時はアンマウントさせるようにする
        nn::fs::Unmount("save");
    }

    return result;
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
static ::nn::Result ImportSystemSaveData(uint64_t systemSaveDataId)
{
    nn::Result result;

    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string hostImportPathBase = exportPath + systemExportDirName + GetSystemSaveDataIdString(systemSaveDataId);

    // システムセーブデータディレクトリ以下にある各セーブデータをそれぞれインポートする
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, hostImportPathBase.c_str(), nn::fs::OpenDirectoryMode_Directory));
    int64_t entryNum = 0;
    int64_t readNum = 0;

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        std::string hostPath = hostImportPathBase + "/" + dEntryArea[i].name;
        uint64_t saveDataUserIdH, saveDataUserIdL, applicationId, saveDataSize;
        nn::Bit64 ownerId;
        uint32_t saveDataFlags;
        int64_t saveDataAvailableSize, saveDataJournalSize;
        std::string saveDataTypeString;
        uint32_t saveDataSpaceId;
        int32_t cacheIndex;
        RETURN_IF_FAILED(ParseSaveDataInfoFile(hostPath, saveDataTypeString, saveDataUserIdH, saveDataUserIdL, applicationId, saveDataSize, ownerId, saveDataFlags,
                                                (uint64_t&)saveDataAvailableSize, (uint64_t&)saveDataJournalSize, saveDataSpaceId, cacheIndex));
        nn::fs::UserId userId = {{saveDataUserIdH, saveDataUserIdL}};
        nn::fs::SaveDataType saveDataType = GetSaveDataType(saveDataTypeString);
        if(saveDataSize == 0x0ULL)
        {
            NN_LOG("info: calculate save data size.\n");
            RETURN_IF_FAILED(GetSystemSaveDataSize(saveDataSize, systemSaveDataId, dEntryArea[i].name));
        }
        if(saveDataType != nn::fs::SaveDataType::System)
        {
            NN_LOG("warn: Not system save data type (%s)\n", saveDataTypeString.c_str());
        }
        RETURN_IF_FAILED(CreateSaveDataForSystem(systemSaveDataId, userId, ownerId, saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags));
        nn::fs::SystemSaveDataId ssdId = systemSaveDataId;
        RETURN_IF_FAILED(nn::fs::MountSystemSaveData("ssave", ssdId, userId));
        RETURN_IF_FAILED(CopySaveDataDirectory("ssave:/", hostPath, true, saveDataJournalSize));
        RETURN_IF_FAILED(nn::fs::CommitSaveData("ssave"));
       //TODO: quit ラベルをつけてエラー時はアンマウントさせるようにする
        nn::fs::Unmount("ssave");
    }

    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result ImportAllTitleSaveData(bool isReplaceAccount = false)
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string exportPathTitle = exportPath + userExportDirName;

    // host側からタイトルを全て取得してインポートを行う
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, exportPathTitle.c_str(), nn::fs::OpenDirectoryMode_Directory));
    int64_t entryNum = 0;
    int64_t readNum = 0;

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        if(!CheckTitleIdString(dEntryArea[i].name))
        {
            continue;
        }
        uint64_t titleId = std::stoull(dEntryArea[i].name, nullptr, 16);
        if(CheckBackupTitileId(titleId))
        {
            RETURN_IF_FAILED(ImportSaveData(titleId, isReplaceAccount));
        }
    }

    result = nn::ResultSuccess();

    return result;
}

static ::nn::Result DeleteAndImportPartialTitleSaveData(bool isReplaceAccount)
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string exportPathTitle = exportPath + userExportDirName;

    // host側からタイトルを全て取得してインポートを行う
    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, exportPathTitle.c_str(), nn::fs::OpenDirectoryMode_Directory));
    int64_t entryNum = 0;
    int64_t readNum = 0;

    std::list<uint64_t> titleIdList;
    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        if(!CheckTitleIdString(dEntryArea[i].name))
        {
            continue;
        }
        uint64_t titleId = std::stoull(dEntryArea[i].name, nullptr, 16);
        if(CheckBackupTitileId(titleId))
        {
            titleIdList.push_back(titleId);
        }
    }

    RETURN_IF_FAILED(DeletePartialTitleSaveData(titleIdList));

    std::list<uint64_t>::iterator it = titleIdList.begin();
    while( it != titleIdList.end() )
    {
        RETURN_IF_FAILED(ImportSaveData(*it, isReplaceAccount));
        ++it;
    }

    result = nn::ResultSuccess();

    return result;
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
static ::nn::Result ImportAllSaveData_system()
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string exportPathSystem = exportPath + systemExportDirName;

    // host側からシステムセーブデータIDを全て取得してインポートを行う
    result = nn::fs::OpenDirectory(&dHandle, exportPathSystem.c_str(), nn::fs::OpenDirectoryMode_Directory);
    if(!result.IsSuccess())
    {
        if(nn::fs::ResultPathNotFound().Includes(result))
        {
            NN_LOG("waring: system directory is not found.\n");
            return nn::ResultSuccess();
        }
        else
        {
            NN_LOG("error: directory open error %s (%08x)\n", exportPathSystem.c_str(), result.GetInnerValueForDebug());
            return result;
        }
    }

    int64_t entryNum = 0;
    int64_t readNum = 0;

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        uint64_t systemSaveDataId = std::stoull(dEntryArea[i].name, nullptr, 16);
        RETURN_IF_FAILED(ImportSystemSaveData(systemSaveDataId));
    }

    result = nn::ResultSuccess();

    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

namespace
{
    #if 0
    struct SaveDataIdPair
    {
        nn::fs::SaveDataId          saveDataId;
        nn::fs::SaveDataSpaceId     saveDataSpaceId;
    };
    #endif

    ::nn::Result GetSaveDataIds(nn::fs::SaveDataIterator& iter, std::list<nn::fs::SaveDataInfo>& saveDataInfoList)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupTitileId(info.applicationId.value))
            {
                continue;
            }
            saveDataInfoList.push_back(info);
        }
        return nn::ResultSuccess();
    }

    ::nn::Result GetSaveDataIds(std::list<uint64_t> &titleIdList, nn::fs::SaveDataIterator& iter, std::list<nn::fs::SaveDataInfo>& saveDataInfoList)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupTitileId(info.applicationId.value))
            {
                continue;
            }
            if(!CheckTitleIdFromList(titleIdList, info.applicationId.value))
            {
                continue;
            }
            saveDataInfoList.push_back(info);
        }
        return nn::ResultSuccess();
    }
}

static ::nn::Result DeleteAllTitleSaveData()
{
    nn::Result result;
    std::list<nn::fs::SaveDataInfo> saveDataInfoList;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    std::unique_ptr<nn::fs::SaveDataIterator> itersd;
    RETURN_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
    RETURN_IF_FAILED(GetSaveDataIds(*iter, saveDataInfoList));
    {
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&itersd, nn::fs::SaveDataSpaceId::SdUser).IsSuccess())
        {
            RETURN_IF_FAILED(GetSaveDataIds(*itersd, saveDataInfoList));
        }
    }

    for(auto itr = saveDataInfoList.begin(); itr != saveDataInfoList.end(); ++itr) {
        nn::fs::SaveDataInfo info = *itr;
        RETURN_IF_FAILED(DeleteSaveDataTitle(info.saveDataId, info.saveDataSpaceId));
    }

    result = nn::ResultSuccess();

    return result;
}

static ::nn::Result DeletePartialTitleSaveData(std::list<uint64_t> &titleIdList)
{
    nn::Result result;
    std::list<nn::fs::SaveDataInfo> saveDataInfoList;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    std::unique_ptr<nn::fs::SaveDataIterator> itersd;
    RETURN_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
    RETURN_IF_FAILED(GetSaveDataIds(titleIdList, *iter, saveDataInfoList));
    {
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&itersd, nn::fs::SaveDataSpaceId::SdUser).IsSuccess())
        {
            RETURN_IF_FAILED(GetSaveDataIds(titleIdList, *itersd, saveDataInfoList));
        }
    }

    for(auto itr = saveDataInfoList.begin(); itr != saveDataInfoList.end(); ++itr) {
        nn::fs::SaveDataInfo info = *itr;
        RETURN_IF_FAILED(DeleteSaveDataTitle(info.saveDataId, info.saveDataSpaceId));
    }

    result = nn::ResultSuccess();

    return result;
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
namespace
{
    ::nn::Result DeleteAllSaveData_system(nn::fs::SaveDataIterator& iter, std::list<nn::fs::SaveDataInfo>& saveDataInfoList)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupSystemSaveDataId(info.systemSaveDataId))
            {
                continue;
            }
            saveDataInfoList.push_back(info);
        }
        return nn::ResultSuccess();
    }
}

static ::nn::Result DeleteAllSaveData_system()
{
    nn::Result result;
    std::list<nn::fs::SaveDataInfo> saveDataInfoList;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    RETURN_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::System));
    RETURN_IF_FAILED(DeleteAllSaveData_system(*iter, saveDataInfoList));
    {
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::SdSystem).IsSuccess())
        {
            RETURN_IF_FAILED(DeleteAllSaveData_system(*iter, saveDataInfoList));
        }
    }

    for(auto itr = saveDataInfoList.begin(); itr != saveDataInfoList.end(); ++itr) {
        nn::fs::SaveDataInfo info = *itr;
        RETURN_IF_FAILED(DeleteSaveDataSystem(info.saveDataId));
    }

    result = nn::ResultSuccess();

    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

namespace
{
    ::nn::Result ExportSaveDataAll_user(bool isExistAccountOnly, nn::fs::SaveDataIterator& iter, std::string& hostPathBase)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            #if 0
            NN_LOG("save : %d\n       %016llx\n       %016llx\n       %x\n       %x\n       %016llx %016llx\n",
                                count, info.saveDataId, info.applicationId.value,
                                info.saveDataSpaceId, info.saveDataType,
                                info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
            #endif
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupTitileId(info.applicationId.value))
            {
                continue;
            }
            if(info.saveDataType == nn::fs::SaveDataType::Account &&
               isExistAccountOnly && !CheckExistAccount(info.saveDataUserId))
            {
                continue;
            }
            std::string hostPath = hostPathBase + GetAppIdString(info.applicationId.value);
            NN_LOG("hostPath : %s\n", hostPath.c_str());
            if(!IsExistFile(hostPath))
            {
                RETURN_IF_FAILED(nn::fs::CreateDirectory(hostPath.c_str()));
            }
            RETURN_IF_FAILED(ExportSaveData(info));
        }
        return nn::ResultSuccess();
    }

    ::nn::Result ExportSaveDataPartial_user(std::list<uint64_t> &titleIdList, bool isExistAccountOnly, nn::fs::SaveDataIterator& iter, std::string& hostPathBase)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            #if 0
            NN_LOG("save : %d\n       %016llx\n       %016llx\n       %x\n       %x\n       %016llx %016llx\n",
                                count, info.saveDataId, info.applicationId.value,
                                info.saveDataSpaceId, info.saveDataType,
                                info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
            #endif
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupTitileId(info.applicationId.value))
            {
                continue;
            }
            if(!CheckTitleIdFromList(titleIdList, info.applicationId.value))
            {
                continue;
            }
            if(isExistAccountOnly && !CheckExistAccount(info.saveDataUserId))
            {
                continue;
            }
            std::string hostPath = hostPathBase + GetAppIdString(info.applicationId.value);
            NN_LOG("hostPath : %s\n", hostPath.c_str());
            if(!IsExistFile(hostPath))
            {
                RETURN_IF_FAILED(nn::fs::CreateDirectory(hostPath.c_str()));
            }
            RETURN_IF_FAILED(ExportSaveData(info));
        }

        return nn::ResultSuccess();

    }
}

static ::nn::Result ExportSaveDataAll_user(bool isExistAccountOnly = false)
{
    nn::Result result;

    TitleSaveDataNumMap.clear();

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    std::unique_ptr<nn::fs::SaveDataIterator> itersd;

    RETURN_IF_FAILED(nn::fs::MountHost("host", mountHostPath.c_str()));
    std::string hostPathBase = exportPath + userExportDirName;
    QUIT_IF_FAILED(nn::fs::CreateDirectory(hostPathBase.c_str()));

    QUIT_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
    QUIT_IF_FAILED(ExportSaveDataAll_user(isExistAccountOnly, *iter, hostPathBase));

    {
        // MEMO: SD 非挿入時はデフォルトではアボートが発生するため無効にした (SIGLO-78145)
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&itersd, nn::fs::SaveDataSpaceId::SdUser).IsSuccess())
        {
            QUIT_IF_FAILED(ExportSaveDataAll_user(isExistAccountOnly, *itersd, hostPathBase));
        }
    }

    result = nn::ResultSuccess();

quit:
    nn::fs::Unmount("host");

    return result;
}

static ::nn::Result ExportSaveDataPartial_user(std::list<uint64_t> &titleIdList, bool isExistAccountOnly = false)
{
    nn::Result result;

    RETURN_IF_FAILED(CheckUserSaveDataExists(titleIdList));

    TitleSaveDataNumMap.clear();

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    std::unique_ptr<nn::fs::SaveDataIterator> itersd;

    RETURN_IF_FAILED(nn::fs::MountHost("host", mountHostPath.c_str()));
    std::string hostPathBase = exportPath + userExportDirName;
    QUIT_IF_FAILED(nn::fs::CreateDirectory(hostPathBase.c_str()));

    QUIT_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
    QUIT_IF_FAILED(ExportSaveDataPartial_user(titleIdList, isExistAccountOnly, *iter, hostPathBase));

    {
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&itersd, nn::fs::SaveDataSpaceId::SdUser).IsSuccess())
        {
        QUIT_IF_FAILED(ExportSaveDataPartial_user(titleIdList, isExistAccountOnly, *itersd, hostPathBase));
        }
    }

    result = nn::ResultSuccess();

quit:
    nn::fs::Unmount("host");

    return result;
}

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
namespace
{
    ::nn::Result ExportSaveDataAll_system(nn::fs::SaveDataIterator& iter, std::string& hostPathBase)
    {
        nn::Result result;

        for(;;)
        {
            nn::fs::SaveDataInfo info;
            int64_t count;
            RETURN_IF_FAILED(iter.ReadSaveDataInfo(&count, &info, 1));
            #if 0
            NN_LOG("save : %d\n       %016llx\n       %016llx\n       %x\n       %x\n       %016llx %016llx\n",
                                count, info.saveDataId, info.systemSaveDataId,
                                info.saveDataSpaceId, info.saveDataType,
                                info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
            #endif
            if (count == 0)
            {
                break;
            }
            if(!CheckBackupSystemSaveDataId(info.systemSaveDataId))
            {
                continue;
            }
            #if 0
            if(info.saveDataType != nn::fs::SaveDataType::System)
            {
                NN_LOG("warn: saveDataType is not System (%d)\n", (int)info.saveDataType);
            }
            #endif
            std::string hostPath = hostPathBase + GetSystemSaveDataIdString(info.systemSaveDataId);
            NN_LOG("hostPath : %s\n", hostPath.c_str());
            if(IsExistFile(hostPath))
            {
                RETURN_IF_FAILED(nn::fs::CreateDirectory(hostPath.c_str()));
            }
            RETURN_IF_FAILED(ExportSaveDataSystem(info));
        }

        return nn::ResultSuccess();
    }
}

static ::nn::Result ExportSaveDataAll_system()
{
    nn::Result result;

    SystemSaveDataNumMap.clear();

    std::unique_ptr<nn::fs::SaveDataIterator> iter;

    RETURN_IF_FAILED(nn::fs::MountHost("host", mountHostPath.c_str()));
    std::string hostPathBase = exportPath + systemExportDirName;
    QUIT_IF_FAILED(nn::fs::CreateDirectory(hostPathBase.c_str()));

    QUIT_IF_FAILED(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::System));
    QUIT_IF_FAILED(ExportSaveDataAll_system(*iter, hostPathBase));

    {
        nn::fs::ScopedAutoAbortDisabler fsDisableAbort;
        if(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::SdSystem).IsSuccess())
        {
            QUIT_IF_FAILED(ExportSaveDataAll_system(*iter, hostPathBase));
        }
    }

    result = nn::ResultSuccess();
quit:
    nn::fs::Unmount("host");

    return result;
}
#endif  // defined(NN_BACKUPSAVEDATA_SYSTEM)

static ::nn::Result ParseAccountFromConfigFiles(std::string baseDirectoryPath)
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string configFileDirectory = baseDirectoryPath + userExportDirName;
    int64_t entryNum = 0;
    int64_t readNum = 0;

    result = nn::fs::OpenDirectory(&dHandle, configFileDirectory.c_str(), nn::fs::OpenDirectoryMode_Directory);
    if(result.IsFailure())
    {
        if(nn::fs::ResultPathNotFound().Includes(result))
        {
            std::vector<nn::fs::UserId> emptyVector;
            SetBackuppedAccountVector(emptyVector);
            return nn::ResultSuccess();
        }
        else
        {
            return result;
        }
    }

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        // MEMO: titleId のチェックは不要とした
        uint64_t titleId = std::stoull(dEntryArea[i].name, nullptr, 16);
        if(CheckBackupTitileId(titleId))
        {
            RETURN_IF_FAILED(ParseAccountFromConfigFiles(baseDirectoryPath, titleId));
        }
    }

    result = nn::ResultSuccess();

    return result;
}

static ::nn::Result ParseAccountFromConfigFiles(std::string baseDirectoryPath, uint64_t titleId)
{
    nn::Result result;
    nn::fs::DirectoryHandle dHandle;
    nn::fs::DirectoryEntry* dEntryArea;
    std::string configTitleDirectory = baseDirectoryPath + userExportDirName + GetAppIdString(titleId);
    int64_t entryNum = 0;
    int64_t readNum = 0;

    RETURN_IF_FAILED(nn::fs::OpenDirectory(&dHandle, configTitleDirectory.c_str(), nn::fs::OpenDirectoryMode_Directory));

    RETURN_IF_FAILED(nn::fs::GetDirectoryEntryCount(&entryNum, dHandle));
    SET_AND_CHECK_ALLCATION(dEntryArea, reinterpret_cast<nn::fs::DirectoryEntry*>(SaveDataHeapAllocate(sizeof(nn::fs::DirectoryEntry) * (entryNum + 1))));
    RETURN_IF_FAILED(nn::fs::ReadDirectory(&readNum, dEntryArea, dHandle, entryNum + 1));
    nn::fs::CloseDirectory(dHandle);
    for(int i=0; i<readNum; i++)
    {
        std::string hostPath = configTitleDirectory + "/" + dEntryArea[i].name;
        uint64_t saveDataUserIdH, saveDataUserIdL, applicationId_dummy, saveDataSize;
        nn::Bit64 ownerId;
        uint32_t saveDataFlags;
        int64_t saveDataAvailableSize, saveDataJournalSize;
        std::string saveDataTypeString;
        uint32_t saveDataSpaceId;
        int32_t cacheIndex;
        RETURN_IF_FAILED(ParseSaveDataInfoFile(hostPath, saveDataTypeString, saveDataUserIdH, saveDataUserIdL, applicationId_dummy, saveDataSize, ownerId, saveDataFlags,
                                                (uint64_t&)saveDataAvailableSize, (uint64_t&)saveDataJournalSize, saveDataSpaceId, cacheIndex));
        nn::fs::UserId userIdFromInfoFile = {{saveDataUserIdH, saveDataUserIdL}};
        nn::fs::SaveDataType saveDataType = GetSaveDataType(saveDataTypeString);
        if(saveDataType != nn::fs::SaveDataType::Account)
        {
            continue;
        }
        SetBackuppedAccount(userIdFromInfoFile);
    }

    return result;
}

::nn::Result ExportSaveDataAll(bool isExistAccountOnly)
{
    nn::Result result;

    RETURN_IF_FAILED(GetAccountVectorFromDevkit());

    RETURN_IF_FAILED(CreateExportDirectory());

    RETURN_IF_FAILED(ExportSaveDataAll_user(isExistAccountOnly));
#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    RETURN_IF_FAILED(ExportSaveDataAll_system());
#endif
    return nn::ResultSuccess();
}

::nn::Result ExportSaveDataPartial(std::string titleIds, bool isExistAccountOnly)
{
    nn::Result result;

    RETURN_IF_FAILED(GetAccountVectorFromDevkit());

    // parse title Ids
    std::list<uint64_t> titleIdList;
    RETURN_IF_FAILED(ConvertTitileIdsStringToList(&titleIdList, titleIds))

    RETURN_IF_FAILED(CreateExportDirectory(true));

    RETURN_IF_FAILED(ExportSaveDataPartial_user(titleIdList, isExistAccountOnly));
#if defined(NN_BACKUPSAVEDATA_SYSTEM)
    // MEMO: System save data is not support partial.
#endif
    return nn::ResultSuccess();
}

::nn::Result ImportSaveDataAll(bool isReplaceAccount)
{
    nn::Result result;

    RETURN_IF_FAILED(GetAccountVectorFromDevkit());

    RETURN_IF_FAILED(nn::fs::MountHost("host", mountHostPath.c_str()));

    std::string backupType;
    QUIT_IF_FAILED(ParseConfigFile(exportPath, backupType));

    if(isReplaceAccount)
    {
        result = ParseAccountInfoFile(exportPath);
        if(ResultNotFound().Includes(result))
        {
            NN_LOG("[INFO] account.xml is not found, so parse accounts from export files.\n");
            QUIT_IF_FAILED(ParseAccountFromConfigFiles(exportPath));
        }
        if(!AccountNumCheck())
        {
            result = ResultApplicationError();
            goto quit;
        }
        DisplayConvertAccountInfo();
    }

    if(backupType == "Default")
    {
        QUIT_IF_FAILED(DeleteAllTitleSaveData());
        QUIT_IF_FAILED(ImportAllTitleSaveData(isReplaceAccount));

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
        QUIT_IF_FAILED(DeleteAllSaveData_system());
        QUIT_IF_FAILED(ImportAllSaveData_system());
#endif
    }
    else if(backupType == BackupTypeStringPartial)
    {
        NN_LOG("[INFO] This export data is Partial type.\n");

        QUIT_IF_FAILED(DeleteAndImportPartialTitleSaveData(isReplaceAccount));

#if defined(NN_BACKUPSAVEDATA_SYSTEM)
        // MEMO: System save data is not support partial.
#endif
    }

    result = nn::ResultSuccess();
quit:
    nn::fs::Unmount("host");

    return result;
}
