﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include "Microphone.h"

Microphone::Microphone()
    : m_Allocator(nullptr)
    , m_BufferEvent()
    , m_BufferLengthMilliseconds(0)
    , m_AudioInInfo()
    , m_AudioIn()
    , m_IsInitialized(false)
    , m_IsStarted(false)
    , m_IsOpen(false)
    , m_IsGainSupported(true)
    , m_DeviceGain(0.0f)
{}

void Microphone::Initialize(int bufferLengthMilliseconds, nn::mem::StandardAllocator* allocator) NN_NOEXCEPT
{
    NN_ASSERT(!m_IsInitialized && !IsOpen() && !IsStarted());
    NN_ASSERT(allocator != nullptr);
    m_BufferLengthMilliseconds = bufferLengthMilliseconds;
    m_Allocator = allocator;
    m_IsInitialized = true;
    m_IsGainSupported = true;

    if(m_BufferLengthMilliseconds > MaxBufferLengthMilliseconds)
    {
        NN_LOG("Clamping m_BufferLengthMilliseconds(%d) to MaxBufferLengthMilliseconds(%d)\n",
                m_BufferLengthMilliseconds, MaxBufferLengthMilliseconds);
        m_BufferLengthMilliseconds = MaxBufferLengthMilliseconds;
    }
    else if (m_BufferLengthMilliseconds < MinBufferLengthMilliseconds)
    {
        NN_LOG("Clamping m_BufferLengthMilliseconds(%d) to MinBufferLengthMilliseconds(%d)\n",
                m_BufferLengthMilliseconds, MinBufferLengthMilliseconds);
        m_BufferLengthMilliseconds = MinBufferLengthMilliseconds;
    }
}

void Microphone::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && !IsOpen() && !IsStarted());
    m_IsInitialized = false;
}

bool Microphone::Open(nn::audio::AudioInInfo info) NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && !IsOpen() && !IsStarted());
    nn::audio::AudioInParameter audioInParameter;
    nn::audio::InitializeAudioInParameter(&audioInParameter);

    m_AudioInInfo = info;
    NN_LOG("Trying to start %s\n", info.name);

    auto result = nn::audio::OpenAudioIn(&m_AudioIn, &m_BufferEvent, info.name, audioInParameter);
    if(result.IsSuccess())
    {
        std::strncpy(m_AudioInInfo.name, info.name, nn::audio::AudioIn::NameLength);
        if(result.IsSuccess())
        {
            int channelCount = nn::audio::GetAudioInChannelCount(&m_AudioIn);
            int sampleRate = nn::audio::GetAudioInSampleRate(&m_AudioIn);
            nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioInSampleFormat(&m_AudioIn);

            int frameSampleCount = m_BufferLengthMilliseconds * (sampleRate / 1000);
            size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
            size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioInBuffer::SizeGranularity);
            int allocationCount = 0;
            for (int i = 0; i < NumBuffers; ++i)
            {
                void* buffer = m_Allocator->Allocate(bufferSize, nn::audio::AudioInBuffer::AddressAlignment);
                if(!buffer)
                {
                    NN_LOG("Failed to allocate buffer\n");
                    break;
                }
                ++allocationCount;
                nn::audio::SetAudioInBufferInfo(&m_BufferInfo[i], buffer, bufferSize, dataSize);
            }
            if(allocationCount != NumBuffers)
            {
                for(int i = 0; i < allocationCount; ++i)
                {
                    m_Allocator->Free(m_BufferInfo[i].buffer);
                }
                return false;
            }
            m_IsOpen = true;
            m_DeviceGain = nn::audio::GetAudioInDeviceGain(&m_AudioIn);
            return true;
        }
        else
        {
            nn::audio::CloseAudioIn(&m_AudioIn);
            NN_LOG("%s already disconnected\n", info.name);
            return false;
        }
    }
    else
    {
        NN_LOG("Failed to Open\n");
        return false;
    }
}

void Microphone::Close() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen() && !IsStarted());
    nn::audio::CloseAudioIn(&m_AudioIn);
    nn::os::DestroySystemEvent(m_BufferEvent.GetBase());
    for(int i = 0; i < NumBuffers; ++i)
    {
        m_Allocator->Free(m_BufferInfo[i].buffer);
    }
    m_IsOpen = false;
}

bool Microphone::Start() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen() && !IsStarted());
    for (int i = 0; i < NumBuffers; ++i)
    {
        nn::audio::AppendAudioInBuffer(&m_AudioIn, &m_BufferInfo[i]);
    }

    auto result = nn::audio::StartAudioIn(&m_AudioIn);
    m_IsStarted = result.IsSuccess();

    if(m_IsStarted)
    {
        result = nn::audio::SetAudioInDeviceGain(&m_AudioIn, m_DeviceGain);
        m_IsGainSupported = result.IsSuccess();
    }

    return m_IsStarted;
}

void Microphone::Stop() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen() && IsStarted());
    nn::audio::StopAudioIn(&m_AudioIn);
    m_IsStarted = false;
}

void Microphone::Update(BufferCallback callback) NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen() && IsStarted());
    nn::audio::AudioInBuffer* pAudioInBuffer = nullptr;

    pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&m_AudioIn);

    while(pAudioInBuffer)
    {
        // データをコピーし、再度登録します。
        void* pInBuffer = nn::audio::GetAudioInBufferDataPointer(pAudioInBuffer);
        size_t inSize = nn::audio::GetAudioInBufferDataSize(pAudioInBuffer);

        if(callback)
        {
            callback(pInBuffer, inSize);
        }

        memset(pInBuffer, 0, inSize);
        nn::audio::AppendAudioInBuffer(&m_AudioIn, pAudioInBuffer);
        pAudioInBuffer = nn::audio::GetReleasedAudioInBuffer(&m_AudioIn);
    }
}

bool Microphone::IsOpen() NN_NOEXCEPT
{
    return m_IsOpen;
}

bool Microphone::IsStarted() NN_NOEXCEPT
{
    return m_IsStarted;
}

nn::os::SystemEventType* Microphone::GetBufferEvent() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen());
    return m_BufferEvent.GetBase();
}

const char* Microphone::GetName() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized && IsOpen());
    return m_AudioInInfo.name;
}

const nn::audio::AudioInInfo& Microphone::GetInfo() NN_NOEXCEPT
{
    return m_AudioInInfo;
}

float Microphone::GetDeviceGain() NN_NOEXCEPT
{
    return m_DeviceGain;
}

void Microphone::SetDeviceGain(float gain) NN_NOEXCEPT
{
    // Clamp gain by [nn::audio::AudioInDeviceGainMin, std::min(nn::audio::AudioInDeviceGainMax].
    gain = std::max(nn::audio::AudioInDeviceGainMin, std::min(nn::audio::AudioInDeviceGainMax, gain));

    if(m_IsGainSupported)
    {
        auto result = nn::audio::SetAudioInDeviceGain(&m_AudioIn, gain);
        if(result.IsSuccess())
        {
            m_DeviceGain = gain;
        }
    }
}

bool Microphone::IsDeviceGainSupported() NN_NOEXCEPT
{
    return m_IsGainSupported;
}
