﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <mutex>
#include <vector>

#include <nn/audio.h>
#include <nn/audio/audio_AudioInApi.private.h>
#include <nn/audio/audio_AudioOutTypes.private.h>
#include <nn/audio/audio_AudioOutApi.private.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nn_Log.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/web.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "GraphicSystem.h"
#include "Microphone.h"
#include "ScopedDummyAudioOut.h"
#include "ScopedDummyAudioRenderer.h"


struct MicData
{
    MicData()
    : appendEvent(nn::os::EventClearMode_AutoClear)
    , mutex(false)
    , frameSize(0)
    , readIndex(0)
    , writeIndex(0)
    , readyFrames(0)
    , numFrames(0)
    , rmsIndex(0)
    {
        memset(rms, 0, sizeof(rms) / sizeof(rms[0]));
    }

    void Reset()
    {
        std::lock_guard<nn::os::Mutex> lock(mutex);
        readyFrames = readIndex = writeIndex = 0;
        count = 0;
    }


    void SetFrameSize(size_t _frameSize)
    {
        frameSize = _frameSize;
        numFrames = static_cast<int>(sizeof(buffer) / frameSize);
    }

    void PutFrame(void* data, size_t size)
    {
        NN_UNUSED(size);
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            memcpy(&buffer[writeIndex * frameSize], data, frameSize);
            writeIndex = (writeIndex + 1) % numFrames;
            ++readyFrames;


            auto p = static_cast<int16_t*>(data);
            auto sampleCount = static_cast<int>(frameSize / sizeof(int16_t));
            float sum = 0;
            for (auto i = 0; i < sampleCount; ++i)
            {
                auto sample = p[i] / 32768.0f;
                sum += sample * sample;
            }
            sum /= sampleCount;
            rms[rmsIndex] = std::sqrtf(sum);
            if (++rmsIndex >= RmsCountMax)
            {
                rmsIndex = 0;
            }
        }
        appendEvent.Signal();
    }

    void GetFrame(void* data, size_t size)
    {
        NN_UNUSED(size);
        if(readyFrames == 0)
        {
            appendEvent.Wait();
        }
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            memcpy(data, &buffer[readIndex * frameSize], frameSize);
            memset(&buffer[readIndex * frameSize], 0, frameSize);
            readIndex = (readIndex + 1) % numFrames;
            --readyFrames;
        }
    }

    void GetFrameAdd(void* data, size_t size, int32_t volume)
    {
        while (inUse && readyFrames <= 0)
        {
            appendEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(5));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(mutex);
            int16_t* in = reinterpret_cast<int16_t*>(&buffer[readIndex * frameSize]);
            int16_t* out = reinterpret_cast<int16_t*>(data);
            const auto Q = 15;
            for(size_t i = 0; i < size / 2; ++i)
            {
                out[i] += (in[i] * volume) >> Q;
            }
            readIndex = (readIndex + 1) % numFrames;
            --readyFrames;
        }
    }

    nn::os::Event appendEvent;
    nn::os::Mutex mutex;
    size_t frameSize;
    int readIndex;
    int writeIndex;
    int readyFrames;
    int numFrames;
    bool inUse;
    int count;
    char buffer [4096 * 8];
    static const int RmsCountMax = 16;
    float rms[RmsCountMax];
    int rmsIndex;
};

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const int FrameRate = 60;
const auto FrameWait = nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate);

const char* DocumentPathForOfflineWebApplet = "sample.htdocs/index.html";

const int MaxAudioIns = 2;
char g_HeapBuffer[128 * 1024 * 16];
MicData g_MicData[MaxAudioIns];
const size_t ThreadStackSize = 8192 * 4;
NN_ALIGNAS(4096) char g_AudioOutThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioOutThread;
NN_ALIGNAS(4096) char g_AudioInThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioInThread;
Microphone g_Mics[MaxAudioIns];


}  // namespace

struct AudioOutArgs
{
    nn::os::SystemEvent* bufferEvent;
    nn::audio::AudioOut* audioOut;
    bool isRunning;
};
AudioOutArgs g_OutArgs;

void AudioOutThread(void* arg)
{
    AudioOutArgs* args = reinterpret_cast<AudioOutArgs*>(arg);
    while(args->isRunning)
    {
        // 録音が完了したバッファを取得します。
        args->bufferEvent->Wait();
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);

        int bufferCount = 0;
        while(pAudioOutBuffer)
        {
            ++bufferCount;
            // データをコピーし、再度登録します。
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);

            memset(pOutBuffer, 0, outSize);
            for(int i = 0; i < MaxAudioIns; ++i)
            {
                if (g_MicData[i].inUse)
                {
                    g_MicData[i].GetFrameAdd(pOutBuffer, outSize, std::numeric_limits<int16_t>::max() / MaxAudioIns);
                }
            }

            nn::audio::AppendAudioOutBuffer(args->audioOut, pAudioOutBuffer);
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);
        }
        if (bufferCount != 1)
        {
           //NN_LOG("out bufferCount: %d\n", bufferCount);
        }
    }
}

struct AudioInArgs
{
    nn::mem::StandardAllocator* allocator;
    int bufferLengthInMilliseconds;
    bool isRunning;
    Microphone* mics;
};

enum EventType
{
    EventType_Invalid,
    EventType_Connect,
    EventType_Disconnect,
    EventType_Buffer,
};

class EventData
{
public:
    EventData()
    : m_Event(nullptr)
    , m_Index(-1)
    , m_Type(0)
    , m_InUse(false)
    {}

    void Acquire(int audioIndex, int eventType, nn::os::SystemEventType* event)
    {
        NN_ASSERT(!m_InUse);
        m_Index = audioIndex;
        m_Type = eventType;
        m_Event = event;
        m_InUse = true;
    }

    void Release()
    {
        NN_ASSERT(m_InUse);
        m_InUse = false;
        m_Index = -1;
        m_Event = nullptr;
        m_Type = EventType_Invalid;
    }

    int GetIndex()
    {
        return m_Index;
    }

    int GetType()
    {
        return m_Type;
    }

    bool IsInUse()
    {
        return m_InUse;
    }

    void ClearEvent()
    {
        if(m_Event)
        {
            nn::os::ClearSystemEvent(m_Event);
        }
    }
private:
    nn::os::SystemEventType* m_Event;
    int m_Index;
    int m_Type;
    bool m_InUse;
};

AudioInArgs g_InArgs;

bool IsMicrophoneUsed(Microphone* mics, int numMics, const char* name)
{
    for(int i = 0; i < numMics; ++i)
    {
        if(mics[i].IsOpen() && strcmp(name, mics[i].GetName()) == 0)
        {
            return true;
        }
    }
    return false;
}

bool IsMicrophoneConnected(Microphone* mic, nn::audio::AudioInInfo* infos, int infoCount)
{
    for(int i = 0; i < infoCount; ++i)
    {
        auto& info = infos[i];
        if(mic->IsOpen() && strcmp(info.name, mic->GetName()) == 0)
        {
            return true;
        }
    }
    return false;
}

struct MultiWaitInfo
{
public:
    void Link(nn::os::SystemEventType* bufferEvent, int micIndex)
    {
        int bufferEventIndex = (2 * micIndex) + 1;
        eventData[bufferEventIndex].Acquire(micIndex, EventType_Buffer, bufferEvent);
        nn::os::InitializeMultiWaitHolder(&bufferReadyHolder[micIndex], bufferEvent);
        nn::os::SetMultiWaitHolderUserData(&bufferReadyHolder[micIndex], static_cast<uintptr_t>(bufferEventIndex));
        nn::os::LinkMultiWaitHolder(multiWait, &bufferReadyHolder[micIndex]);
    }

    void Unlink(int micIndex)
    {
        int bufferEventIndex = (2 * micIndex) + 1;
        eventData[bufferEventIndex].Release();
        nn::os::UnlinkMultiWaitHolder(&bufferReadyHolder[micIndex]);
        nn::os::FinalizeMultiWaitHolder(&bufferReadyHolder[micIndex]);
    }

    EventData* eventData;
    nn::os::MultiWaitType* multiWait;
    nn::os::MultiWaitHolderType* bufferReadyHolder;
};

void HandleConnect(Microphone* mics, int numMics, MultiWaitInfo* multiWaitInfo, nn::audio::AudioInInfo* audioInInfos, int count)
{
    for(int i = 0; i < count; ++i)
    {
        auto& info = audioInInfos[i];
        if(strcmp(info.name, "BuiltInHeadset") == 0)
        {
            continue;
        }
        if(!IsMicrophoneUsed(mics, numMics, info.name))
        {
            for(int j = 0; j < numMics; ++j)
            {
                if(!mics[j].IsOpen())
                {
                    auto open = mics[j].Open(info);
                    if(open)
                    {
                        multiWaitInfo->Link(mics[j].GetBufferEvent(), j);
                        mics[j].Start();
                        g_MicData[j].inUse = true;
                        g_MicData[j].Reset();
                        //TODO: Write this to screen
                        NN_LOG("Starting %s\n", mics[j].GetName());
                    }
                    break;
                }
            }
        }
    }
}

void HandleDisconnect(Microphone* mics, int index, MultiWaitInfo* multiWaitInfo)
{
    auto& mic = mics[index];
    if(mics[index].IsOpen())
    {
        auto info = mic.GetInfo();
        //TODO: Write this to screen
        NN_LOG("Stopping %s\n", info.name);
        g_MicData[index].inUse = false;
        mic.Stop();
        mic.Close();
        multiWaitInfo->Unlink(index);
        ::std::memset(g_MicData[index].rms, 0, sizeof(g_MicData[index].rms));
        g_MicData[index].rmsIndex = 0;
    }
}

void HandleUpdate(Microphone* mics, int index)
{
    auto& mic = mics[index];

    auto callback = [=] (void* buffer, size_t bufferSize)
    {
        auto& micData = g_MicData[index];
        micData.PutFrame(buffer, bufferSize);
    };

    mic.Update(callback);
}

void CheckConnections(Microphone* mics, int numMics, MultiWaitInfo* multiWaitInfo)
{
    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);
    for(int i = 0; i < numMics; ++i)
    {
        if(mics[i].IsStarted())
        {
            bool nameFound = false;
            auto name = mics[i].GetName();
            for(int j = 0; j < count; ++j)
            {
                if(strcmp(name, audioInInfos[j].name) == 0)
                {
                    nameFound = true;
                    break;
                }
            }
            //Disconnect any microphones that haven't received data for a while
            if(!nameFound)
            {
                HandleDisconnect(mics, i, multiWaitInfo);
            }
        }
    }

    //Try to connect to any new microphones
    HandleConnect(mics, numMics, multiWaitInfo, audioInInfos, count);
}

void AudioInThread(void* arg)
{
    AudioInArgs* args = reinterpret_cast<AudioInArgs*>(arg);

    nn::os::SystemEvent connectEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     bufferReadyHolder[MaxAudioIns];
    nn::os::MultiWaitHolderType     connectHolder;

    EventData eventData[MaxAudioIns * 2 + 1];

    MultiWaitInfo multiWaitInfo;
    multiWaitInfo.eventData = eventData;
    multiWaitInfo.multiWait = &multiWait;
    multiWaitInfo.bufferReadyHolder = bufferReadyHolder;

    nn::os::InitializeMultiWait(&multiWait);

    nn::audio::AcquireAudioDeviceNotificationForInput(&connectEvent);

    nn::os::InitializeMultiWaitHolder(&connectHolder, connectEvent.GetBase());
    eventData[0].Acquire(0, EventType_Connect, connectEvent.GetBase());
    nn::os::SetMultiWaitHolderUserData(&connectHolder, static_cast<uintptr_t>(0));
    nn::os::LinkMultiWaitHolder(&multiWait, &connectHolder);

    const auto mics = args->mics;
    for(int i = 0; i < MaxAudioIns; ++i)
    {
        mics[i].Initialize(args->bufferLengthInMilliseconds, args->allocator);
    }

    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);
    HandleConnect(mics, MaxAudioIns, &multiWaitInfo, audioInInfos, count);

    while(args->isRunning)
    {
        auto holder = nn::os::WaitAny(&multiWait);

        auto dataIndex = nn::os::GetMultiWaitHolderUserData(holder);
        auto data = eventData[dataIndex];
        data.ClearEvent();

        switch(data.GetType())
        {
            case EventType_Connect:
                CheckConnections(mics, MaxAudioIns, &multiWaitInfo);
                break;
            case EventType_Buffer:
                HandleUpdate(mics, data.GetIndex());
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    for(int i = 0; i < MaxAudioIns; ++i)
    {
        HandleDisconnect(mics, i, &multiWaitInfo);
    }

    nn::os::UnlinkMultiWaitHolder(&connectHolder);
    nn::os::FinalizeMultiWaitHolder(&connectHolder);
    nn::os::FinalizeMultiWait(&multiWait);
    nn::os::DestroySystemEvent(connectEvent.GetBase());
}

void UpdateFrame(GraphicsSystem* pGraphicsSystem)
{
    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();

    const nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    const nn::util::Color4u8Type color1 = { { 255, 128, 0, 255 } };

    pTextWriter->SetScale(2.0f, 2.0f);

    pTextWriter->SetTextColor(color0);
    pTextWriter->SetCursor(0, 0);
    pTextWriter->Print("USB マイクテスト / USB Microphones Test\n");
    pTextWriter->Print("\n");
    pTextWriter->Print("L ボタン: デバイスゲインダウン / L button: device gain down\n");
    pTextWriter->Print("R ボタン: デバイスゲインアップ / R button: device gain up\n");
    pTextWriter->Print("A ボタン: OfflineWebApplet 呼び出し / A button: show OfflineWebApplet\n");
    pTextWriter->Print("\n");
    for (auto i = 0; i < 2; ++i)
    {
        pTextWriter->SetTextColor(color0);
        if (g_MicData[i].inUse)
        {
            pTextWriter->SetTextColor(color1);
        }
        pTextWriter->Print("マイク %d / Mic %d : %s\n", i, i, g_MicData[i].inUse ? "接続済 / Connected" : "未接続 / Disconnected");
        std::string level = "レベル / Level : ";
        float max = 0;
        for (auto j = 0; j < g_MicData[i].RmsCountMax; ++j)
        {
            max = std::max(max, g_MicData[i].rms[j]);
        }
        int count = static_cast<int>((max) * 20);
        for (auto j = 0; j < count; ++j)
        {
            level.append("■");
        }
        pTextWriter->Print("%s\n", level.c_str());
        const auto deviceGain = g_Mics[i].GetDeviceGain();
        if(!g_Mics[i].IsDeviceGainSupported())
        {
            pTextWriter->SetTextColor(color0);
            pTextWriter->Print("デバイスゲイン %.2f (非サポート) / DeviceGain %.2f *Not Supported*\n", deviceGain, deviceGain);
        }
        else
        {
            pTextWriter->Print("デバイスゲイン %.2f / DeviceGain %.2f\n", deviceGain, deviceGain);
        }

        pTextWriter->Print("\n");
    }

    pGraphicsSystem->BeginDraw();
    pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());

    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(FrameWait);
}

//
// メイン関数です。
//
extern "C" void nnMain()
{
    nn::oe::SetExpectedVolumeBalance(0.6f, 0.4f);

    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    // 多数の AudioOut をオープンしている状態を実現するために LibraryApplet 向け API を転用
    const auto audioOutCountMax = nn::audio::AudioOutCountMaxForLibraryApplet;
    nn::audio::SetAudioOutCountMaxForLibraryApplet(audioOutCountMax);
    ScopedDummyAudioOut scopedDummyAudioOuts[audioOutCountMax - 1];
    ScopedDummyAudioRenderer scopedDummyAudioRenderers[nn::audio::AudioRendererCountMax] = { allocator, allocator };

    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    g_InArgs.allocator = &allocator;

    nn::audio::SetAudioInEnabled(true);
    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::AudioOut audioOut;

    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
    int bufferLengthInMilliseconds = 5;
    if(argc >= 2)
    {
        bufferLengthInMilliseconds = atoi(argv[1]);
    }
    if(bufferLengthInMilliseconds > 50 || bufferLengthInMilliseconds < 0)
    {
        NN_LOG("Clamping bufferLengthInMilliseconds from %d to 50\n", bufferLengthInMilliseconds);
        bufferLengthInMilliseconds = 50;
    }

    g_InArgs.bufferLengthInMilliseconds = bufferLengthInMilliseconds;
    nn::os::SystemEvent audioInConectEvent;
    nn::os::SystemEvent audioOutBufferEvent;
    // オーディオ入力、オーディオ出力をオープンします。

    nn::audio::InitializeAudioOutParameter(&audioOutParameter);

    NN_ABORT_UNLESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, audioOutParameter).IsSuccess(),
        "Failed to open AudioOut."
    );

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_LOG("channelCount: %d, sampleRate: %d, sampleFormat: %d\n", channelCount, sampleRate, sampleFormat);
    // バッファに関するパラメータを準備します。
    NN_LOG("bufferLengthInMilliseconds = %d\n", bufferLengthInMilliseconds);
    int millisecondsPerSecond = 1000;
    int frameRate = millisecondsPerSecond / bufferLengthInMilliseconds;
    int frameSampleCount = sampleRate / frameRate;
    size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioInBuffer::SizeGranularity);
    const int outBufferCount = 2;

    for(int i = 0; i < MaxAudioIns; ++i)
    {
        g_MicData[i].SetFrameSize(dataSize);
    }

    nn::audio::AudioOutBuffer audioOutBuffer[outBufferCount];
    void* outBuffer[outBufferCount];

    for (int i = 0; i < outBufferCount; ++i)
        {
            outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
            NN_ASSERT(outBuffer[i]);
            std::memset(outBuffer[i], 0, bufferSize);
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        }

    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
    );

    // 1 フレーム待ちます。
    const nn::TimeSpan interval(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / frameRate));
    nn::os::SleepThread(interval);

    g_OutArgs.bufferEvent = &audioOutBufferEvent;
    g_OutArgs.audioOut = &audioOut;
    g_OutArgs.isRunning = true;

    g_InArgs.isRunning = true;
    g_InArgs.mics = g_Mics;
    nn::os::CreateThread(&g_AudioInThread, AudioInThread, &g_InArgs, g_AudioInThreadStack,
                ThreadStackSize, nn::os::DefaultThreadPriority);


    nn::os::CreateThread(&g_AudioOutThread, AudioOutThread, &g_OutArgs, g_AudioOutThreadStack,
                    ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&g_AudioInThread);
    nn::os::StartThread(&g_AudioOutThread);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

    while (true)
    {
        UpdateFrame(pGraphicsSystem);

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
        }

        // Show OfflineWebApplet (A)
        if (npadButtonCurrent.Test< ::nn::hid::NpadButton::A >())
        {
            nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(DocumentPathForOfflineWebApplet);
            showOfflinePageArg.SetWebAudioEnabled(true);
            nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;
            // 表示中に強制的にキャンセルされた場合に nn::web::ResultTerminated が返るが、今回はハンドリングする必要はないので無視
            nn::Result result = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
            NN_UNUSED(result);
        }


        // Change DeviceGain (L/R)
        if( npadButtonCurrent.Test< ::nn::hid::NpadButton::L >() || npadButtonCurrent.Test< ::nn::hid::NpadButton::R >() )
        {
            for (auto& mic : g_Mics)
            {
                if (mic.IsStarted())
                {
                    const auto deviceGainStep = 0.05f;
                    const auto currentGain = mic.GetDeviceGain();

                    if (npadButtonCurrent.Test< ::nn::hid::NpadButton::L >())
                    {
                        mic.SetDeviceGain(currentGain - deviceGainStep);
                    }

                    if (npadButtonCurrent.Test< ::nn::hid::NpadButton::R >())
                    {
                        mic.SetDeviceGain(currentGain + deviceGainStep);
                    }
                }
            }
        }
    }

    nn::os::WaitThread(&g_AudioInThread);
    nn::os::WaitThread(&g_AudioOutThread);

    nn::os::DestroyThread(&g_AudioInThread);
    nn::os::DestroyThread(&g_AudioOutThread);

    NN_LOG("Stop audio echoback\n");

    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);

    nn::os::DestroySystemEvent(audioOutBufferEvent.GetBase());
    nn::os::DestroySystemEvent(audioInConectEvent.GetBase());


    for (int i = 0; i < outBufferCount; ++i)
    {
        allocator.Free(outBuffer[i]);
    }
    return;
}  // NOLINT(readability/fn_size)
