﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/audio.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/htcs.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "GraphicSystem.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const int FrameRate = 60;
const auto FrameWait = nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate);

const char PortName[] = "AudioRedirector";

const int InvalidSocket = -1;
const int ChannelCount = 2;
const int SamplingRate = 48000;
const int AudioOutBufferCount = 2;

NN_ALIGNAS(nn::audio::AudioOutBuffer::AddressAlignment) char g_Buffer[AudioOutBufferCount][64 * 1024];

std::string g_Status;

}  // namespace

void UpdateFrame(GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
{
    auto pTextWriter = &pGraphicsSystem->GetDebugFont();

    const nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };

    pTextWriter->SetScale(1.5f, 1.5f);

    pTextWriter->SetTextColor(color0);
    pTextWriter->SetCursor(32, 32);
    pTextWriter->Print("[Audio Redirector]\n\n");
    pTextWriter->Print("%s\n", g_Status.c_str());

    pGraphicsSystem->BeginDraw();
    pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());

    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(FrameWait);
}

void ThreadFunc(void* arg) NN_NOEXCEPT
{
    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    ::nn::util::Strlcpy(
        serviceAddr.portName.name,
        PortName,
        ::std::extent<decltype(serviceAddr.portName.name)>::value);

    g_Status = "waiting for connection";
    while (true)
    {
        int serviceSocket = nn::htcs::Socket();
        if (serviceSocket == InvalidSocket)
        {
            g_Status = "error (invalid socket)";
            continue;
        }

        if (nn::htcs::Bind(serviceSocket, &serviceAddr) != 0)
        {
            g_Status = "error (bind failed)";
            nn::htcs::Close(serviceSocket);
            continue;
        }

        if (nn::htcs::Listen(serviceSocket, 1) != 0)
        {
            g_Status = "error (listen failed)";
            nn::htcs::Close(serviceSocket);
            continue;
        }

        auto socket = nn::htcs::Accept(serviceSocket, nullptr);
        if (socket < 0)
        {
            g_Status = "error (accept failed)";
            nn::htcs::Close(serviceSocket);
            continue;
        }

        nn::audio::AudioOut audioOut;
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        parameter.sampleRate = SamplingRate;
        parameter.channelCount = ChannelCount;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));

        nn::audio::AudioOutBuffer audioOutBuffer[AudioOutBufferCount];
        void* outBuffer[AudioOutBufferCount];
        for (int i = 0; i < AudioOutBufferCount; ++i)
        {
            outBuffer[i] = g_Buffer[i];
            memset(g_Buffer[i], 0, sizeof(g_Buffer[i]));
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], sizeof(g_Buffer[i]), 1 * ChannelCount * sizeof(short));
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        }

        g_Status = "connected, redirecting";

        auto exit = false;
        while (!exit)
        {
            systemEvent.Wait();

            auto pAudioOutBuffer = GetReleasedAudioOutBuffer(&audioOut);
            while(pAudioOutBuffer != nullptr)
            {
                auto buffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                auto size = nn::audio::GetAudioOutBufferBufferSize(pAudioOutBuffer);
                auto received = nn::htcs::Recv(socket, buffer, size, 0);
                if (received <= 0)
                {
                    exit = true;
                    break;
                }

                nn::audio::SetAudioOutBufferInfo(pAudioOutBuffer, buffer, size, received);
                nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);
                pAudioOutBuffer = GetReleasedAudioOutBuffer(&audioOut);
            }
        }

        nn::audio::StopAudioOut(&audioOut);
        nn::audio::CloseAudioOut(&audioOut);
        nn::os::DestroySystemEvent(systemEvent.GetBase());

        nn::htcs::Close(socket);
        nn::htcs::Close(serviceSocket);

        g_Status = "waiting for connection";
    }
}

void InitializeAudioRedirectorServer() NN_NOEXCEPT
{
    static char s_HtcsHeap[4 * 1024];
    NN_SDK_ASSERT(sizeof(s_HtcsHeap) >= nn::htcs::GetWorkingMemorySize(2));
    nn::htcs::Initialize(s_HtcsHeap, sizeof(s_HtcsHeap));

    static nn::os::ThreadType s_Thread = {};
    const size_t StackSize = 4 * nn::os::ThreadStackAlignment;
    NN_OS_ALIGNAS_THREAD_STACK static char s_Stack[StackSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &s_Thread,
            &ThreadFunc,
            nullptr,
            reinterpret_cast<void*>(s_Stack),
            sizeof(s_Stack),
            nn::os::DefaultThreadPriority));
    nn::os::StartThread(&s_Thread);
}

extern "C" void nnMain() NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&applicationHeap);
    pGraphicsSystem->Initialize();

    InitializeAudioRedirectorServer();

    while (true)
    {
        UpdateFrame(pGraphicsSystem);
    }
}
