﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vi.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/mem.h>
#include <nn/nn_Assert.h>

class ApplicationHeap
{
    NN_DISALLOW_COPY(ApplicationHeap);
    NN_DISALLOW_MOVE(ApplicationHeap);

private:
    void* m_pMemoryHeap;
    nn::mem::StandardAllocator m_ApplicationHeapHandle;

public:
    ApplicationHeap() NN_NOEXCEPT;
    ~ApplicationHeap() NN_NOEXCEPT;

    void Initialize(size_t size) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT;
    void* Allocate(size_t size) NN_NOEXCEPT;
    void Deallocate(void* ptr);
};

class GraphicsSystem
{
    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);

private:
    ApplicationHeap* m_pApplicationHeap;

    nn::gfx::MemoryPool m_MemoryPool;
    void* m_pMemoryPool;
    ptrdiff_t m_pMemoryPoolOffset;
    ptrdiff_t m_pMemoryPoolCommandOffset;

    nn::gfx::MemoryPool m_MemoryPoolInvisible;
    void* m_pMemoryPoolInvisible;
    ptrdiff_t m_pMemoryPoolInvisibleOffset;

    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureSlotCount;

    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerSlotCount;

    nn::vi::Display* m_pDisplay;
    int m_DisplayWidth;
    int m_DisplayHeight;
    nn::vi::Layer* m_pLayer;
    nn::vi::NativeWindowHandle m_NativeWindowHandle;
    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::Fence m_Fence;
    nn::gfx::Texture m_ColorTargetTexture;
    nn::gfx::ColorTargetView m_ColorTargetView;
    nn::gfx::Texture m_DepthStencilTexture;
    nn::gfx::DepthStencilView m_DepthStencilView;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissorState;
    nn::gfx::BlendState m_BlendState;
    nn::gfx::DepthStencilState m_DepthStencilState;
    nn::gfx::RasterizerState m_RasterizerState;

    nn::util::BytePtr m_DebugFontHeap;
    nn::gfx::util::DebugFontTextWriter m_DebugFont;

    std::vector<void*> m_CommandBufferControlMemoryChunks;

    uint64_t m_FrameCount;

public:
    GraphicsSystem() NN_NOEXCEPT;
    ~GraphicsSystem() NN_NOEXCEPT;

    void SetApplicationHeap(ApplicationHeap* pValue) NN_NOEXCEPT;
    nn::gfx::Device& GetDevice() NN_NOEXCEPT;
    nn::vi::NativeWindowHandle GetNativeWindowHandle() NN_NOEXCEPT;
    nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT;
    nn::gfx::util::DebugFontTextWriter& GetDebugFont() NN_NOEXCEPT;

    uint64_t GetFrameCount() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void BeginDraw() NN_NOEXCEPT;
    void EndDraw() NN_NOEXCEPT;
    void DrawDone() NN_NOEXCEPT;
    void Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT;

    void AllocateTexture(nn::gfx::Texture* pTexture,
         const nn::gfx::Texture::InfoType* pInfo
         ) NN_NOEXCEPT;
    void AllocateBuffer(nn::gfx::Buffer* pBuffer,
        const nn::gfx::Buffer::InfoType* pInfo
        ) NN_NOEXCEPT;
    void FreeTexture(nn::gfx::Texture* pTexture
        ) NN_NOEXCEPT;
    void FreeBuffer(nn::gfx::Buffer* pBuffer
        ) NN_NOEXCEPT;

    void RegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::TextureView& textureView
        ) NN_NOEXCEPT;
    void RegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::Sampler& sampler) NN_NOEXCEPT;
    void UnregisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::TextureView& textureView
        ) NN_NOEXCEPT;
    void UnregisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::Sampler& sampler) NN_NOEXCEPT;

    void RegisterDebugFontTextWriter(nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_NOEXCEPT;
    void UnregisterDebugFontTextWriter(nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_NOEXCEPT;

private:
    static void OutOfCommandMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;
    static void OutOfControlMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;

    static void* AllocateFunction(size_t size,
        size_t alignment,
        void* pUserData) NN_NOEXCEPT;
    static void FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT;

    void InitializeDisplay() NN_NOEXCEPT;
    void InitializeDevice() NN_NOEXCEPT;
    void InitializeMemoryPool() NN_NOEXCEPT;
    void InitializeMemoryPoolInvisible() NN_NOEXCEPT;
    void InitializeSwapChain() NN_NOEXCEPT;
    void InitializeQueue() NN_NOEXCEPT;
    void InitializeFence() NN_NOEXCEPT;
    void InitializeColorTargetTexture() NN_NOEXCEPT;
    void InitializeColorTargetView() NN_NOEXCEPT;
    void InitializeDepthStencilTexture() NN_NOEXCEPT;
    void InitializeDepthStencilView() NN_NOEXCEPT;
    void InitializeTextureDescriptorPool() NN_NOEXCEPT;
    void InitializeSamplerDescriptorPool() NN_NOEXCEPT;
    void InitializeCommandBuffer() NN_NOEXCEPT;
    void InitializeViewportScissorState() NN_NOEXCEPT;
    void InitializeBlendState() NN_NOEXCEPT;
    void InitializeDepthStencilState() NN_NOEXCEPT;
    void InitializeRasterizerState() NN_NOEXCEPT;
    void InitializeDebugFont() NN_NOEXCEPT;
    void FinalizeMemoryPool() NN_NOEXCEPT;
    void FinalizeMemoryPoolInvisible() NN_NOEXCEPT;
    void FinalizeViewportScissorState() NN_NOEXCEPT;
    void FinalizeBlendState() NN_NOEXCEPT;
    void FinalizeDebugFont() NN_NOEXCEPT;
    void ResetCommandBuffer() NN_NOEXCEPT;
    void AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
        ) NN_NOEXCEPT;
    void AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
        ) NN_NOEXCEPT;
};

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT;
void NvFree(void* addr, void* userPtr) NN_NOEXCEPT;
void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT;
#endif

