﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_ParseFilename.h"

#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Endian.h>

namespace {

    bool TryParseEoi(const char** pOutNext, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        if(strLength != 0)
        {
            return false;
        }
        *pOutNext = str;
        return true;
    }

    template<typename TIntType>
    bool TryParseDecimal1(TIntType* pOutValue, char c) NN_NOEXCEPT
    {
        if(c < '0' || c > '9')
        {
            return false;
        }
        *pOutValue = static_cast<TIntType>(c - '0');
        return true;
    }

    template<typename TIntType>
    bool TryParseDecimal(const char** pOutNext, TIntType* pOutValue, int digits, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        if(strLength < digits)
        {
            return false;
        }

        TIntType value = 0;
        for(int i = 0; i < digits; i++)
        {
            TIntType v;
            if(!TryParseDecimal1(&v, str[i]))
            {
                return false;
            }
            value *= 10;
            value += v;
        }
        *pOutNext = str + digits;
        *pOutValue = value;
        return true;
    }

    template<typename TIntType>
    bool TryParseHexaUpper1(TIntType* pOutValue, char c) NN_NOEXCEPT
    {
        if(TryParseDecimal1(pOutValue, c))
        {
            return true;
        }
        if(c < 'A' || c > 'F')
        {
            return false;
        }
        *pOutValue = static_cast<TIntType>(c - 'A' + 10);
        return true;
    }

    bool TryParseHexaByte(const char** pOutNext, char* pOutValue, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        if(strLength < 2)
        {
            return false;
        }

        uint8_t value = 0;
        for(int i = 0; i < 2; i++)
        {
            uint8_t v;
            if(!TryParseHexaUpper1(&v, str[i]))
            {
                return false;
            }
            value <<= 4;
            value += v;
        }
        *pOutValue = static_cast<char>(value);
        *pOutNext = str + 2;
        return true;
    }

    bool MatchString(const char** pOutNext, const char* pattern, int patternLength, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(nn::util::Strnlen(pattern, patternLength), patternLength);
        if(strLength < patternLength)
        {
            return false;
        }

        int result = nn::util::Strncmp(str, pattern, patternLength);
        if(result != 0)
        {
            return false;
        }

        *pOutNext = str + patternLength;
        return true;
    }


    // <DateTime> ::= YYYYMMDDhhmmssii
    bool TryParseDateTime(const char** pOutNext, nn::capsrv::AlbumFileDateTime* pOutValue, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        if(strLength < 16)
        {
            return false;
        }

        nn::capsrv::AlbumFileDateTime time;
        const char* p = str;
        const char*const pEnd = str + strLength;
        // year
        if(!TryParseDecimal(&p, &time.year, 4, p, pEnd - p))
        {
            return false;
        }
        // month
        if(!TryParseDecimal(&p, &time.month, 2, p, pEnd - p))
        {
            return false;
        }
        // day
        if(!TryParseDecimal(&p, &time.day, 2, p, pEnd - p))
        {
            return false;
        }
        // hour
        if(!TryParseDecimal(&p, &time.hour, 2, p, pEnd - p))
        {
            return false;
        }
        // minute
        if(!TryParseDecimal(&p, &time.minute, 2, p, pEnd - p))
        {
            return false;
        }
        // second
        if(!TryParseDecimal(&p, &time.second, 2, p, pEnd - p))
        {
            return false;
        }
        // id
        if(!TryParseDecimal(&p, &time.id, 2, p, pEnd - p))
        {
            return false;
        }

        *pOutValue = time;
        *pOutNext = p;
        return true;
    }

    // <ApplicationId> ::= -app.XXXXXXXXXXXXXXXX
    bool TryParseApplicationId(const char** pOutNext, nn::ncm::ApplicationId* pOutValue, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        const char Prefix[] = "-app.";

        if(strLength < static_cast<ptrdiff_t>(std::strlen(Prefix) + 2 * sizeof(nn::ncm::ApplicationId)))
        {
            return false;
        }

        NN_ALIGNAS(8) char id[sizeof(nn::ncm::ApplicationId)] = {};
        const char* p = str;
        const char*const pEnd = str + strLength;

        if(!MatchString(&p, Prefix, static_cast<int>(std::strlen(Prefix)), p, pEnd - p))
        {
            return false;
        }

        for(int i = 0; i < sizeof(nn::ncm::ApplicationId); i++)
        {
            if(!TryParseHexaByte(&p, &id[i], p, pEnd - p))
            {
                return false;
            }
        }

        pOutValue->value = nn::util::LoadBigEndian(reinterpret_cast<const nn::Bit64*>(id));
        *pOutNext = p;
        return true;
    }

    // <ScreenShotExtension> ::= .jpg
    bool TryParseScreenShotExtension(const char** pOutNext, nn::capsrv::AlbumFileContentsType* pOutValue, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
    {
        const char Extension[] = ".jpg";

        if(strLength < static_cast<ptrdiff_t>(std::strlen(Extension)))
        {
            return false;
        }

        const char* p = str;
        const char*const pEnd = str + strLength;

        if(!MatchString(&p, Extension, static_cast<int>(std::strlen(Extension)), p, pEnd - p))
        {
            return false;
        }

        *pOutValue = nn::capsrv::AlbumFileContents_ScreenShot;
        *pOutNext = p;
        return true;
    }
}


// <Filename> ::= <DateTime> <ApplicationId> <ScreenShotExtension> <eoi>
bool TryParseFilename(FilenameInfo* pOutValue, const char* filename, int length) NN_NOEXCEPT
{
    const char* p = filename;
    const char*const pEnd = filename + length;

    FilenameInfo value = {};

    if(!TryParseDateTime(&p, &value.time, p, pEnd - p))
    {
        return false;
    }

    if(!TryParseApplicationId(&p, &value.applicationId, p, pEnd - p))
    {
        return false;
    }

    if(!TryParseScreenShotExtension(&p, &value.contents, p, pEnd - p))
    {
        return false;
    }

    if(!TryParseEoi(&p, p, pEnd - p))
    {
        return false;
    }

    *pOutValue = value;
    return true;
}
