﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>

#include "AlbumSynchronizer_Config.h"
#include "AlbumSynchronizer_ProgramOption.h"
#include "AlbumSynchronizer_StorageUtility.h"

#ifdef ALBUMSYNC_SUPPORT_ACTION_UPLOAD
#include "AlbumSynchronizer_ExecuteUploadAction.h"
#endif
#ifdef ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
#include "AlbumSynchronizer_ExecuteDownloadAction.h"
#endif
#ifdef ALBUMSYNC_SUPPORT_ACTION_CLEAN
#include "AlbumSynchronizer_ExecuteCleanAction.h"
#endif

#if !defined(ALBUMSYNC_IS_PRIVATETOOL) && !defined(ALBUMSYNC_IS_PUBLICTOOL)
#error
#endif
#if defined(ALBUMSYNC_IS_PRIVATETOOL) && defined(ALBUMSYNC_IS_PUBLICTOOL)
#error
#endif

namespace {

    void PrintHelp() NN_NOEXCEPT
    {
        const char* name = ALBUMSYNC_PROGRAM_NAME;
        //      |------------------------------------------------------------------------------|
        NN_LOG("Usage:\n");
#if ALBUMSYNC_SUPPORT_ACTION_HELP
        NN_LOG("  %s --help\n", name);
#endif
#if ALBUMSYNC_SUPPORT_ACTION_UPLOAD
        NN_LOG("  %s --upload --storage <storage> --directory <path>\n", name);
        NN_LOG("      [--force] [--raw]\n");
#endif
#if ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
        NN_LOG("  %s --download --storage <storage> --directory <path>\n", name);
        NN_LOG("      [--force]\n");
#endif
#if ALBUMSYNC_SUPPORT_ACTION_CLEAN
        NN_LOG("  %s --clean --storage <storage>\n", name);
#endif

        NN_LOG("\n");
        NN_LOG("Actions:\n");
#if ALBUMSYNC_SUPPORT_ACTION_HELP
        NN_LOG("  --help     ... Print this message.\n");
#endif
#if ALBUMSYNC_SUPPORT_ACTION_UPLOAD
        NN_LOG("  --upload   ... Copy files from Host to Target.\n");
        NN_LOG("                 By default, this action will fail if a file already exists on\n");
        NN_LOG("                 the Target storage. See --force option.\n");
        NN_LOG("                 If a source filename matches the specific pattern, the file \n");
        NN_LOG("                 will be converted to the capsrv's format.\n");
        NN_LOG("                 See FileConversion section and --raw option.\n");
#endif
#if ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
        NN_LOG("  --download ... Copy files from Target to Host.\n");
        NN_LOG("                 By default, this action will fail if a file already exists on\n");
        NN_LOG("                 the Host directory.\n");
        NN_LOG("                 See --force option.\n");
#endif
#if ALBUMSYNC_SUPPORT_ACTION_CLEAN
        NN_LOG("  --clean    ... Delete all files in the Target's storage.\n");
#endif
        NN_LOG("\n");
        NN_LOG("Arguments:\n");
#if ALBUMSYNC_SUPPORT_OPTION_STORAGE
        NN_LOG("  --storage <storage> ... Specify a storage on the Target.\n");
        NN_LOG("                          <storage> can be 'BuiltIn' or 'SdCard'.\n");
#endif
#if ALBUMSYNC_SUPPORT_OPTION_DIRECTORY
        NN_LOG("  --directory <path>  ... Specify a directory on the Host.\n");
        NN_LOG("                          <path> can be an absolute path on the Host.\n");
        NN_LOG("                          <path> must not be a root path.\n");
#endif
#if ALBUMSYNC_SUPPORT_OPTION_FORCE
        NN_LOG("  --force             ... Force copy files even if the destination\n");
        NN_LOG("                          is not empty.\n");
        NN_LOG("                          If the same filepath exists, the file will be\n");
        NN_LOG("                          overwritten.\n");
#endif
#if ALBUMSYNC_SUPPORT_OPTION_RAW
        NN_LOG("  --raw               ... Disable file conversions.\n");
#endif
#if ALBUMSYNC_SUPPORT_FILECONVERSION
        NN_LOG("\n");
        NN_LOG("File Conversion\n");
        NN_LOG("  The capsrv APIs will check filenames and file data in an album.\n");
        NN_LOG("  AlbumSynchronizer converts files to the suitable format in uploading. For a\n");
        NN_LOG("  file to be converted, you should name the file in the following pattern:\n");
        NN_LOG("\n");
        NN_LOG("    YYYYMMDDhhmmssii-app.XXXXXXXXXXXXXXXX.jpg\n");
        NN_LOG("\n");
        NN_LOG("      'YYYY', 'MM', 'DD' : Year, month, day.\n");
        NN_LOG("      'hh', 'mm', 'ss'   : Hour, minute, second.\n");
        NN_LOG("      'ii'               : A serial number in a time.\n");
        NN_LOG("      'XXX...XXX'        : A raw application ID in 16 digits of hexadecimals.\n");
        NN_LOG("    for example, '2016040120301000-app.0123456789ABCDEF.jpg' represents that\n");
        NN_LOG("    date is 2016/04/01 20:30:10-00 and the application ID is 0x0123456789ABCDEF.\n");
        NN_LOG("\n");
        NN_LOG("  This conversion is enabled by default. If you want to copy all files without\n");
        NN_LOG("  any conversion, use --raw option.\n");
#endif
        //      |------------------------------------------------------------------------------|
    }
}

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    if(argc <= 1)
    {
        PrintHelp();
        return;
    }

    ProgramOption opts;
    if(!ProgramOption::TryParseArgument(&opts, argc - 1, argv + 1))
    {
        NN_LOG("Parsing options failed.\n");
        NN_LOG("Given options:\n");
        for(int i = 1; i < argc; i++)
        {
            NN_LOG("  %s\n", argv[i]);
        }
        return;
    }

    switch(opts.GetAction())
    {
#if ALBUMSYNC_SUPPORT_ACTION_HELP
    case Action_PrintHelp:
        {
            PrintHelp();
            break;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_UPLOAD
    case Action_Upload:
        {
            ExecuteUploadAction(opts);
            break;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
    case Action_Download:
        {
            ExecuteDownloadAction(opts);
            break;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_CLEAN
    case Action_Clean:
        {
            ExecuteCleanAction(opts);
            break;
        }
#endif
    default: NN_UNEXPECTED_DEFAULT;
    }
}

