﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_ExecuteUploadAction.h"

#include <string>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>

#include "AlbumSynchronizer_Config.h"
#include "AlbumSynchronizer_StorageUtility.h"
#include "AlbumSynchronizer_ParseFilename.h"
#include "AlbumSynchronizer_ReduceImage.h"

#include "../../Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"
#include "../../Libraries/capsrv/server/detail/visrv_ConstructExifBuilder.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_CalculateJpegMac.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_ExtractJpegMakerNoteRange.h"

#if !ALBUMSYNC_SUPPORT_ACTION_UPLOAD
#error
#endif

nn::capsrv::server::EnvironmentInfo g_EnvironmentInfo;

namespace {
    // Default date to use for filenames that cannot be
    static const int DefaultYear = 2016;
    static const int DefaultMonth = 1;
    static const int DefaultDay = 1;

    nn::Result CreateDirectories(const std::string& path) NN_NOEXCEPT
    {
        std::vector<char> work;
        work.assign(path.begin(), path.end());
        work.push_back('\0');
        int length = static_cast<int>(path.size());

        for(;;)
        {
            NN_RESULT_THROW_UNLESS(length > 0, nn::capsrv::ResultAlbumError());
            if(work[length - 1] == '/')
            {
                work[length - 1] = '\0';
                length--;
            }
            else
            {
                break;
            }
        }
        work[length] = '/';
        length++;

        int pos = 0;

        // skip "ST:/"
        for(pos = 0; pos < length; pos++)
        {
            if(work[pos] == ':')
            {
                pos++;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(work[pos] == '/', nn::capsrv::ResultAlbumError());
        pos++;

        for(; pos < length; pos++)
        {
            if(work[pos] == '/')
            {
                work[pos] = '\0';
                auto result = nn::fs::CreateDirectory(work.data());
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                        // ok
                    }
                NN_RESULT_END_TRY;
                work[pos] = '/';
            }
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result CreateParentDirectories(const std::string& path) NN_NOEXCEPT
    {
        std::string work = path;
        int length = static_cast<int>(work.size());
        int pos = 0;
        for(pos = length - 1; pos >= 0; pos--)
        {
            if(work[pos] == '/')
            {
                work[pos] = '\0';
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(pos >= 0, nn::capsrv::ResultAlbumError());
        NN_RESULT_DO(CreateDirectories(work));
        NN_RESULT_SUCCESS;
    }


    nn::Result LoadFile(std::vector<uint8_t>& outValue, const std::string& filepath) NN_NOEXCEPT
    {
        nn::fs::FileHandle hFile = {};
        NN_RESULT_DO(nn::fs::OpenFile(&hFile, filepath.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(hFile);
        };

        int64_t size = 0;
        NN_RESULT_DO(nn::fs::GetFileSize(&size, hFile));

        if(size == 0)
        {
            outValue.clear();
            NN_RESULT_SUCCESS;
        }

        outValue.resize(static_cast<size_t>(size));

        size_t readSize = 0;
        NN_RESULT_DO(nn::fs::ReadFile(&readSize, hFile, 0, outValue.data(), outValue.size()));

        NN_RESULT_SUCCESS;
    }

    nn::Result SaveFile(const std::vector<uint8_t>& srcData, const std::string& filepath, bool isForced) NN_NOEXCEPT
    {
        NN_RESULT_DO(CreateParentDirectories(filepath));

        nn::fs::FileHandle hDstFile = {};
        int64_t size = static_cast<int64_t>(srcData.size());
        NN_RESULT_TRY(nn::fs::CreateFile(filepath.c_str(), size))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                if(!isForced)
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY;

        if(size == 0)
        {
            NN_RESULT_SUCCESS;
        }

        int writeOpenMode = nn::fs::OpenMode_Write;
        if(isForced)
        {
            writeOpenMode |= nn::fs::OpenMode_AllowAppend;
        }
        NN_RESULT_DO(nn::fs::OpenFile(&hDstFile, filepath.c_str(), writeOpenMode));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(hDstFile);
        };

        NN_RESULT_DO(nn::fs::SetFileSize(hDstFile, size));
        NN_RESULT_DO(nn::fs::WriteFile(hDstFile, 0, srcData.data(), srcData.size(), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    nn::Result EncodeJpeg(
        std::vector<uint8_t>& outData,
        const std::vector<uint8_t>& rawData,
        nn::image::Dimension dimension,
        nn::image::ExifBuilder* pExifBuilder
    ) NN_NOEXCEPT
    {
        const int QualityStart    = 90;
        const int QualityLowLimit = 50;
        const int QualityDelta    = 10;
        std::vector<uint8_t> workData;
        {
            nn::image::JpegEncoder encoder;
            encoder.SetPixelData(rawData.data(), dimension, 1);
            encoder.SetSamplingRatio(nn::image::JpegSamplingRatio_422);
            int quality;
            size_t encodedSize = 0;
            for(quality = QualityStart; quality >= QualityLowLimit; quality -= QualityDelta)
            {
                encoder.SetQuality(quality);
                NN_RESULT_THROW_UNLESS(
                    encoder.Analyze() == nn::image::JpegStatus_Ok,
                    nn::capsrv::ResultScreenShotEncodingFailed()
                );
                size_t workSize = encoder.GetAnalyzedWorkBufferSize();
                workData.resize(workSize);

                outData.resize(nn::capsrv::server::ScreenShotImageSizeLimit);

                encodedSize = 0;
                auto encodeResult = encoder.Encode(&encodedSize, outData.data(), outData.size(), workData.data(), workData.size(), pExifBuilder);
                if(encodeResult == nn::image::JpegStatus_Ok)
                {
                    break;
                }
                else if(encodeResult == nn::image::JpegStatus_ShortOutput)
                {
                    continue;
                }
                else
                {
                    NN_RESULT_THROW(nn::capsrv::ResultScreenShotEncodingFailed());
                }
            }
            if(quality < QualityLowLimit)
            {
                NN_RESULT_THROW(nn::capsrv::ResultScreenShotEncodingFailed());
            }

            outData.resize(encodedSize);
        }
        NN_RESULT_SUCCESS;
    }

    // capsrv で認識できる形式にエンコードしなおしてコピー
    nn::Result ConvertCopyFile(
        const std::string& srcRootPath,
        const std::string& dstRootPath,
        const std::string& srcFilename,
        const FilenameInfo& info,
        bool isForced
        ) NN_NOEXCEPT
    {
        NN_LOG("Converting file ... %s\n", srcFilename.c_str());

        std::string dstFilename;
        dstFilename.resize(nn::capsrv::server::album::AlbumPathUtility::FilenameSize);
        int n = 0;
        NN_RESULT_DO(nn::capsrv::server::album::AlbumPathUtility::ValidateTime(&info.time));
        NN_RESULT_DO(nn::capsrv::server::album::AlbumPathUtility::ValidateApplicationId(info.applicationId));
        NN_RESULT_DO(nn::capsrv::server::album::AlbumPathUtility::ValidateFileContents(info.contents, g_EnvironmentInfo));
        nn::capsrv::server::album::AlbumPathUtility::GetFilename(&n, &dstFilename[0], dstFilename.size(), &info.time, info.applicationId, info.contents, g_EnvironmentInfo);
        NN_LOG("  destination file: %s\n", dstFilename.c_str());

        std::string srcPath = srcRootPath + "/" + srcFilename;
        char dstSubDirectoryPath[nn::capsrv::server::album::AlbumPathUtility::PathSize] = {};
        nn::util::SNPrintf(
            dstSubDirectoryPath,
            sizeof(dstSubDirectoryPath),
            "%04d/%02d/%02d/",
            static_cast<int>(info.time.year),
            static_cast<int>(info.time.month),
            static_cast<int>(info.time.day)
        );
        std::string dstPath = dstRootPath + "/" + dstSubDirectoryPath + dstFilename;

        std::vector<uint8_t> fileData;
        std::vector<uint8_t> thumbData;
        std::vector<uint8_t> rawData;
        std::vector<uint8_t> rawThumbData;
        std::vector<uint8_t> workData;

        nn::image::Dimension dimension;

        NN_RESULT_DO(LoadFile(fileData, srcPath));
        NN_RESULT_THROW_UNLESS(fileData.size() > 0, nn::capsrv::ResultAlbumInvalidFileData());

        // 一旦デコード。
        {
            nn::image::JpegDecoder decoder;
            decoder.SetImageData(fileData.data(), fileData.size());
            NN_RESULT_THROW_UNLESS(
                decoder.Analyze() == nn::image::JpegStatus_Ok,
                nn::capsrv::ResultAlbumInvalidFileData()
            );
            dimension = decoder.GetAnalyzedDimension();
            auto workSize = decoder.GetAnalyzedWorkBufferSize();


            rawData.resize(4 * dimension.width * dimension.height);
            workData.resize(workSize);

            NN_RESULT_THROW_UNLESS(
                decoder.Decode(rawData.data(), rawData.size(), 1, workData.data(), workData.size()) == nn::image::JpegStatus_Ok,
                nn::capsrv::ResultAlbumInvalidFileData()
            );
        }

        // サムネイル生成
        rawThumbData = ReduceImage(
            rawData,
            4,
            static_cast<int>(dimension.width),
            static_cast<int>(dimension.height),
            nn::capsrv::ViewerThumbnailImageSize_Width,
            nn::capsrv::ViewerThumbnailImageSize_Height
        );
        NN_RESULT_TRY(EncodeJpeg(thumbData, rawThumbData, { nn::capsrv::ViewerThumbnailImageSize_Width, nn::capsrv::ViewerThumbnailImageSize_Height }, nullptr))
            NN_RESULT_CATCH_ALL
            {
                thumbData.clear();
            }
        NN_RESULT_END_TRY;



        // exif 情報を計算
        NN_ALIGNAS(NN_ALIGNOF(std::max_align_t)) char exifMemory[nn::capsrv::server::ScreenShotExifWorkSize];
        nn::image::ExifBuilder* pExifBuilder = nullptr;
        {
            nn::capsrv::server::detail::MakerNoteInfo minfo = {};
            // Version1 の署名は spl が必要なため version0 のみ生成
            minfo.version = nn::capsrv::server::detail::MakerNoteVersion_Version0;
            // signature は 0 のまま
            minfo.encryptedApplicationId = nn::capsrv::server::detail::EncryptApplicationId(info.applicationId, false);

            pExifBuilder = nn::capsrv::server::detail::ConstructExifBuilder(exifMemory, sizeof(exifMemory), minfo, thumbData.data(), thumbData.size());
        }

        // エンコードしなおす
        NN_RESULT_DO(EncodeJpeg(fileData, rawData, dimension, pExifBuilder));
        nn::capsrv::server::detail::DestructExifBuilder(pExifBuilder);

        // 署名部分を計算
        {
            int64_t makerNoteOffset = 0;
            int64_t makerNoteSize = 0;
            NN_RESULT_DO(nn::capsrv::server::detail::ExtractJpegMakerNoteRange(&makerNoteOffset, &makerNoteSize, fileData.data(), fileData.size(), exifMemory, sizeof(exifMemory)));

            nn::capsrv::server::detail::Signature signature;
            int64_t signatureOffset = 0;
            NN_RESULT_DO(nn::capsrv::server::detail::CalculateJpegMac(&signature, &signatureOffset, fileData.data(), fileData.size(), nn::capsrv::server::detail::MakerNoteVersion_Version0, makerNoteOffset, makerNoteSize));

            std::memcpy(fileData.data() + signatureOffset, &signature, sizeof(signature));
        }

        NN_RESULT_DO(SaveFile(fileData, dstPath, isForced));

        NN_RESULT_SUCCESS;
    }   // NOLINT(impl/function_size)

    // ファイルをそのままコピー
    nn::Result RawCopyFile(
        const std::string& srcRootPath,
        const std::string& dstRootPath,
        const std::string& filename,
        bool isForced
        ) NN_NOEXCEPT
    {
        NN_LOG("Copying file ... %s\n", filename.c_str());
        std::string srcPath = srcRootPath + "/" + filename;

        std::vector<uint8_t> srcData;
        NN_RESULT_DO(LoadFile(srcData, srcPath));

        nn::capsrv::AlbumFileId id = {};
        const char* p = filename.c_str();
        nn::Result result = nn::capsrv::server::album::AlbumPathUtility::ParseAlbumFilename(&p, &id, p, filename.size(), g_EnvironmentInfo);

        int year = -1;
        int month = -1;
        int day = -1;
        if (result.IsFailure())
        {
            NN_LOG("Filename parse for file '%s' failed; uploading with with a date of %04d/%02d/%02d\n", filename.c_str(), DefaultYear, DefaultMonth, DefaultDay);
            year = DefaultYear;
            month = DefaultMonth;
            day = DefaultDay;
        }
        else
        {
            year = id.time.year;
            month = id.time.month;
            day = id.time.day;
        }

        char dstSubDirectoryPath[nn::capsrv::server::album::AlbumPathUtility::PathSize] = {};
        nn::util::SNPrintf(
            dstSubDirectoryPath,
            sizeof(dstSubDirectoryPath),
            "%04d/%02d/%02d/",
            static_cast<int>(year),
            static_cast<int>(month),
            static_cast<int>(day)
        );
        std::string dstPath = dstRootPath + "/" + dstSubDirectoryPath + filename;
        NN_RESULT_DO(SaveFile(srcData, dstPath, isForced));

        NN_RESULT_SUCCESS;
    }

    nn::Result UploadAllFiles(const char* dstMountName, const char* srcMountName, bool isForced, bool isRawMode) NN_NOEXCEPT
    {
        std::string srcRootPath = std::string(srcMountName) + ":/";
        std::string dstRootPath = std::string(dstMountName) + ":/";

        nn::fs::DirectoryHandle hSrcDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hSrcDir, srcRootPath.c_str(), nn::fs::OpenDirectoryMode_File));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hSrcDir);
        };
        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hSrcDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> srcEntryList;
        srcEntryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, srcEntryList.data(), hSrcDir, static_cast<int64_t>(srcEntryList.size())));
        srcEntryList.resize(static_cast<size_t>(n));

        for(auto& e : srcEntryList)
        {
            FilenameInfo info = {};
            if(isRawMode)
            {
                NN_RESULT_DO(RawCopyFile(srcRootPath, dstRootPath, e.name, isForced));
            }
            else if(TryParseFilename(&info, e.name, nn::util::Strnlen(e.name, static_cast<int>(sizeof(e.name)))))
            {
                NN_RESULT_TRY(ConvertCopyFile(srcRootPath, dstRootPath, e.name, info, isForced))
                    NN_RESULT_CATCH_ALL
                    {
                        NN_LOG("Conversion failed. Performing RawCopy ...(ErrorCode=0x%08X)\n", _nn_result_try_temporary.GetInnerValueForDebug());
                        NN_RESULT_DO(RawCopyFile(srcRootPath, dstRootPath, e.name, isForced));
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                NN_RESULT_DO(RawCopyFile(srcRootPath, dstRootPath, e.name, isForced));
            }
        }

        NN_LOG("Copied %d files\n", static_cast<int>(n));
        NN_RESULT_SUCCESS;
    }

}

void ExecuteUploadAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Uploading Album ...\n");
    g_EnvironmentInfo = nn::capsrv::server::EnvironmentInfo::GetValueForPrivateTool();
    NN_UTIL_SCOPE_EXIT{ g_EnvironmentInfo.Finalize(); };

    if(!MountTargetStorage(*opts.GetStorage()))
    {
        return;
    }
    if(!MountHostDirectory(opts.GetHostDirectory()))
    {
        return;
    }
    if(!opts.IsForceEnabled())
    {
        bool isEmpty = false;
        IsEmptyStorage(&isEmpty, "TARG");
        if(!isEmpty)
        {
            NN_LOG("Failed. The destination is not empty.\n");
            NN_LOG("  dest: %s\n", opts.GetStorageName());
            return;
        }
    }
    if(!UploadAllFiles("TARG", "HOST", opts.IsForceEnabled(), opts.IsRawModeEnabled()).IsSuccess())
    {
        return;
    }
    UnmountHostDirectory();
    UnmountTargetStorage();
    NN_LOG("Uploading Album ... complete\n");
}
