﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_ExecuteCleanAction.h"

#include <vector>
#include <string>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>

#include "AlbumSynchronizer_Config.h"
#include "AlbumSynchronizer_StorageUtility.h"

#if !ALBUMSYNC_SUPPORT_ACTION_CLEAN
#error
#endif

namespace {

    nn::Result DeleteAllFiles(const char* mountName) NN_NOEXCEPT
    {
        std::string rootPath = std::string(mountName) + ":/";

        nn::fs::DirectoryHandle hDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hDir, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hDir);
        };
        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> entryList;
        entryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, entryList.data(), hDir, static_cast<int64_t>(entryList.size())));

        for(auto& e : entryList)
        {
            std::string path = rootPath + "/" + e.name;
            if(e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_LOG("Deleting directory ... %s\n", e.name);
                NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path.c_str()));
            }
            else
            {
                NN_LOG("Deleting file ... %s\n", e.name);
                NN_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
            }
        }

        NN_LOG("Deleted %d files/directories.\n", static_cast<int>(n));
        NN_RESULT_SUCCESS;
    }
}

void ExecuteCleanAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Cleaning Album ...\n");
    if(!MountTargetStorage(*opts.GetStorage()))
    {
        return;
    }
    if(!DeleteAllFiles("TARG").IsSuccess())
    {
        return;
    }
    UnmountTargetStorage();
    NN_LOG("Cleaning Album ... complete\n");
}
