﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumFileMaker_ExtractScreenShotMakerNote.h"

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "AlbumFileMaker_FileIo.h"

#include "../../Libraries/capsrv/server/detail/capsrvServer_MakerNoteInfo.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_ExtractJpegMakerNoteRange.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_EncryptMakerNote.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_GenerateMakerNote.h"

namespace nn{ namespace album{

    namespace {
        template<typename MakerNotePolicy>
        bool DecryptMakerNote(std::vector<char>& makernote) NN_NOEXCEPT
        {
            static const size_t Offset   = MakerNotePolicy::EncryptionStartOffset;
            static const size_t UnitSize = MakerNotePolicy::EncryptionPolicy::UnitSize;

            if(makernote.size() < Offset)
            {
                return false;
            }

            if(((makernote.size() - Offset) % UnitSize) != 0)
            {
                return false;
            }

            MakerNotePolicy::EncryptionPolicy::DecryptInplace(makernote.data(), makernote.size(), Offset);
            return true;
        }
    }

    bool ExtractScreenShotMakerNote(const ProgramOption& opts) NN_NOEXCEPT
    {
        NN_ALBUM_CHECK_OPTION_INPUTPATH(opts, false);
        NN_ALBUM_CHECK_OPTION_OUTPUTPATH(opts, false);

        std::vector<char> fileData;
        if(!LoadFile(&fileData, *opts.inputPath))
        {
            return false;
        }

        int64_t offset = 0;
        int64_t size = 0;
        {
            std::vector<char> work(4096);
            auto result = nn::capsrv::server::detail::ExtractJpegMakerNoteRange(&offset, &size, fileData.data(), fileData.size(), work.data(), work.size());
            if(result.IsFailure())
            {
                return false;
            }
        }
        NN_ALBUM_LOG("makernote offset=%lld, size=%lld\n", offset, size);

        std::vector<char> makernote;
        makernote.insert(makernote.end(), fileData.begin() + offset, fileData.begin() + offset + size);

        bool isDecrypted = false;
        if(makernote.size() >= 4)
        {
            uint32_t version = 0;
            std::memcpy(&version, makernote.data(), 4);

            switch(version)
            {
            #define NN_ALBUMFILEMAKER_DECRYPTION(Policy) \
            case Policy::Version: \
                isDecrypted = DecryptMakerNote<Policy>(makernote); \
                break;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_ALBUMFILEMAKER_DECRYPTION)

            #undef NN_ALBUMFILEMAKER_DECRYPTION
            default: ;
            }
        }

        if(!isDecrypted)
        {
            NN_ALBUM_LOG_WARN("makernote size doesn't match any supported version(s). exporting makernote as-is.\n");
        }

        if(!SaveFile(*opts.outputPath, makernote))
        {
            return false;
        }

        return true;
    }

}}

