﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Thread.h>
#include <memory>

#include <nn/manu/server/manu_Log.h>
#include <nn/manu/manu_Result.public.h>

#include "manu_CommandServer.h"
#include "manu_Command.h"


namespace nn { namespace manu {

const int CommandServer::bufferSize = 4096;
NN_ALIGNAS(4096) char CommandServer::receiveBuffer[bufferSize];

void CommandServer::RunThread(void* arg) NN_NOEXCEPT
{
    reinterpret_cast<CommandServer*>(arg)->MainLoop();
}

void CommandServer::MainLoop()
{
    for(;;)
    {
        nn::Result result;
        command::CommandHeader header = {0};

        result = WaitForReceiveCommand(header);
        if( result.IsFailure() )
        {
            continue;
        }
        ::std::unique_ptr<nn::Bit8[]> body(new nn::Bit8[header.BodySize]);

        if( WaitForReceiveBody(body.get(), header.BodySize).IsFailure() )
        {
            NN_MANU_WARNING_LOG("WaitForReceiveBody() Fail (Inner value : %x)\n", result.GetInnerValueForDebug());
            continue;
        }

        result = command::ProcessCommand(header, body.get());
        command::ResultData resultData;
        resultData.Id = result.IsSuccess() ? command::ResultId::Success : command::ResultId::ProcessCommandError;
        resultData.NnResult = result.GetInnerValueForDebug();
        if( SendResultData(resultData).IsFailure() )
        {
            NN_MANU_WARNING_LOG("SendResultData() Fail (Inner value : %x)\n", result.GetInnerValueForDebug());
            continue;
        }
    }
}

nn::Result CommandServer::Initialize(void* pStack, size_t stackSize) NN_NOEXCEPT
{
    NN_RESULT_DO(usbTransferPipe.Initialize(usbInterfaceNumber));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&this->serverThread, RunThread, this, pStack, stackSize, nn::os::DefaultThreadPriority));
    NN_RESULT_SUCCESS;
}

void CommandServer::Run() NN_NOEXCEPT
{
    nn::os::StartThread(&this->serverThread);
}

nn::Result CommandServer::WaitForReceiveBody(void* body, size_t size) NN_NOEXCEPT
{
    uint32_t bytesRead =0;

    NN_RESULT_DO(usbTransferPipe.Receive(&bytesRead, receiveBuffer, size));

    std::memcpy(body, receiveBuffer, size);

    NN_RESULT_DO(SendResultId(command::ResultId::Success));
    NN_RESULT_SUCCESS;
}

nn::Result CommandServer::WaitForReceiveCommand(command::CommandHeader& header) NN_NOEXCEPT
{
    uint32_t bytesRead =0;

    NN_RESULT_DO(usbTransferPipe.Receive(&bytesRead, receiveBuffer, sizeof(header)));
    header = *reinterpret_cast<command::CommandHeader*>(receiveBuffer);

    if(header.MagicNumber != command::CommandHeaderMagicNumber)
    {
        NN_MANU_ERROR_LOG("Invalid Magic Number (Expect : %016X, Actual : %016X)\n", command::CommandHeaderMagicNumber, header.MagicNumber);
        NN_RESULT_DO(SendResultId(command::ResultId::DataError));
        ResultInvalidCommand();
    }

    NN_RESULT_DO(SendResultId(command::ResultId::Success));
    NN_RESULT_SUCCESS;
}

nn::Result CommandServer::SendResultId(command::ResultId resultId) NN_NOEXCEPT
{
    uint32_t bytesWrite =0;
    *reinterpret_cast<command::ResultId *>(receiveBuffer) = resultId;
    NN_RESULT_DO(usbTransferPipe.Send(&bytesWrite, receiveBuffer, sizeof(resultId)));
    NN_RESULT_SUCCESS;
}

nn::Result CommandServer::SendResultData(command::ResultData& resultData) NN_NOEXCEPT
{
    uint32_t bytesWrite =0;
    *reinterpret_cast<command::ResultData *>(receiveBuffer) = resultData;
    NN_RESULT_DO(usbTransferPipe.Send(&bytesWrite, receiveBuffer, sizeof(resultData)));
    NN_RESULT_SUCCESS;
}

}}
