﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/server/ldn_HipcServer.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Thread.h>
#include <nn/ndd/detail/ndd_Main.h>

namespace
{
    NN_ALIGNAS(4096) char g_HeapMemory[32 * 1024];
    nn::lmem::HeapHandle g_HeapHandle;

    const int IpcServerThreadCount = 2;
    NN_ALIGNAS(4096) char g_HipcServerThreadStack[IpcServerThreadCount][16 * 1024];

    void HipcServerThread(void* argc)
    {
        NN_UNUSED(argc);
        nn::ldn::server::LoopServer();
    }

    NN_ALIGNAS(4096) char g_NddMainThreadStack[4 * 1024];
}

void* operator new(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

extern "C" void nninitStartup()
{
    g_HeapHandle = nn::lmem::CreateExpHeap(
        g_HeapMemory, sizeof(g_HeapMemory), nn::lmem::CreationOption_NoOption);
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    nn::os::ThreadType nddMainThread;
    if(nn::ndd::detail::IsNddServiceEnabled())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &nddMainThread, nn::ndd::detail::NddMainThread, nullptr,
            g_NddMainThreadStack, sizeof(g_NddMainThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(ndd, Main)));
        nn::os::SetThreadNamePointer(&nddMainThread, NN_SYSTEM_THREAD_NAME(ndd, Main));
        nn::os::StartThread(&nddMainThread);
    }

    nn::ldn::server::InitializeServer();

    // HIPC サーバーを専用のスレッドで起動します。
    nn::os::ThreadType thread[IpcServerThreadCount];
    for (int i = 0; i < IpcServerThreadCount; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &thread[i], HipcServerThread, nullptr,
            g_HipcServerThreadStack[i], sizeof(g_HipcServerThreadStack[i]),
            NN_SYSTEM_THREAD_PRIORITY(ldn, IpcServer)));
        nn::os::SetThreadNamePointer(&thread[i], NN_SYSTEM_THREAD_NAME(ldn, IpcServer));
        nn::os::StartThread(&thread[i]);
    }

    // HIPC サーバースレッドの終了まで待機します。
    for (int i = 0; i < IpcServerThreadCount; ++i)
    {
        nn::os::WaitThread(&thread[i]);
        nn::os::DestroyThread(&thread[i]);
    }
}
