﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <memory>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/hidbus/hidbus_IHidbusServer.sfdl.h>
#include <nn/hidbus/hidbus_ServiceName.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_LockableMutexType.h"
#include "hid_StaticObject.h"

#include "hidbus_HidbusServer.h"

namespace {

//!< セッション数の最大値
const int SessionCountMax = 100;

//!< ポート数の最大値
const int PortCountMax = 2;

//!< hidbus IPC サーバーのスタックサイズ
const size_t HidbusStackSize = 4096;

//!< hidbus IPC サーバーのスタック
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_HidbusServerStack[HidbusStackSize];

//!< hidbus IPC サーバーのスレッド
nn::os::ThreadType g_HidbusServerThread;

//!< サーバマネージャのオプション宣言です。
struct ServerManagerOption
{
    static const size_t PointerTransferBufferSize = 512;
};

//!< サーバマネージャを扱うためのクラスです。
class ServerManager final
    : public ::nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax,
                                                       PortCountMax,
                                                       ServerManagerOption>
{
};

//!< ServerManager に HidbusServer を登録します。
void RegisterHidbusServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;



void HidbusServerThread(void* arg)
{
    ServerManager& serverManager =
        ::nn::hid::detail::StaticObject<ServerManager>::Get();

    serverManager.LoopAuto();
}


} // namespace

namespace nn { namespace hid {

void StartHidbusServer()
{
    ServerManager& serverManager =
        ::nn::hid::detail::StaticObject<ServerManager>::Get();
    RegisterHidbusServerWithServerManager(&serverManager);

    serverManager.Start();

    auto result = nn::os::CreateThread(&g_HidbusServerThread,
                                       HidbusServerThread,
                                       nullptr,
                                       g_HidbusServerStack,
                                       sizeof(g_HidbusServerStack),
                                       NN_SYSTEM_THREAD_PRIORITY(hidbus, IpcServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    ::nn::os::SetThreadNamePointer(
        &g_HidbusServerThread, NN_SYSTEM_THREAD_NAME(hidbus, IpcServer));

    nn::os::StartThread(&g_HidbusServerThread);
}

}}

namespace {

void RegisterHidbusServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::hidbus::IHidbusServer> pHidbusServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hidbus::server::CreateHidbusServerProxy(&pHidbusServer));

    const char* const serviceName = ::nn::hidbus::HidbusServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pHidbusServer, SessionCountMax, serviceName));
}

} // namespace
