﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/grc/sf/grc_IUserManager.sfdl.h>
#include <nn/grc/grc_ServiceName.h>
#include <nn/grc/grc_Config.h>
#include "grc_Server.h"

namespace {

enum PortIndex
{
    PortIndex_GrcUser,
    PortIndex_Count
};

const int HipcServerThreadCount = 1;
const size_t HipcServerThreadStackSize = 4096;
NN_ALIGNAS(4096) char g_HipcServerThreadStacks[HipcServerThreadCount][HipcServerThreadStackSize];
nn::os::ThreadType  g_HipcServerThreads[HipcServerThreadCount];

static const int TotalSessionCount = nn::grc::UserSessionCount;

struct MyServerOption
{
    static const size_t PointerTransferBufferSize = 4 * 1024;
};

class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count, MyServerOption>
{
};

std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

void LoopHipcServerThreadFunction(void* arg)
{
    MyServerManager* pManager = reinterpret_cast<MyServerManager*>(arg);

    // IPCリクエスト処理開始
    pManager->LoopAuto();
}

} // anonymous

namespace nn { namespace grc {
nn::sf::SharedPointer<nn::grc::sf::IUserManager> CreateUserManagerByDfc() NN_NOEXCEPT;
}};

void InitializeServer() NN_NOEXCEPT
{
    NN_GRC_LOG_INFO("Starting InitializeServer ...\n");

    NN_ABORT_UNLESS(!g_pMyServerManager, "Server is already initialized.\n");

    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    nn::Result result;
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::grc::CreateUserManagerByDfc(), nn::grc::UserSessionCount, nn::grc::UserPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    g_pMyServerManager->Start();

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::Result result = nn::os::CreateThread(&g_HipcServerThreads[i],
                                                 LoopHipcServerThreadFunction,
                                                 g_pMyServerManager,
                                                 g_HipcServerThreadStacks[i],
                                                 HipcServerThreadStackSize,
                                                 NN_SYSTEM_THREAD_PRIORITY(grc, HipcServer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SetThreadNamePointer(&g_HipcServerThreads[i], NN_SYSTEM_THREAD_NAME(grc, HipcServer));
    }
}

void LoopServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager, "Server is not initialized.\n");

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::os::StartThread( &g_HipcServerThreads[i] );
    }

    g_pMyServerManager->LoopAuto(); // RequestStop() が呼ばれるまで返らない
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager, "Server is not initialized.\n");

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::os::WaitThread(&g_HipcServerThreads[i]);
        nn::os::DestroyThread(&g_HipcServerThreads[i]);
    }

    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}
