﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <new>
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/hid/hid_BatteryLevel.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_AbstractedPad.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/htcs.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_TypedStorage.h>

#include "cs_HidServer.h"

#define NN_HTCS_SOCKET_ERROR_NONE \
    return ::nn::htcs::HTCS_ENONE

#define NN_HTCS_SOCKET_ERROR_DO(r) \
    do \
    { \
        const auto& _nn_htcs_socket_error_do_temporary((r)); \
        if (_nn_htcs_socket_error_do_temporary != ::nn::htcs::HTCS_ENONE) \
        { \
            return _nn_htcs_socket_error_do_temporary; \
        } \
    } while (NN_STATIC_CONDITION(0))

namespace nn { namespace hid { namespace detail {

::nn::Result GetXpadIds(
    int* pOutCount, BasicXpadId outXpadIds[], int count) NN_NOEXCEPT;

::nn::Result GetXpadPlayerNumber(
    int* pOutValue, const BasicXpadId& xpadId) NN_NOEXCEPT;

::nn::Result InitializeXpad(const BasicXpadId& xpadId) NN_NOEXCEPT;

}}} // namespace nn::hid::detail

namespace nn { namespace cs {

namespace {

//!< ポート名です。
const char PortName[] = "iywys@$hid";

//!< ディスプレイの幅
const int32_t DisplayWidth = 1280;

//!< ディスプレイの高さ
const int32_t DisplayHeight = 720;

//!< 接続試行間隔
const ::nn::TimeSpanType ConnectionTrialInterval =
    ::nn::TimeSpanType::FromMilliSeconds(200);

//!< ポーリング間隔
const ::nn::TimeSpanType PollingInterval =
    ::nn::TimeSpanType::FromMilliSeconds(8);

//!< レシーバスレッドのスタックサイズ
const size_t ReceiverThreadStackSize = 4 * ::nn::os::ThreadStackAlignment;

//!< センダスレッドのスタックサイズ
const size_t SenderThreadStackSize = 4 * ::nn::os::ThreadStackAlignment;

class Socket;

//!< ソケットチャネルを扱うクラスです。
class SocketChannel final
{
private:
    //!< ソケット
    Socket* m_pParent;

    //!< セッション ID
    uint64_t m_SessionId;

public:
    SocketChannel() NN_NOEXCEPT;

    //!< データを受信します。
    ::nn::htcs::SocketError Receive(size_t* pOutSize,
                                    void* buffer,
                                    size_t bufferSize,
                                    int flags = 0) NN_NOEXCEPT;

    //!< データを送信します。
    ::nn::htcs::SocketError Send(size_t* pOutSize,
                                 const void* buffer,
                                 size_t bufferSize,
                                 int flags = 0) NN_NOEXCEPT;

    //!< ソケットチャネルを閉じます。
    void Close() NN_NOEXCEPT;

private:
    friend class Socket;

    SocketChannel(Socket* pParent, uint64_t sessionId) NN_NOEXCEPT;
};

//!< ソケットを扱うクラスです。
class Socket final
{
    NN_DISALLOW_COPY(Socket);
    NN_DISALLOW_MOVE(Socket);

private:
    //!< 無効なソケット
    const int InvalidSocket = -1;

    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< バインド対象のターゲットとポート
    ::nn::htcs::SockAddrHtcs m_SockAddrHtcs;

    //!< ソケット
    int m_Socket;

    //!< サブソケット
    int m_SubSocket;

    //!< セッション ID
    uint64_t m_SessionId;

public:
    Socket() NN_NOEXCEPT;

    //!< ポート名を設定します。
    void SetPortName(const char* name) NN_NOEXCEPT;

    //!< ソケットチャネルを作成します。
    ::nn::htcs::SocketError CreateSocketChannel(SocketChannel* pOutValue
                                                ) NN_NOEXCEPT;

private:
    friend class SocketChannel;

    //!< データを受信します。
    ::nn::htcs::SocketError Receive(size_t* pOutSize,
                                    void* buffer,
                                    size_t bufferSize,
                                    int flags,
                                    uint64_t sessionId) NN_NOEXCEPT;

    //!< データを送信します。
    ::nn::htcs::SocketError Send(size_t* pOutSize,
                                 const void* buffer,
                                 size_t bufferSize,
                                 int flags,
                                 uint64_t sessionId) NN_NOEXCEPT;

    //!< 最後に発生したソケットエラーを返します。
    ::nn::htcs::SocketError GetLastError() const NN_NOEXCEPT;

    //!< セッションを取得します。
    ::nn::htcs::SocketError AcquireSession(uint64_t* pOutSessionId,
                                           int* pOutSocket) NN_NOEXCEPT;

    //!< セッションを取得します。
    ::nn::htcs::SocketError AcquireSession(uint64_t* pOutSessionId,
                                           int* pOutSocket,
                                           int* pOutSubSocket) NN_NOEXCEPT;

    //!< セッションを閉じます。
    void CloseSession(uint64_t sessionId) NN_NOEXCEPT;
};

//!< 境界の種別を表す列挙体です。
enum BorderType
{
    BorderType_None,    //!< 指定無し
    BorderType_Right,   //!< 右端
    BorderType_Left,    //!< 左端
    BorderType_Top,     //!< 上端
    BorderType_Bottom,  //!< 下端
};

//!< メッセージ種別を表す列挙体です。
enum MessageType
{
    MessageType_Unknown,        //!< 未知のメッセージ
    MessageType_Ping,           //!< 生存確認の要求
    MessageType_Pong,           //!< 生存確認の応答
    MessageType_CaptureStart,   //!< キャプチャの開始
    MessageType_CaptureStop,    //!< キャプチャの停止
    MessageType_KeyDown,        //!< キーボードのキーの押下
    MessageType_KeyUp,          //!< キーボードのキーの開放
    MessageType_MouseMove,      //!< マウスの移動
    MessageType_MouseButton,    //!< マウスのボタン操作
    MessageType_MouseWheel,     //!< マウスのホイール操作
    MessageType_GamePadButton,  //!< ゲームパッドのデジタルボタン操作
    MessageType_GamePadStick,   //!< ゲームパッドのスティック操作
    MessageType_GamePadPower,   //!< ゲームパッドの給電状態
    MessageType_TouchBegan,     //!< タッチの開始
    MessageType_TouchMoved,     //!< タッチの移動
    MessageType_TouchEnded,     //!< タッチの終了
    MessageType_HomeButton,     //!< ホームボタンの操作
    MessageType_CaptureButton,  //!< 撮影ボタンの操作
    MessageType_DebugPadButton, //!< デバッグパッドのデジタルボタン操作
    MessageType_DebugPadStick,  //!< デバッグパッドのスティック操作
    MessageType_DebugPadPower,  //!< デバッグパッドの給電状態
    MessageType_AbstractedPadDeviceData,    //!< AbstractedPad のデバイス情報
    MessageType_AbstractedPadColor,         //!< AbstractedPad のデバイスカラー
    MessageType_AbstractedPadPowerState,    //!< AbstractedPad の電源状態
    MessageType_AbstractedPadButton,        //!< AbstractedPad のボタン
    MessageType_AbstractedPadStick,         //!< AbstractedPad のスティック
};

enum DeviceType
{
    DeviceType_SwitchProController = 0,
    DeviceType_Handheld,
    DeviceType_HandheldJoyLeft,
    DeviceType_HandheldJoyRight,
    DeviceType_JoyConLeft,
    DeviceType_JoyConRight,
    DeviceType_Palma,
    DeviceType_Unknown = 31,
};

//!< メッセージを扱うクラスです。
class Message final
{
private:
    //!< キャプチャのデータを表す構造体です。
    struct CaptureData final
    {
        uint8_t borderType;         //!< 始端の種類
        uint8_t isTriggerdByMouse;  //!< マウス操作によって開始されたか否か
        uint16_t borderLength;      //!< 始端の長さ
        int16_t positionX;          //!< マウスの座標 ( x 軸 )
        int16_t positionY;          //!< マウスの座標 ( y 軸 )
    };

    //!< キーボードのデータを表す構造体です。
    struct KeyboardData final
    {
        uint8_t keyCode;    //!< キーボードのキーコード
        NN_PADDING7;
    };

    //!< マウスのデータを表す構造体です。
    struct MouseData final
    {
        uint8_t buttons;    //!< ボタンの押下状態
        NN_PADDING1;
        int16_t deltaWheel; //!< ホイールの回転差分
        int16_t deltaX;     //!< 座標の移動差分 ( x 軸 )
        int16_t deltaY;     //!< 座標の移動差分 ( y 軸 )
    };

    //!< ゲームパッドのデータを表す構造体です。
    struct GamePadData final
    {
        uint8_t gamePadId;          //!< ゲームパッド識別子
        NN_PADDING1;
        uint8_t powerState;         //!< ゲームパッドのデバイスの電源状態
        uint8_t batteryLevel;       //!< ゲームパッドのバッテリー残量
        uint32_t buttons;           //!< ゲームパッドのボタンの押下状態
    };

    //!< ゲームパッドのスティックのデータを表す構造体です。
    struct GamePadStickData final
    {
        uint8_t gamePadId;          //!< ゲームパッド識別子
        uint8_t stickId;            //!< スティック識別子
        NN_PADDING2;
        int16_t x;                  //!< スティック座標 ( x 軸 )
        int16_t y;                  //!< スティック座標 ( y 軸 )
    };

    //!< タッチのデータを表す構造体です。
    struct TouchData final
    {
        uint8_t fingerId;   //!< タッチ識別子
        NN_PADDING3;
        int16_t x;          //!< タッチ座標 ( x 軸 )
        int16_t y;          //!< タッチ座標 ( y 軸 )
    };

    //!< システムボタンのデータを表す構造体です。
    struct SystemButtonData final
    {
        uint32_t buttons;   //!< システムボタンの押下状態
        NN_PADDING4;
    };

    //!< AbstractedPad のデバイスのデータを表す構造体です。
    struct AbstractedPadDeviceData final
    {
        uint8_t gamePadId;
        uint8_t interfaceType;
        uint8_t deviceType;
        ::nn::Bit8 reserved;
        uint32_t attribute;
    };

    //!< AbstractedPad の色のデータを表す構造体です。
    struct AbstractedPadColorData final
    {
        uint8_t gamePadId;
        uint8_t colorType;
        ::nn::Bit8 reserved[2];
        uint8_t red;
        uint8_t green;
        uint8_t blue;
        uint8_t alpha;
    };

    //!< AbstractedPad の状態のデータを表す構造体です。
    struct AbstractedPadStateData final
    {
        uint8_t gamePadId;
        uint8_t isPowered;
        uint8_t isCharging;
        uint8_t batteryLevel;
        ::nn::Bit8 reserved[4];
    };

    //!< メッセージチャンクを表す構造体です。
    struct Chunk
    {
        uint8_t type;   //!< メッセージ種別
        NN_PADDING3;
        union
        {
            CaptureData captureData;            //!< キャプチャのデータ
            KeyboardData keyboardData;          //!< キーボードのデータ
            MouseData mouseData;                //!< マウスのデータ
            GamePadData gamePadData;            //!< ゲームパッドのデータ
            GamePadStickData gamePadStickData;  //!< スティックのデータ
            TouchData touchData;                //!< タッチのデータ
            SystemButtonData systemButtonData;  //!< システムボタンのデータ
            AbstractedPadDeviceData abstractedPadDeviceData;    //!< デバイスのデータ
            AbstractedPadColorData abstractedPadColorData;      //!< 色のデータ
            AbstractedPadStateData abstractedPadStateData;      //!< 状態のデータ
        };
    };

    //!< メッセージチャンク
    Chunk m_Chunk;

public:
    Message() NN_NOEXCEPT;

    //!< メッセージをリセットします。
    void Reset() NN_NOEXCEPT;

    //!< メッセージ種別を返します。
    MessageType GetType() const NN_NOEXCEPT;

    //!< メッセージ種別を設定します。
    void SetType(MessageType value) NN_NOEXCEPT;

    //!< キャプチャの始端の種類を返します。
    BorderType GetCaptureBorderType() const NN_NOEXCEPT;

    //!< キャプチャがマウスによって開始されたか否かを表す値を返します。
    bool IsCaptureTriggerdByMouse() const NN_NOEXCEPT;

    //!< キャプチャの始端の長さを返します。
    uint16_t GetCaptureBorderLength() const NN_NOEXCEPT;

    //!< キャプチャのマウスの座標 ( x 軸 ) を返します。
    int16_t GetCapturePositionX() const NN_NOEXCEPT;

    //!< キャプチャのマウスの座標 ( y 軸 ) を返します。
    int16_t GetCapturePositionY() const NN_NOEXCEPT;

    //!< キャプチャのマウスの座標を設定します。
    void SetCapturePosition(int16_t x, int16_t y) NN_NOEXCEPT;

    //!< キーボードのキーコードを返します。
    int GetKeyboardKeyCode() const NN_NOEXCEPT;

    //!< キーボードのキーコードを設定します。
    void SetKeyboardKeyCode(int value) NN_NOEXCEPT;

    //!< マウスのボタンの押下状態を返します。
    ::nn::hid::MouseButtonSet GetMouseButtons() const NN_NOEXCEPT;

    //!< マウスのボタンの押下状態を設定します。
    void SetMouseButtons(::nn::hid::MouseButtonSet value) NN_NOEXCEPT;

    //!< マウスのホイールの回転差分を返します。
    int16_t GetMouseDeltaWheel() const NN_NOEXCEPT;

    //!< マウスのホイールの回転差分を設定します。
    void SetMouseDeltaWheel(int16_t value) NN_NOEXCEPT;

    //!< マウスの座標の移動差分 ( x 軸 ) を返します。
    int16_t GetMouseDeltaPositionX() const NN_NOEXCEPT;

    //!< マウスの座標の移動差分 ( y 軸 ) を返します。
    int16_t GetMouseDeltaPositionY() const NN_NOEXCEPT;

    //!< マウスの座標の移動差分を設定します。
    void SetMouseDeltaPosition(int16_t x, int16_t y) NN_NOEXCEPT;

    //!< ゲームパッド識別子を返します。
    uint8_t GetGamePadId() const NN_NOEXCEPT;

    //!< ゲームパッド識別子を設定します。
    void SetGamePadId(uint8_t value) NN_NOEXCEPT;

    //!< ゲームパッドの電源状態を返します。
    ::nn::hid::PowerState GetGamePadPowerState() const NN_NOEXCEPT;

    //!< ゲームパッドの電源状態を設定します。
    void SetGamePadPowerState(::nn::hid::PowerState value) NN_NOEXCEPT;

    //!< ゲームパッドのバッテリー残量を返します。
    ::nn::hid::BatteryLevel GetGamePadBatteryLevel() const NN_NOEXCEPT;

    //!< ゲームパッドのバッテリー残量を設定します。
    void SetGamePadBatteryLevel(::nn::hid::BatteryLevel value) NN_NOEXCEPT;

    //!< ゲームパッドのボタンの押下状態を返します。
    ::nn::hid::BasicXpadButtonSet GetGamePadButtons() const NN_NOEXCEPT;

    //!< ゲームパッドのボタンの押下状態を設定します。
    void SetGamePadButtons(::nn::hid::BasicXpadButtonSet value) NN_NOEXCEPT;

    //!< ゲームパッドのスティック識別子を返します。
    uint8_t GetGamePadStickId() const NN_NOEXCEPT;

    //!< ゲームパッドのスティック識別子を返します。
    void SetGamePadStickId(uint8_t value) NN_NOEXCEPT;

    //!< ゲームパッドのスティックの座標 ( x 軸 ) を返します。
    int16_t GetGamePadStickPositionX() const NN_NOEXCEPT;

    //!< ゲームパッドのスティックの座標 ( y 軸 ) を返します。
    int16_t GetGamePadStickPositionY() const NN_NOEXCEPT;

    //!< ゲームパッドのスティックの座標を設定します。
    void SetGamePadStickPosition(int16_t x, int16_t y) NN_NOEXCEPT;

    //!< タッチの識別子を返します。
    uint8_t GetTouchFingerId() const NN_NOEXCEPT;

    //!< タッチの識別子を設定します。
    void SetTouchFingerId(uint8_t value) NN_NOEXCEPT;

    //!< タッチの座標 ( x 軸 ) を返します。
    int16_t GetTouchPositionX() const NN_NOEXCEPT;

    //!< タッチの座標 ( y 軸 ) を返します。
    int16_t GetTouchPositionY() const NN_NOEXCEPT;

    //!< タッチの座標を設定します。
    void SetTouchPosition(int16_t x, int16_t y) NN_NOEXCEPT;

    //!< ホームボタンの押下状態を返します。
    ::nn::hid::system::HomeButtonSet GetHomeButtons() const NN_NOEXCEPT;

    //!< ホームボタンの押下状態を設定します。
    void SetHomeButtons(::nn::hid::system::HomeButtonSet value) NN_NOEXCEPT;

    //!< 撮影ボタンの押下状態を返します。
    ::nn::hid::system::CaptureButtonSet GetCaptureButtons() const NN_NOEXCEPT;

    //!< 撮影ボタンの押下状態を設定します。
    void SetCaptureButtons(
        ::nn::hid::system::CaptureButtonSet value) NN_NOEXCEPT;

    //!< デバッグパッドの接続状態を返します。
    bool GetDebugPadConnectionState() const NN_NOEXCEPT;

    //!< デバッグパッドの接続状態を設定します。
    void SetDebugPadConnectionState(bool value) NN_NOEXCEPT;

    //!< デバッグパッドのボタンの押下状態を返します。
    ::nn::hid::DebugPadButtonSet GetDebugPadButtons() const NN_NOEXCEPT;

    //!< デバッグパッドのボタンの押下状態を設定します。
    void SetDebugPadButtons(::nn::hid::DebugPadButtonSet value) NN_NOEXCEPT;

    //!< AbstractedPad の id を返します。
    uint8_t GetAbstractedPadId() const NN_NOEXCEPT;

    //!< AbstractedPad のデバイスタイプを返します。
    ::nn::hid::system::DeviceTypeSet GetAbstractedPadDeviceType() const NN_NOEXCEPT;

    //!< AbstractedPad の接続インターフェースを返します。
    ::nn::hid::system::InterfaceType GetAbstractedPadInterfaceType() const NN_NOEXCEPT;

    //!< AbstractedPad のカラータイプを返します。
    uint8_t GetAbstractedPadColorType() const NN_NOEXCEPT;

    //!< AbstractedPad のカラーを返します。
    ::nn::util::Color4u8Type  GetAbstractedPadColor() const NN_NOEXCEPT;

    //!< AbstractedPad の Attribute フラグセットを返します。
    ::nn::hid::debug::AbstractedPadAttributeSet GetAbstractedPadAttributes() const NN_NOEXCEPT;

    //!< AbstractedPad の Attribute フラグセットを返します。
    ::nn::hid::system::PowerInfo GetAbstractedPadPowerInfo() const NN_NOEXCEPT;

    //!< AbstractedPad のボタン押下状態を返します。
    ::nn::hid::detail::AbstractedPadButtonSet GetAbstractedPadButtons() const NN_NOEXCEPT;

    // AbstractedPad の ID を設定します。
    void SetAbstractedPadId(uint8_t value) NN_NOEXCEPT;

    // AbstractedPad の属性を設定します。
    void SetAbstractedPadAttribute(::nn::hid::debug::AbstractedPadAttributeSet value) NN_NOEXCEPT;

    // AbstractedPad の電源状態を設定します。
    void SetAbstractedPadPowerInfo(::nn::hid::system::PowerInfo value) NN_NOEXCEPT;

    // AbstractedPad のブタン押下状態を設定します。
    void SetAbstractedPadButtons(::nn::hid::detail::AbstractedPadButtonSet value) NN_NOEXCEPT;

    // AbstractedPad のスティック識別子を設定します。
    void SetAbstractedPadStickId(uint8_t value) NN_NOEXCEPT;

    // AbstractedPad のスティックの座標を設定します。
    void SetAbstractedPadStickPosition(int16_t x, int16_t y) NN_NOEXCEPT;

    // AbstractedPad のデバイスタイプを設定します。
    void SetAbstractedPadDeviceType(::nn::hid::system::DeviceTypeSet value) NN_NOEXCEPT;

    // AbstractedPad のインターフェースタイプを設定します。
    void SetAbstractedPadInterfaceType(::nn::hid::system::InterfaceType value) NN_NOEXCEPT;

    // AbstractedPad の色の識別子を設定します。
    void SetAbstractedPadColorType(uint8_t value) NN_NOEXCEPT;

    // AbstractedPad の色を設定します。
    void SetAbstractedPadColorData(nn::util::Color4u8Type value) NN_NOEXCEPT;

    //!< ソケットチャネルからメッセージを読み出します。
    ::nn::htcs::SocketError ReadFrom(SocketChannel& channel) NN_NOEXCEPT;

    //!< ソケットチャネルへメッセージを書き出します。
    ::nn::htcs::SocketError WriteTo(SocketChannel& channel) NN_NOEXCEPT;
};

//!< Ping-Pong サーバを扱うクラスです。
class PingPongServer final
{
    NN_DISALLOW_COPY(PingPongServer);
    NN_DISALLOW_MOVE(PingPongServer);

private:
    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< 応答待ちの生存確認の数
    int m_RequestCount;

public:
    PingPongServer() NN_NOEXCEPT;

    //!< リセットします。
    void Reset() NN_NOEXCEPT;

    //!< メッセージを受理します。
    bool AcceptMessage(const Message& message) NN_NOEXCEPT;

    //!< メッセージを取得します。
    bool AcquireMessage(Message* pOutValue) NN_NOEXCEPT;

private:
    //!< 生存確認の要求メッセージを受理します。
    void AcceptPingMessage(const Message& message) NN_NOEXCEPT;

    //!< 生存確認の応答メッセージを取得します。
    bool AcquirePongMessage(Message* pOutValue) NN_NOEXCEPT;
};

//!< Input Director を扱うクラスです。
class InputDirector final
{
    NN_DISALLOW_COPY(InputDirector);
    NN_DISALLOW_MOVE(InputDirector);

private:
    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< ディスプレイの幅
    int32_t m_DisplayWidth;

    //!< ディスプレイの高さ
    int32_t m_DisplayHeight;

    //!< キャプチャ中か否か
    bool m_IsCapturing;

    //!< キャプチャを停止する必要があるか否か
    bool m_NeedsToStopCapture;

    //!< マウス操作によるキャプチャの開始端の種別
    BorderType m_CaptureBorderType;

    //!< マウス操作によるキャプチャの開始端の長さ
    int32_t m_CaptureBorderLength;

    //!< キーボードが初期化状態にあるか否か
    bool m_IsKeyboardInitialized;

    //!< キーボードの自動操作状態
    ::nn::hid::debug::KeyboardAutoPilotState m_KeyboardState;

    //!< マウスが初期化状態にあるか否か
    bool m_IsMouseInitialized;

    //!< マウスの自動操作状態
    ::nn::hid::debug::MouseAutoPilotState m_MouseState;

    //!< ゲームパッドが初期化状態にあるか否か
    bool m_IsGamePadInitialized;

    //!< ゲームパッドの識別子の数
    int m_GamePadIdCount;

    //!< ゲームパッドの識別子
    ::nn::hid::BasicXpadId m_GamePadIds[::nn::hid::XpadIdCountMax];

    //!< ゲームパッドの識別子
    ::nn::hid::debug::BasicXpadAutoPilotState m_GamePadStates[
        ::nn::hid::XpadIdCountMax];

    //!< タッチパネルが初期化状態にあるか否か
    bool m_IsTouchScreenInitialized;

    //!< タッチパネルの状態
    ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax
        > m_TouchScreenState;

    //!< タッチパネルが自動操作状態にあるか否か
    bool m_IsTouchScreenAutoPilotEnabled;

    //!< タッチパネルの自動操作状態
    ::nn::hid::debug::TouchScreenAutoPilotState<::nn::hid::TouchStateCountMax
        > m_TouchScreenAutoPilotState;

    //!< タッチに関するメッセージの位置
    size_t m_TouchMessageIndex;

    //!< タッチに関するメッセージの数
    size_t m_TouchMessageCount;

    //!< タッチに関するメッセージ
    Message m_TouchMessages[2 * ::nn::hid::TouchStateCountMax];

    //!< ホームボタンが初期化状態にあるか否か
    bool m_IsHomeButtonInitialized;

    //!< ホームボタンが自動操作状態にあるか否か
    bool m_IsHomeButtonAutoPilotEnabled;

    //!< ホームボタンの自動操作状態
    ::nn::hid::debug::HomeButtonAutoPilotState m_HomeButtonAutoPilotState;

    //!< 撮影ボタンが初期化状態にあるか否か
    bool m_IsCaptureButtonInitialized;

    //!< 撮影ボタンが自動操作状態にあるか否か
    bool m_IsCaptureButtonAutoPilotEnabled;

    //!< 撮影ボタンの自動操作状態
    ::nn::hid::debug::CaptureButtonAutoPilotState m_CaptureButtonAutoPilotState;

    //!< デバッグパッドが初期化状態にあるか否か
    bool m_IsDebugPadInitialized;

    //!< デバッグパッドの状態
    ::nn::hid::DebugPadState m_DebugPadState;

    //!< デバッグパッドが自動操作状態にあるか否か
    bool m_IsDebugPadAutoPilotEnabled;

    //!< デバッグパッドの自動操作状態
    ::nn::hid::debug::DebugPadAutoPilotState m_DebugPadAutoPilotState;

    //!< デバッグパッドに関するメッセージの位置
    size_t m_DebugPadMessageIndex;

    //!< デバッグパッドに関するメッセージの数
    size_t m_DebugPadMessageCount;

    //!< デバッグパッドに関するメッセージ
    Message m_DebugPadMessages[1 + 1 + 2];

    //!< AbstractedPad が初期化状態にあるか否か
    bool m_IsAbstractedPadInitialized;

    //!< AbstractedPad の自動操作状態
    ::nn::hid::debug::AbstractedPadState m_AbstractedPadAutoPilotStates[
        ::nn::hid::debug::AbstractedVirtualPadIdCountMax];

    struct AbstractedPadInfo
    {
        bool isEnabled;
        ::nn::hid::debug::AbstractedPadHandle handle;
        ::nn::hid::debug::AbstractedPadState state;
    };

    //!< AbstractedPad に関する情報
    AbstractedPadInfo m_AbstractedPadInfo[::nn::hid::debug::AbstractedPadHandleCountMax];

    //!< AbstractedPad に関するメッセージの位置
    size_t m_AbstractedPadMessageIndex;

    //!< AbstractedPad に関するメッセージの数
    size_t m_AbstractedPadMessageCount;

    // 1つのindexにつき最大で以下全てが送られる
    // MessageType_AbstractedPadDeviceData * 2 (接続、切断）
    // MessageType_AbstractedPadColor * 2
    // MessageType_AbstractedPadPowerState
    // MessageType_AbstractedPadButton
    // MessageType_AbstractedPadStick * 2

    //!< AbstractedPad に関するメッセージ
    Message m_AbstractedPadMessages[8 * ::nn::hid::debug::AbstractedPadHandleCountMax];


public:
    InputDirector() NN_NOEXCEPT;

    //!< ディスプレイのサイズを設定します。
    void SetDisplaySize(int32_t width, int32_t height) NN_NOEXCEPT;

    //!< リセットします。
    void Reset() NN_NOEXCEPT;

    //!< メッセージを受理します。
    bool AcceptMessage(const Message& message) NN_NOEXCEPT;

    //!< メッセージを用意します。
    void PrepareMessage() NN_NOEXCEPT;

    //!< メッセージを取得します。
    bool AcquireMessage(Message* pOutValue) NN_NOEXCEPT;

private:
    //!< キャプチャをリセットします。
    void ResetCapture() NN_NOEXCEPT;

    //!< キャプチャの開始メッセージを受理します。
    void AcceptCaptureStartMessage(const Message& message) NN_NOEXCEPT;

    //!< キャプチャの停止メッセージを受理します。
    void AcceptCaptureStopMessage(const Message& message) NN_NOEXCEPT;

    //!< キーボードのキーに関するメッセージを受理します。
    void AcceptKeyMessage(
        const Message& message,
        ::nn::hid::debug::KeyboardAutoPilotState(*function)(
            const Message&,
            const ::nn::hid::debug::KeyboardAutoPilotState&) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    //!< キーボードのキーの押下メッセージを受理します。
    void AcceptKeyDownMessage(const Message& message) NN_NOEXCEPT;

    //!< キーボードのキーの開放メッセージを受理します。
    void AcceptKeyUpMessage(const Message& message) NN_NOEXCEPT;

    //!< マウスの移動メッセージを受理します。
    void AcceptMouseMoveMessage(const Message& message) NN_NOEXCEPT;

    //!< マウスのボタン操作メッセージを受理します。
    void AcceptMouseButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< マウスのホイール操作メッセージを受理します。
    void AcceptMouseWheelMessage(const Message& message) NN_NOEXCEPT;

    //!< ゲームパッドに関するメッセージを受理します。
    void AcceptGamePadMessage(
        const Message& message,
        ::nn::hid::debug::BasicXpadAutoPilotState(*function)(
            const Message&,
            const ::nn::hid::debug::BasicXpadAutoPilotState&) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    //!< ゲームパッドのデジタルボタン操作メッセージを受理します。
    void AcceptGamePadButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< ゲームパッドのスティック操作メッセージを受理します。
    void AcceptGamePadStickMessage(const Message& message) NN_NOEXCEPT;

    //!< ゲームパッドの給電状態メッセージを受理します。
    void AcceptGamePadPowerMessage(const Message& message) NN_NOEXCEPT;

    //!< タッチに関するメッセージを受理します。
    void AcceptTouchMessage(
        const Message& message,
        ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax>(*function)(
                const Message&,
                const ::nn::hid::debug::TouchScreenAutoPilotState<
                    ::nn::hid::TouchStateCountMax>&) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    //!< タッチの開始メッセージを受理します。
    void AcceptTouchBeganMessage(const Message& message) NN_NOEXCEPT;

    //!< タッチの移動メッセージを受理します。
    void AcceptTouchMovedMessage(const Message& message) NN_NOEXCEPT;

    //!< タッチの終了メッセージを受理します。
    void AcceptTouchEndedMessage(const Message& message) NN_NOEXCEPT;

    //!< ホームボタン操作メッセージを受理します。
    void AcceptHomeButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< 撮影ボタン操作メッセージを受理します。
    void AcceptCaptureButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< デバッグパッドに関するメッセージを受理します。
    void AcceptDebugPadMessage(
        const Message& message,
        ::nn::hid::debug::DebugPadAutoPilotState(*function)(
            const Message&,
            const ::nn::hid::debug::DebugPadAutoPilotState&) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    //!< デバッグパッドのデジタルボタン操作メッセージを受理します。
    void AcceptDebugPadButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< デバッグパッドのスティック操作メッセージを受理します。
    void AcceptDebugPadStickMessage(const Message& message) NN_NOEXCEPT;

    //!< デバッグパッドの給電状態メッセージを受理します。
    void AcceptDebugPadPowerMessage(const Message& message) NN_NOEXCEPT;

    //!< AbstractedPad に関するメッセージを受理します。
    void AcceptAbstractedPadMessage(
        const Message& message,
        ::nn::hid::debug::AbstractedPadState(*function)(
            const Message&,
            const ::nn::hid::debug::AbstractedPadState&) NN_NOEXCEPT
        ) NN_NOEXCEPT;

    //!< AbstractedPad のデバイスタイプメッセージを受理します。
    void AcceptAbstractedPadDeviceDataMessage(const Message& message) NN_NOEXCEPT;

    //!< AbstractedPad のメインカラーメッセージを受理します。
    void AcceptAbstractedPadColorMessage(const Message& message) NN_NOEXCEPT;

    //!< AbstractedPad の電源状態メッセージを受理します。
    void AcceptAbstractedPadPowerStateMessage(const Message& message) NN_NOEXCEPT;

    //!< AbstractedPad のボタンメッセージを受理します。
    void AcceptAbstractedPadButtonMessage(const Message& message) NN_NOEXCEPT;

    //!< AbstractedPad のスティックメッセージを受理します。
    void AcceptAbstractedPadStickMessage(const Message& message) NN_NOEXCEPT;

    //!< タッチのメッセージを用意します。
    void PrepareTouchMessage() NN_NOEXCEPT;

    //!< デバッグパッドのメッセージを用意します。
    void PrepareDebugPadMessage() NN_NOEXCEPT;

    //!< 抽象パッドに関するメッセージを用意します。
    void PrepareAbstractedPadMessage() NN_NOEXCEPT;

    //!< 抽象パッドの更新メッセージを用意します。
    void PrepareUpdatingAbstractedPadMessage(
        int id,
        const ::nn::hid::debug::AbstractedPadState& prevState,
        const ::nn::hid::debug::AbstractedPadState& nextState) NN_NOEXCEPT;

    //!< 抽象パッドを追加します
    void PrepareNewAbstractedPadMessage(
        int id, const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT;

    //!< キャプチャの停止メッセージを取得します。
    bool AcquireCaptureStopMessage(Message* pOutValue) NN_NOEXCEPT;

    //!< タッチに関するメッセージを取得します。
    bool AcquireTouchMessage(Message* pOutValue) NN_NOEXCEPT;

    //!< デバッグパッドに関するメッセージを取得します。
    bool AcquireDebugPadMessage(Message* pOutValue) NN_NOEXCEPT;

    //!< AbstractedPad に関するメッセージを取得します。
    bool AcquireAbstractedPadMessage(Message* pOutValue) NN_NOEXCEPT;
};

//!< HTCS API の返すエラー値が成功を意味するか否かを表す値を返します。
NN_FORCEINLINE
bool IsSuccess(::nn::htcs::SocketError error) NN_NOEXCEPT
{
    return (error == ::nn::htcs::HTCS_ENONE);
}

//!< メッセージの書き込みを扱うクラスです。
template<class T>
class MessageWriter final
{
    NN_DISALLOW_COPY(MessageWriter);
    NN_DISALLOW_MOVE(MessageWriter);

private:
    //!< ミューテックス
    ::nn::os::Mutex* m_pMutex;

    //!< ソケットチャネル
    SocketChannel*  m_pSocketChannel;

    //!< メッセージソース
    T* m_pSource;

public:
    MessageWriter() NN_NOEXCEPT
        : m_pMutex(nullptr)
        , m_pSocketChannel(nullptr)
        , m_pSource(nullptr)
    {
        // 何もしない
    }

    //!< ミューテックスを設定します。
    void SetMutex(::nn::os::Mutex* pMutex) NN_NOEXCEPT
    {
        m_pMutex = pMutex;
    }

    //!< ソケットチャネルを設定します。
    void SetSocketChannel(SocketChannel* pSocketChannel) NN_NOEXCEPT
    {
        m_pSocketChannel = pSocketChannel;
    }

    //!< メッセージソースを設定します。
    void SetSource(T* pSource) NN_NOEXCEPT
    {
        m_pSource = pSource;
    }

    //!< メッセージソース上のデータをチャネルへ書き出します。
    ::nn::htcs::SocketError Flush() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pMutex);
        NN_SDK_REQUIRES_NOT_NULL(m_pSocketChannel);
        NN_SDK_REQUIRES_NOT_NULL(m_pSource);

        while (NN_STATIC_CONDITION(true))
        {
            SocketChannel channel;

            Message message;

            {
                ::std::lock_guard<decltype(*m_pMutex)> locker(*m_pMutex);

                channel = *m_pSocketChannel;

                if (!m_pSource->AcquireMessage(&message))
                {
                    NN_HTCS_SOCKET_ERROR_NONE;
                }
            }

            NN_HTCS_SOCKET_ERROR_DO(message.WriteTo(channel));
        }
    }
};

//!< 指定された型のオブジェクトを配置するためのメモリ領域を表す構造体です。
template<class T>
using Storage = ::nn::util::TypedStorage<T, sizeof(T), NN_ALIGNOF(T)>;

//!< スレッドの関数エントリの引数を表す構造体です。
struct ThreadArgs final
{
    ::nn::os::Mutex* pMutex;            //!< ミューテックス
    ::nn::os::Event* pSocketEvent;      //!< ソケットイベント
    SocketChannel* pSocketChannel;      //!< ソケットチャネル
    PingPongServer* pPingPongServer;    //!< Ping-Pong サーバ
    InputDirector* pInputDirector;      //!< Input Director
};

//!< レシーバスレッドの関数エントリ
void ReceiverThreadFunc(void* arg) NN_NOEXCEPT;

//!< センダスレッドの関数エントリ
void SenderThreadFunc(void* arg) NN_NOEXCEPT;

} // namespace

void InitializeHidServer() NN_NOEXCEPT
{
    static Storage<::nn::os::Mutex> s_MutexStorage;
    ::nn::os::Mutex* pMutex = &::nn::util::Get(s_MutexStorage);
    new(pMutex) ::nn::os::Mutex(false);

    static Storage<::nn::os::Event> s_SocketEventStorage;
    ::nn::os::Event* pSocketEvent = &::nn::util::Get(s_SocketEventStorage);
    new(pSocketEvent) ::nn::os::Event(::nn::os::EventClearMode_AutoClear);

    static Storage<SocketChannel> s_SocketChannelStorage;
    SocketChannel* pSocketChannel = &::nn::util::Get(s_SocketChannelStorage);
    new(pSocketChannel) SocketChannel();

    static Storage<PingPongServer> s_PingPongServerStorage;
    PingPongServer* pPingPongServer = &::nn::util::Get(s_PingPongServerStorage);
    new(pPingPongServer) PingPongServer();

    static Storage<InputDirector> s_InputDirectorStorage;
    InputDirector* pInputDirector = &::nn::util::Get(s_InputDirectorStorage);
    new(pInputDirector) InputDirector();
    pInputDirector->SetDisplaySize(DisplayWidth, DisplayHeight);

    static ThreadArgs s_Args = {};
    s_Args.pMutex = pMutex;
    s_Args.pSocketEvent = pSocketEvent;
    s_Args.pSocketChannel = pSocketChannel;
    s_Args.pPingPongServer = pPingPongServer;
    s_Args.pInputDirector = pInputDirector;

    static ::nn::os::ThreadType s_ReceiverThread = {};
    NN_ALIGNAS(::nn::os::ThreadStackAlignment)
    static char s_ReceiverThreadStack[ReceiverThreadStackSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::CreateThread(&s_ReceiverThread,
                               &ReceiverThreadFunc,
                               &s_Args,
                               reinterpret_cast<void*>(s_ReceiverThreadStack),
                               sizeof(s_ReceiverThreadStack),
                               NN_SYSTEM_THREAD_PRIORITY(cs, HidReceiver)));
    ::nn::os::SetThreadNamePointer(
        &s_ReceiverThread, NN_SYSTEM_THREAD_NAME(cs, HidReceiver));
    ::nn::os::StartThread(&s_ReceiverThread);

    static ::nn::os::ThreadType s_SenderThread = {};
    NN_ALIGNAS(::nn::os::ThreadStackAlignment)
    static char s_SenderThreadStack[SenderThreadStackSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::CreateThread(&s_SenderThread,
                               &SenderThreadFunc,
                               &s_Args,
                               reinterpret_cast<void*>(s_SenderThreadStack),
                               sizeof(s_SenderThreadStack),
                               NN_SYSTEM_THREAD_PRIORITY(cs, HidSender)));
    ::nn::os::SetThreadNamePointer(
        &s_SenderThread, NN_SYSTEM_THREAD_NAME(cs, HidSender));
    ::nn::os::StartThread(&s_SenderThread);
}

namespace {

SocketChannel::SocketChannel() NN_NOEXCEPT
    : m_pParent(nullptr)
    , m_SessionId(0)
{
    // 何もしない
}

SocketChannel::SocketChannel(Socket* pParent, uint64_t sessionId) NN_NOEXCEPT
    : m_pParent(pParent)
    , m_SessionId(sessionId)
{
    NN_SDK_REQUIRES_NOT_NULL(pParent);
}

::nn::htcs::SocketError SocketChannel::Receive(size_t* pOutSize,
                                               void* buffer,
                                               size_t bufferSize,
                                               int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pParent);

    return m_pParent->Receive(pOutSize, buffer, bufferSize, flags, m_SessionId);
}

::nn::htcs::SocketError SocketChannel::Send(size_t* pOutSize,
                                            const void* buffer,
                                            size_t bufferSize,
                                            int flags) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pParent);

    return m_pParent->Send(pOutSize, buffer, bufferSize, flags, m_SessionId);
}

void SocketChannel::Close() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pParent);

    m_pParent->CloseSession(m_SessionId);
}

Socket::Socket() NN_NOEXCEPT
    : m_Mutex(false)
    , m_SockAddrHtcs()
    , m_Socket(InvalidSocket)
    , m_SubSocket(InvalidSocket)
    , m_SessionId(0)
{
    // 何もしない
}

void Socket::SetPortName(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    ::nn::util::Strlcpy(
        m_SockAddrHtcs.portName.name,
        name,
        ::std::extent<decltype(m_SockAddrHtcs.portName.name)>::value);
}

::nn::htcs::SocketError Socket::CreateSocketChannel(SocketChannel* pOutValue
                                                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    auto sessionId = uint64_t();
    auto socket = int();
    NN_HTCS_SOCKET_ERROR_DO(this->AcquireSession(&sessionId, &socket));

    SocketChannel channel(this, sessionId);
    *pOutValue = channel;

    NN_HTCS_SOCKET_ERROR_NONE;
}

::nn::htcs::SocketError Socket::GetLastError() const NN_NOEXCEPT
{
    return static_cast<::nn::htcs::SocketError>(::nn::htcs::GetLastError());
}

::nn::htcs::SocketError Socket::AcquireSession(uint64_t* pOutSessionId,
                                               int* pOutSocket) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSessionId);
    NN_SDK_REQUIRES_NOT_NULL(pOutSocket);

    if (m_Socket == InvalidSocket)
    {
        auto needsRollback = true;

        int socket = ::nn::htcs::Socket();

        if (socket == InvalidSocket)
        {
            return this->GetLastError();
        }

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                ::nn::htcs::Close(socket);
            }
        };

        m_SockAddrHtcs.family = ::nn::htcs::HTCS_AF_HTCS;
        m_SockAddrHtcs.peerName = ::nn::htcs::GetPeerNameAny();
        if (::nn::htcs::Bind(socket, &m_SockAddrHtcs) != 0)
        {
            return this->GetLastError();
        }

        if (::nn::htcs::Listen(socket, 1) != 0)
        {
            return this->GetLastError();
        }

        needsRollback = false;

        m_Socket = socket;

        m_SessionId = m_SessionId + 1;
    }

    *pOutSessionId = m_SessionId;

    *pOutSocket = m_Socket;

    NN_HTCS_SOCKET_ERROR_NONE;
}

::nn::htcs::SocketError Socket::AcquireSession(uint64_t* pOutSessionId,
                                               int* pOutSocket,
                                               int* pOutSubSocket) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSessionId);
    NN_SDK_REQUIRES_NOT_NULL(pOutSocket);
    NN_SDK_REQUIRES_NOT_NULL(pOutSubSocket);

    auto sessionId = uint64_t();
    auto socket = int();
    NN_HTCS_SOCKET_ERROR_DO(this->AcquireSession(&sessionId, &socket));

    if (m_SubSocket == InvalidSocket)
    {
        ::nn::htcs::SockAddrHtcs sockAddrHtcs = {};

        int subSocket = ::nn::htcs::Accept(socket, &sockAddrHtcs);

        if (subSocket == InvalidSocket)
        {
            const ::nn::htcs::SocketError error = this->GetLastError();

            ::nn::htcs::Close(socket);

            m_Socket = InvalidSocket;

            return error;
        }

        m_SubSocket = subSocket;
    }

    *pOutSessionId = sessionId;

    *pOutSocket = socket;

    *pOutSubSocket = m_SubSocket;

    NN_HTCS_SOCKET_ERROR_NONE;
}

void Socket::CloseSession(uint64_t sessionId) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_SessionId == sessionId)
    {
        if (m_SubSocket != InvalidSocket)
        {
            ::nn::htcs::Close(m_SubSocket);

            m_SubSocket = InvalidSocket;
        }

        if (m_Socket != InvalidSocket)
        {
            ::nn::htcs::Close(m_Socket);

            m_Socket = InvalidSocket;
        }
    }
}

::nn::htcs::SocketError Socket::Receive(size_t* pOutSize,
                                        void* buffer,
                                        size_t bufferSize,
                                        int flags,
                                        uint64_t sessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    auto subSocket = int();

    {
        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        auto id = uint64_t();
        auto socket = int();
        NN_HTCS_SOCKET_ERROR_DO(this->AcquireSession(&id, &socket, &subSocket));

        if (id != sessionId)
        {
            return ::nn::htcs::HTCS_ENOTCONN;
        }
    }

    auto size = static_cast<int>(
        ::nn::htcs::Recv(subSocket, buffer, bufferSize, flags));

    if (size == 0)
    {
        this->CloseSession(sessionId);

        return ::nn::htcs::HTCS_ENOTCONN;
    }
    else if (size < 0)
    {
        const ::nn::htcs::SocketError error = this->GetLastError();

        this->CloseSession(sessionId);

        return error;
    }

    *pOutSize = static_cast<size_t>(size);

    NN_HTCS_SOCKET_ERROR_NONE;
}

::nn::htcs::SocketError Socket::Send(size_t* pOutSize,
                                     const void* buffer,
                                     size_t bufferSize,
                                     int flags,
                                     uint64_t sessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    auto subSocket = int();

    {
        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        auto id = uint64_t();
        auto socket = int();
        NN_HTCS_SOCKET_ERROR_DO(this->AcquireSession(&id, &socket, &subSocket));

        if (id != sessionId)
        {
            return ::nn::htcs::HTCS_ENOTCONN;
        }
    }

    auto size = static_cast<int>(
        ::nn::htcs::Send(subSocket, buffer, bufferSize, flags));

    if (size < 0)
    {
        const ::nn::htcs::SocketError error = this->GetLastError();

        this->CloseSession(sessionId);

        return error;
    }

    *pOutSize = static_cast<size_t>(size);

    NN_HTCS_SOCKET_ERROR_NONE;
}

Message::Message() NN_NOEXCEPT
    : m_Chunk()
{
    m_Chunk.type = static_cast<uint8_t>(MessageType_Unknown);
}

void Message::Reset() NN_NOEXCEPT
{
    m_Chunk = Chunk();
}

MessageType Message::GetType() const NN_NOEXCEPT
{
    return static_cast<MessageType>(m_Chunk.type);
}

void Message::SetType(MessageType value) NN_NOEXCEPT
{
    m_Chunk.type = static_cast<uint8_t>(value);
}

BorderType Message::GetCaptureBorderType() const NN_NOEXCEPT
{
    return static_cast<BorderType>(m_Chunk.captureData.borderType);
}

bool Message::IsCaptureTriggerdByMouse() const NN_NOEXCEPT
{
    return (m_Chunk.captureData.isTriggerdByMouse != 0);
}

uint16_t Message::GetCaptureBorderLength() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.captureData.borderLength);
}

int16_t Message::GetCapturePositionX() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.captureData.positionX);
}

int16_t Message::GetCapturePositionY() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.captureData.positionY);
}

void Message::SetCapturePosition(int16_t x, int16_t y) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.captureData.positionX, x);
    ::nn::util::StoreBigEndian(&m_Chunk.captureData.positionY, y);
}

int Message::GetKeyboardKeyCode() const NN_NOEXCEPT
{
    return static_cast<int>(m_Chunk.keyboardData.keyCode);
}

void Message::SetKeyboardKeyCode(int value) NN_NOEXCEPT
{
    m_Chunk.keyboardData.keyCode = static_cast<uint8_t>(value);
}

::nn::hid::MouseButtonSet Message::GetMouseButtons() const NN_NOEXCEPT
{
    ::nn::hid::MouseButtonSet buttons = {};
    buttons._storage[0] = m_Chunk.mouseData.buttons;
    return buttons;
}

void Message::SetMouseButtons(::nn::hid::MouseButtonSet value) NN_NOEXCEPT
{
    m_Chunk.mouseData.buttons = static_cast<uint8_t>(value._storage[0]);
}

int16_t Message::GetMouseDeltaWheel() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.mouseData.deltaWheel);
}

void Message::SetMouseDeltaWheel(int16_t value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.mouseData.deltaWheel, value);
}

int16_t Message::GetMouseDeltaPositionX() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.mouseData.deltaX);
}

int16_t Message::GetMouseDeltaPositionY() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.mouseData.deltaY);
}

void Message::SetMouseDeltaPosition(int16_t x, int16_t y) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.mouseData.deltaX, x);
    ::nn::util::StoreBigEndian(&m_Chunk.mouseData.deltaY, y);
}

uint8_t Message::GetGamePadId() const NN_NOEXCEPT
{
    return m_Chunk.gamePadData.gamePadId;
}

void Message::SetGamePadId(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.gamePadData.gamePadId = value;
}

::nn::hid::PowerState Message::GetGamePadPowerState() const NN_NOEXCEPT
{
    return static_cast<::nn::hid::PowerState>(m_Chunk.gamePadData.powerState);
}

void Message::SetGamePadPowerState(::nn::hid::PowerState value) NN_NOEXCEPT
{
    m_Chunk.gamePadData.powerState = static_cast<uint8_t>(value);
}

::nn::hid::BatteryLevel Message::GetGamePadBatteryLevel() const NN_NOEXCEPT
{
    return static_cast<
        ::nn::hid::BatteryLevel>(m_Chunk.gamePadData.batteryLevel);
}

void Message::SetGamePadBatteryLevel(::nn::hid::BatteryLevel value) NN_NOEXCEPT
{
    m_Chunk.gamePadData.batteryLevel = static_cast<uint8_t>(value);
}

::nn::hid::BasicXpadButtonSet Message::GetGamePadButtons() const NN_NOEXCEPT
{
    ::nn::hid::BasicXpadButtonSet buttons = {};
    buttons._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.gamePadData.buttons);
    return buttons;
}

void Message::SetGamePadButtons(::nn::hid::BasicXpadButtonSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadData.buttons, value._storage[0]);
}

uint8_t Message::GetGamePadStickId() const NN_NOEXCEPT
{
    return m_Chunk.gamePadStickData.stickId;
}

void Message::SetGamePadStickId(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.gamePadStickData.stickId = value;
}

int16_t Message::GetGamePadStickPositionX() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.gamePadStickData.x);
}

int16_t Message::GetGamePadStickPositionY() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.gamePadStickData.y);
}

void Message::SetGamePadStickPosition(int16_t x, int16_t y) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadStickData.x, x);
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadStickData.y, y);
}

uint8_t Message::GetTouchFingerId() const NN_NOEXCEPT
{
    return m_Chunk.touchData.fingerId;
}

void Message::SetTouchFingerId(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.touchData.fingerId = value;
}

int16_t Message::GetTouchPositionX() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.touchData.x);
}

int16_t Message::GetTouchPositionY() const NN_NOEXCEPT
{
    return ::nn::util::LoadBigEndian(&m_Chunk.touchData.y);
}

void Message::SetTouchPosition(int16_t x, int16_t y) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.touchData.x, x);
    ::nn::util::StoreBigEndian(&m_Chunk.touchData.y, y);
}

::nn::hid::system::HomeButtonSet Message::GetHomeButtons() const NN_NOEXCEPT
{
    ::nn::hid::system::HomeButtonSet buttons = {};
    buttons._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.systemButtonData.buttons);
    return buttons;
}

void Message::SetHomeButtons(::nn::hid::system::HomeButtonSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.systemButtonData.buttons,
                               static_cast<uint32_t>(value._storage[0]));
}

::nn::hid::system::CaptureButtonSet Message::GetCaptureButtons(
    ) const NN_NOEXCEPT
{
    ::nn::hid::system::CaptureButtonSet buttons = {};
    buttons._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.systemButtonData.buttons);
    return buttons;
}

void Message::SetCaptureButtons(
    ::nn::hid::system::CaptureButtonSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.systemButtonData.buttons,
                               static_cast<uint32_t>(value._storage[0]));
}

bool Message::GetDebugPadConnectionState() const NN_NOEXCEPT
{
    switch (static_cast<::nn::hid::PowerState>(m_Chunk.gamePadData.powerState))
    {
    case ::nn::hid::PowerState_NoBattery:
        return true;
    default:
        return false;
    }
}

void Message::SetDebugPadConnectionState(bool value) NN_NOEXCEPT
{
    m_Chunk.gamePadData.powerState =
        static_cast<uint8_t>(value ? ::nn::hid::PowerState_NoBattery
                                   : ::nn::hid::PowerState_Disconnected);
}

::nn::hid::DebugPadButtonSet Message::GetDebugPadButtons() const NN_NOEXCEPT
{
    ::nn::hid::DebugPadButtonSet buttons = {};
    buttons._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.gamePadData.buttons);
    return buttons;
}

void Message::SetDebugPadButtons(::nn::hid::DebugPadButtonSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadData.buttons, value._storage[0]);
}

uint8_t Message::GetAbstractedPadId() const NN_NOEXCEPT
{
    return m_Chunk.abstractedPadDeviceData.gamePadId;
}

::nn::hid::system::DeviceTypeSet Message::GetAbstractedPadDeviceType() const NN_NOEXCEPT
{
    ::nn::hid::system::DeviceTypeSet deviceType = {};
    deviceType.Set(m_Chunk.abstractedPadDeviceData.deviceType);
    return deviceType;
}

::nn::hid::system::InterfaceType Message::GetAbstractedPadInterfaceType() const NN_NOEXCEPT
{
    return static_cast<::nn::hid::system::InterfaceType>(m_Chunk.abstractedPadDeviceData.interfaceType);
}

uint8_t Message::GetAbstractedPadColorType() const NN_NOEXCEPT
{
    return m_Chunk.abstractedPadColorData.colorType;
}

::nn::util::Color4u8Type  Message::GetAbstractedPadColor() const NN_NOEXCEPT
{
    nn::util::Color4u8Type color = {};
    color.v[0] = m_Chunk.abstractedPadColorData.red;
    color.v[1] = m_Chunk.abstractedPadColorData.green;
    color.v[2] = m_Chunk.abstractedPadColorData.blue;
    color.v[3] = m_Chunk.abstractedPadColorData.alpha;
    return color;
}

::nn::hid::debug::AbstractedPadAttributeSet Message::GetAbstractedPadAttributes() const NN_NOEXCEPT
{
    ::nn::hid::debug::AbstractedPadAttributeSet attributes;
    attributes._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.abstractedPadDeviceData.attribute);
    return attributes;
}

::nn::hid::system::PowerInfo Message::GetAbstractedPadPowerInfo() const NN_NOEXCEPT
{
    ::nn::hid::system::PowerInfo powerInfo = {};
    powerInfo.isPowered     = m_Chunk.abstractedPadStateData.isPowered;
    powerInfo.isCharging    = m_Chunk.abstractedPadStateData.isCharging;
    powerInfo.batteryLevel  = m_Chunk.abstractedPadStateData.batteryLevel;
    return powerInfo;
}

::nn::hid::detail::AbstractedPadButtonSet Message::GetAbstractedPadButtons() const NN_NOEXCEPT
{
    ::nn::hid::detail::AbstractedPadButtonSet buttons = {};
    buttons._storage[0] =
        ::nn::util::LoadBigEndian(&m_Chunk.gamePadData.buttons);

    return buttons;
}

void Message::SetAbstractedPadId(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.gamePadData.gamePadId = value;
}

void Message::SetAbstractedPadAttribute(
    ::nn::hid::debug::AbstractedPadAttributeSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(
        &m_Chunk.abstractedPadDeviceData.attribute, value._storage[0]);
}

void Message::SetAbstractedPadPowerInfo(::nn::hid::system::PowerInfo value) NN_NOEXCEPT
{
    m_Chunk.abstractedPadStateData.isPowered = value.isPowered ? 1 : 0;
    m_Chunk.abstractedPadStateData.isCharging = value.isCharging ? 1 : 0;
    m_Chunk.abstractedPadStateData.batteryLevel = static_cast<uint8_t>(value.batteryLevel);
}

void Message::SetAbstractedPadButtons(::nn::hid::detail::AbstractedPadButtonSet value) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadData.buttons, value._storage[0]);
}

void Message::SetAbstractedPadStickId(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.gamePadStickData.stickId = value;
}
void Message::SetAbstractedPadStickPosition(int16_t x, int16_t y) NN_NOEXCEPT
{
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadStickData.x, x);
    ::nn::util::StoreBigEndian(&m_Chunk.gamePadStickData.y, y);
}

void Message::SetAbstractedPadDeviceType(::nn::hid::system::DeviceTypeSet value) NN_NOEXCEPT
{
    if (value.Test<nn::hid::system::DeviceType::SwitchProController>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_SwitchProController;
    }
    else if (value.Test<nn::hid::system::DeviceType::Handheld>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_Handheld;
    }
    else if (value.Test<nn::hid::system::DeviceType::HandheldJoyLeft>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_HandheldJoyLeft;
    }
    else if (value.Test<nn::hid::system::DeviceType::HandheldJoyRight>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_HandheldJoyRight;
    }
    else if (value.Test<nn::hid::system::DeviceType::JoyConLeft>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_JoyConLeft;
    }
    else if (value.Test<nn::hid::system::DeviceType::JoyConRight>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_JoyConRight;
    }
    else if (value.Test<nn::hid::system::DeviceType::Palma>())
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_Palma;
    }
    else
    {
        m_Chunk.abstractedPadDeviceData.deviceType = DeviceType_Unknown;
    }
}

void Message::SetAbstractedPadInterfaceType(::nn::hid::system::InterfaceType value) NN_NOEXCEPT
{
    m_Chunk.abstractedPadDeviceData.interfaceType = value;
}

void Message::SetAbstractedPadColorType(uint8_t value) NN_NOEXCEPT
{
    m_Chunk.abstractedPadColorData.colorType = value;
}

void Message::SetAbstractedPadColorData(nn::util::Color4u8Type value) NN_NOEXCEPT
{
    m_Chunk.abstractedPadColorData.red = value.v[0];
    m_Chunk.abstractedPadColorData.green = value.v[1];
    m_Chunk.abstractedPadColorData.blue = value.v[2];
    m_Chunk.abstractedPadColorData.alpha = value.v[3];
}

::nn::htcs::SocketError Message::ReadFrom(SocketChannel& channel) NN_NOEXCEPT
{
    m_Chunk = Chunk();

    auto size = size_t();
    NN_HTCS_SOCKET_ERROR_DO(
        channel.Receive(
            &size, &m_Chunk, sizeof(m_Chunk), ::nn::htcs::HTCS_MSG_WAITALL));

    NN_HTCS_SOCKET_ERROR_NONE;
}

::nn::htcs::SocketError Message::WriteTo(SocketChannel& channel) NN_NOEXCEPT
{
    auto size = size_t();
    NN_HTCS_SOCKET_ERROR_DO(
        channel.Send(
            &size, &m_Chunk, sizeof(m_Chunk), ::nn::htcs::HTCS_MSG_WAITALL));

    NN_HTCS_SOCKET_ERROR_NONE;
}

PingPongServer::PingPongServer() NN_NOEXCEPT
    : m_Mutex(false)
    , m_RequestCount(0)
{
    // 何もしない
}

void PingPongServer::Reset() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_RequestCount = 0;
}

bool PingPongServer::AcceptMessage(const Message& message) NN_NOEXCEPT
{
    switch (message.GetType())
    {
    case MessageType_Ping:
        this->AcceptPingMessage(message);
        return true;

    default:
        return false;
    }
}

bool PingPongServer::AcquireMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if (this->AcquirePongMessage(pOutValue))
    {
        return true;
    }

    return false;
}

void PingPongServer::AcceptPingMessage(const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    ++m_RequestCount;
}

bool PingPongServer::AcquirePongMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_RequestCount == 0)
    {
        return false;
    }

    pOutValue->Reset();

    pOutValue->SetType(MessageType_Pong);

    --m_RequestCount;

    return true;
}

InputDirector::InputDirector() NN_NOEXCEPT
    : m_Mutex(false)
    , m_DisplayWidth(0)
    , m_DisplayHeight(0)
    , m_IsCapturing(false)
    , m_NeedsToStopCapture(false)
    , m_CaptureBorderType(BorderType_None)
    , m_CaptureBorderLength(0)
    , m_IsKeyboardInitialized(false)
    , m_KeyboardState()
    , m_IsMouseInitialized(false)
    , m_MouseState()
    , m_IsGamePadInitialized(false)
    , m_GamePadIdCount(0)
    , m_GamePadIds()
    , m_GamePadStates()
    , m_IsTouchScreenInitialized(false)
    , m_TouchScreenState()
    , m_IsTouchScreenAutoPilotEnabled(false)
    , m_TouchScreenAutoPilotState()
    , m_TouchMessageIndex(0)
    , m_TouchMessageCount(0)
    , m_TouchMessages()
    , m_IsHomeButtonInitialized(false)
    , m_IsHomeButtonAutoPilotEnabled(false)
    , m_HomeButtonAutoPilotState()
    , m_IsCaptureButtonInitialized(false)
    , m_IsCaptureButtonAutoPilotEnabled(false)
    , m_CaptureButtonAutoPilotState()
    , m_IsDebugPadInitialized(false)
    , m_DebugPadState()
    , m_IsDebugPadAutoPilotEnabled(false)
    , m_DebugPadAutoPilotState()
    , m_DebugPadMessageIndex(0)
    , m_DebugPadMessageCount(0)
    , m_DebugPadMessages()
    , m_IsAbstractedPadInitialized(false)
    , m_AbstractedPadAutoPilotStates()
    , m_AbstractedPadInfo()
    , m_AbstractedPadMessageIndex(0)
    , m_AbstractedPadMessageCount(0)
    , m_AbstractedPadMessages()
{
    // 何もしない
}

void InputDirector::SetDisplaySize(int32_t width, int32_t height) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_DisplayWidth = width;
    m_DisplayHeight = height;
}

void InputDirector::Reset() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_IsCapturing = false;
    m_NeedsToStopCapture = false;
    m_CaptureBorderType = BorderType_None;
    m_CaptureBorderLength = 0;

    this->ResetCapture();

    if (m_IsGamePadInitialized)
    {
        for (int i = 0; i < m_GamePadIdCount; ++i)
        {
            m_GamePadStates[i] = ::nn::hid::debug::BasicXpadAutoPilotState();

            ::nn::hid::debug::UnsetXpadAutoPilotState(m_GamePadIds[i]);
        }
    }

    if (m_IsTouchScreenInitialized)
    {
        ::nn::hid::debug::UnsetTouchScreenAutoPilotState();

        m_TouchScreenState =
            ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax>();

        m_IsTouchScreenAutoPilotEnabled = false;

        m_TouchScreenAutoPilotState =
            ::nn::hid::debug::TouchScreenAutoPilotState<
                ::nn::hid::TouchStateCountMax>();

        m_TouchMessageIndex = 0;
        m_TouchMessageCount = 0;
    }

    if (m_IsHomeButtonInitialized)
    {
        ::nn::hid::debug::UnsetHomeButtonAutoPilotState();

        m_IsHomeButtonAutoPilotEnabled = false;

        m_HomeButtonAutoPilotState =
            ::nn::hid::debug::HomeButtonAutoPilotState();
    }

    if (m_IsCaptureButtonInitialized)
    {
        ::nn::hid::debug::UnsetCaptureButtonAutoPilotState();

        m_IsCaptureButtonAutoPilotEnabled = false;

        m_CaptureButtonAutoPilotState =
            ::nn::hid::debug::CaptureButtonAutoPilotState();
    }

    if (m_IsDebugPadInitialized)
    {
        ::nn::hid::debug::UnsetDebugPadAutoPilotState();

        m_DebugPadState = ::nn::hid::DebugPadState();

        m_IsDebugPadAutoPilotEnabled = false;

        m_DebugPadAutoPilotState = ::nn::hid::debug::DebugPadAutoPilotState();

        m_DebugPadMessageIndex = 0;
        m_DebugPadMessageCount = 0;
    }

    if (m_IsAbstractedPadInitialized)
    {
        ::nn::hid::debug::UnsetAllAutoPilotVirtualPadState();
        for (auto& state : m_AbstractedPadAutoPilotStates)
        {
            state = ::nn::hid::debug::AbstractedPadState();
        }
    }

    for (auto& info : m_AbstractedPadInfo)
    {
        info = AbstractedPadInfo();
    }
    m_AbstractedPadMessageIndex = 0;
    m_AbstractedPadMessageCount = 0;
}

bool InputDirector::AcceptMessage(const Message& message) NN_NOEXCEPT
{
    switch (message.GetType())
    {
    case MessageType_CaptureStart:
        this->AcceptCaptureStartMessage(message);
        return true;

    case MessageType_CaptureStop:
        this->AcceptCaptureStopMessage(message);
        return true;

    case MessageType_KeyDown:
        this->AcceptKeyDownMessage(message);
        return true;

    case MessageType_KeyUp:
        this->AcceptKeyUpMessage(message);
        return true;

    case MessageType_MouseMove:
        this->AcceptMouseMoveMessage(message);
        return true;

    case MessageType_MouseButton:
        this->AcceptMouseButtonMessage(message);
        return true;

    case MessageType_MouseWheel:
        this->AcceptMouseWheelMessage(message);
        return true;

    case MessageType_GamePadButton:
        this->AcceptGamePadButtonMessage(message);
        return true;

    case MessageType_GamePadStick:
        this->AcceptGamePadStickMessage(message);
        return true;

    case MessageType_GamePadPower:
        this->AcceptGamePadPowerMessage(message);
        return true;

    case MessageType_TouchBegan:
        this->AcceptTouchBeganMessage(message);
        return true;

    case MessageType_TouchMoved:
        this->AcceptTouchMovedMessage(message);
        return true;

    case MessageType_TouchEnded:
        this->AcceptTouchEndedMessage(message);
        return true;

    case MessageType_HomeButton:
        this->AcceptHomeButtonMessage(message);
        return true;

    case MessageType_CaptureButton:
        this->AcceptCaptureButtonMessage(message);
        return true;

    case MessageType_DebugPadButton:
        this->AcceptDebugPadButtonMessage(message);
        return true;

    case MessageType_DebugPadStick:
        this->AcceptDebugPadStickMessage(message);
        return true;

    case MessageType_DebugPadPower:
        this->AcceptDebugPadPowerMessage(message);
        return true;

    case MessageType_AbstractedPadDeviceData:
        this->AcceptAbstractedPadDeviceDataMessage(message);
        return true;

    case MessageType_AbstractedPadColor:
        this->AcceptAbstractedPadColorMessage(message);
        return true;

    case MessageType_AbstractedPadPowerState:
        this->AcceptAbstractedPadPowerStateMessage(message);
        return true;

    case MessageType_AbstractedPadButton:
        this->AcceptAbstractedPadButtonMessage(message);
        return true;

    case MessageType_AbstractedPadStick:
        this->AcceptAbstractedPadStickMessage(message);
        return true;

    default:
        return false;
    }
}

void InputDirector::PrepareMessage() NN_NOEXCEPT
{
    this->PrepareTouchMessage();

    this->PrepareDebugPadMessage();

    this->PrepareAbstractedPadMessage();
}

bool InputDirector::AcquireMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if (this->AcquireCaptureStopMessage(pOutValue))
    {
        return true;
    }

    if (this->AcquireTouchMessage(pOutValue))
    {
        return true;
    }

    if (this->AcquireDebugPadMessage(pOutValue))
    {
        return true;
    }

    if (this->AcquireAbstractedPadMessage(pOutValue))
    {
        return true;
    }

    return false;
}

void InputDirector::ResetCapture() NN_NOEXCEPT
{
    m_KeyboardState.modifiers.Reset();
    m_KeyboardState.attributes.Reset();
    m_KeyboardState.keys.Reset();

    if (m_IsKeyboardInitialized)
    {
        ::nn::hid::debug::UnsetKeyboardAutoPilotState();
    }

    m_MouseState.deltaX = 0;
    m_MouseState.deltaY = 0;
    m_MouseState.wheelDelta = 0;
    m_MouseState.buttons.Reset();
    m_MouseState.attributes.Reset();

    if (m_IsMouseInitialized)
    {
        ::nn::hid::debug::UnsetMouseAutoPilotState();
    }
}

void InputDirector::AcceptCaptureStartMessage(
    const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_CaptureBorderType = message.GetCaptureBorderType();

    m_CaptureBorderLength =
        static_cast<int32_t>(message.GetCaptureBorderLength());

    this->ResetCapture();

    if (message.IsCaptureTriggerdByMouse())
    {
        m_MouseState.x = message.GetCapturePositionX();
        m_MouseState.y = message.GetCapturePositionY();

        switch (m_CaptureBorderType)
        {
        case BorderType_Right:
            m_MouseState.x = 1;
            m_MouseState.y = static_cast<int32_t>(
                static_cast<float>(m_DisplayHeight) / m_CaptureBorderLength *
                m_MouseState.y);
            break;

        case BorderType_Left:
            m_MouseState.x = m_DisplayWidth - 1;
            m_MouseState.y = static_cast<int32_t>(
                static_cast<float>(m_DisplayHeight) / m_CaptureBorderLength *
                m_MouseState.y);
            break;

        case BorderType_Top:
            m_MouseState.x = static_cast<int32_t>(
                static_cast<float>(m_DisplayWidth) / m_CaptureBorderLength *
                m_MouseState.x);
            m_MouseState.y = m_DisplayHeight - 1;
            break;

        case BorderType_Bottom:
            m_MouseState.x = static_cast<int32_t>(
                static_cast<float>(m_DisplayWidth) / m_CaptureBorderLength *
                m_MouseState.x);
            m_MouseState.y = 1;
            break;

        default:
            break;
        }
    }
    else
    {
        if (m_MouseState.x < 0)
        {
            m_MouseState.x = 0;
        }

        if (m_MouseState.x > m_DisplayWidth)
        {
            m_MouseState.x = m_DisplayWidth;
        }

        if (m_MouseState.y < 0)
        {
            m_MouseState.y = 0;
        }

        if (m_MouseState.y > m_DisplayHeight)
        {
            m_MouseState.y = m_DisplayHeight;
        }
    }

    m_KeyboardState.attributes.Set<::nn::hid::KeyboardAttribute::IsConnected>();

    if (!m_IsKeyboardInitialized)
    {
        ::nn::hid::InitializeKeyboard();

        m_IsKeyboardInitialized = true;
    }

    ::nn::hid::debug::SetKeyboardAutoPilotState(m_KeyboardState);

    m_MouseState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    if (!m_IsMouseInitialized)
    {
        ::nn::hid::InitializeMouse();

        m_IsMouseInitialized = true;
    }

    ::nn::hid::debug::SetMouseAutoPilotState(m_MouseState);

    m_IsCapturing = true;

    m_NeedsToStopCapture = false;
}

void InputDirector::AcceptCaptureStopMessage(const Message& message
                                             ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    this->ResetCapture();

    m_IsCapturing = false;

    m_NeedsToStopCapture = false;
}

void InputDirector::AcceptKeyMessage(
    const Message& message,
    ::nn::hid::debug::KeyboardAutoPilotState(*function)(
        const Message&,
        const ::nn::hid::debug::KeyboardAutoPilotState&) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsCapturing)
    {
        return;
    }

    m_KeyboardState = function(message, m_KeyboardState);

    m_KeyboardState.modifiers.Set<::nn::hid::KeyboardModifier::Control>(
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::LeftControl>() ||
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::RightControl>());

    m_KeyboardState.modifiers.Set<::nn::hid::KeyboardModifier::Shift>(
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::LeftShift>() ||
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::RightShift>());

    m_KeyboardState.modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>(
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::LeftAlt>());

    m_KeyboardState.modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>(
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::RightAlt>());

    m_KeyboardState.modifiers.Set<::nn::hid::KeyboardModifier::Gui>(
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::LeftGui>() ||
        m_KeyboardState.keys.Test<::nn::hid::KeyboardKey::RightGui>());

    ::nn::hid::debug::SetKeyboardAutoPilotState(m_KeyboardState);
}

void InputDirector::AcceptKeyDownMessage(const Message& message) NN_NOEXCEPT
{
    this->AcceptKeyMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::KeyboardAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::KeyboardAutoPilotState value = state;

        value.keys.Set(message.GetKeyboardKeyCode(), true);

        return value;
    });
}

void InputDirector::AcceptKeyUpMessage(const Message& message) NN_NOEXCEPT
{
    this->AcceptKeyMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::KeyboardAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::KeyboardAutoPilotState value = state;

        value.keys.Set(message.GetKeyboardKeyCode(), false);

        return value;
    });
}

void InputDirector::AcceptMouseMoveMessage(const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsCapturing)
    {
        return;
    }

    m_MouseState.deltaX = message.GetMouseDeltaPositionX();
    m_MouseState.deltaY = message.GetMouseDeltaPositionY();

    m_MouseState.x += m_MouseState.deltaX;
    m_MouseState.y += m_MouseState.deltaY;

    auto needsToStopCapture = false;

    if (m_MouseState.x < 0)
    {
        m_MouseState.x = 0;

        if (m_CaptureBorderType == BorderType_Right)
        {
            needsToStopCapture = true;
        }
    }

    if (m_MouseState.x > m_DisplayWidth)
    {
        m_MouseState.x = m_DisplayWidth;

        if (m_CaptureBorderType == BorderType_Left)
        {
            needsToStopCapture = true;
        }
    }

    if (m_MouseState.y < 0)
    {
        m_MouseState.y = 0;

        if (m_CaptureBorderType == BorderType_Bottom)
        {
            needsToStopCapture = true;
        }
    }

    if (m_MouseState.y > m_DisplayHeight)
    {
        m_MouseState.y = m_DisplayHeight;

        if (m_CaptureBorderType == BorderType_Top)
        {
            needsToStopCapture = true;
        }
    }

    if (!needsToStopCapture ||
        m_KeyboardState.keys.IsAnyOn() ||
        m_MouseState.buttons.IsAnyOn())
    {
        ::nn::hid::debug::SetMouseAutoPilotState(m_MouseState);

        m_MouseState.deltaX = 0;
        m_MouseState.deltaY = 0;
    }
    else
    {
        this->ResetCapture();

        m_IsCapturing = false;

        m_NeedsToStopCapture = true;
    }
}

void InputDirector::AcceptMouseButtonMessage(const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsCapturing)
    {
        return;
    }

    m_MouseState.buttons = message.GetMouseButtons();

    ::nn::hid::debug::SetMouseAutoPilotState(m_MouseState);
}

void InputDirector::AcceptMouseWheelMessage(const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsCapturing)
    {
        return;
    }

    m_MouseState.wheelDelta = message.GetMouseDeltaWheel();

    ::nn::hid::debug::SetMouseAutoPilotState(m_MouseState);

    m_MouseState.wheelDelta = 0;
}

void InputDirector::AcceptGamePadMessage(
    const Message& message,
    ::nn::hid::debug::BasicXpadAutoPilotState(*function)(
        const Message&,
        const ::nn::hid::debug::BasicXpadAutoPilotState&) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsGamePadInitialized)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            ::nn::hid::detail::GetXpadIds(
                &m_GamePadIdCount,
                m_GamePadIds, static_cast<int>(NN_ARRAY_SIZE(m_GamePadIds))));

        for (int i = 0; i < m_GamePadIdCount; ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                ::nn::hid::detail::InitializeXpad(m_GamePadIds[i]));
        }

        m_IsGamePadInitialized = true;
    }

    const auto playerNumber = static_cast<int>(message.GetGamePadId());

    for (int i = 0; i < m_GamePadIdCount; ++i)
    {
        const ::nn::hid::BasicXpadId& gamePadId = m_GamePadIds[i];

        auto candidate = int();

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            ::nn::hid::detail::GetXpadPlayerNumber(&candidate, gamePadId));

        if (playerNumber != candidate)
        {
            continue;
        }

        ::nn::hid::debug::BasicXpadAutoPilotState& gamePadState =
            m_GamePadStates[i];

        gamePadState = function(message, gamePadState);

        switch (static_cast<::nn::hid::PowerState>(gamePadState.powerState))
        {
        case ::nn::hid::PowerState_Disconnected:
            ::nn::hid::debug::UnsetXpadAutoPilotState(gamePadId);
            return;

        default:
            ::nn::hid::debug::SetXpadAutoPilotState(gamePadId, gamePadState);
            return;
        }
    }
}

void InputDirector::AcceptGamePadButtonMessage(const Message& message
                                               ) NN_NOEXCEPT
{
    this->AcceptGamePadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::BasicXpadAutoPilotState value = state;

        value.buttons = message.GetGamePadButtons();

        return value;
    });
}

void InputDirector::AcceptGamePadStickMessage(const Message& message
                                              ) NN_NOEXCEPT
{
    this->AcceptGamePadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::BasicXpadAutoPilotState value = state;

        const uint8_t stickId = message.GetGamePadStickId();

        if (stickId == 0u)
        {
            value.analogStickL.x = message.GetGamePadStickPositionX();
            value.analogStickL.y = message.GetGamePadStickPositionY();
        }
        else if (stickId == 1u)
        {
            value.analogStickR.x = message.GetGamePadStickPositionX();
            value.analogStickR.y = message.GetGamePadStickPositionY();
        }

        return value;
    });
}

void InputDirector::AcceptGamePadPowerMessage(const Message& message
                                              ) NN_NOEXCEPT
{
    this->AcceptGamePadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::BasicXpadAutoPilotState value = state;

        value.powerState = message.GetGamePadPowerState();

        value.batteryLevel = message.GetGamePadBatteryLevel();

        return value;
    });
}

void InputDirector::AcceptTouchMessage(
    const Message& message,
    ::nn::hid::debug::TouchScreenAutoPilotState<
        ::nn::hid::TouchStateCountMax>(*function)(
            const Message&,
            const ::nn::hid::debug::TouchScreenAutoPilotState<
                ::nn::hid::TouchStateCountMax>&) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    ::nn::hid::debug::TouchScreenAutoPilotState<::nn::hid::TouchStateCountMax
        >& autoPilotState = m_TouchScreenAutoPilotState;

    autoPilotState = function(message, autoPilotState);

    if (!m_IsTouchScreenInitialized)
    {
        ::nn::hid::InitializeTouchScreen();

        m_IsTouchScreenInitialized = true;
    }

    ::nn::hid::debug::SetTouchScreenAutoPilotState(autoPilotState);

    m_IsTouchScreenAutoPilotEnabled = true;
}

void InputDirector::AcceptTouchBeganMessage(const Message& message) NN_NOEXCEPT
{
    this->AcceptTouchMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax>& state) NN_NOEXCEPT
    {
        if (state.count >= ::nn::hid::TouchStateCountMax)
        {
            return state;
        }

        const auto fingerId = static_cast<int>(message.GetTouchFingerId());

        for (int i = 0; i < state.count; ++i)
        {
            if (state.touches[i].fingerId == fingerId)
            {
                return state;
            }
        }

        ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax> value = state;

        ::nn::hid::TouchState& touch = value.touches[state.count];

        touch = ::nn::hid::TouchState();
        touch.fingerId = fingerId;
        touch.x = message.GetTouchPositionX();
        touch.y = message.GetTouchPositionY();
        touch.diameterX = 64;
        touch.diameterY = 64;

        value.count += 1;

        return value;
    });
}

void InputDirector::AcceptTouchMovedMessage(const Message& message) NN_NOEXCEPT
{
    this->AcceptTouchMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax>& state) NN_NOEXCEPT
    {
        const auto fingerId = static_cast<int>(message.GetTouchFingerId());
        const int x = message.GetTouchPositionX();
        const int y = message.GetTouchPositionY();

        ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax> value = state;

        for (int i = 0; i < state.count; ++i)
        {
            if (state.touches[i].fingerId != fingerId)
            {
                continue;
            }

            value.touches[i].x = x;
            value.touches[i].y = y;
        }

        return value;
    });
}

void InputDirector::AcceptTouchEndedMessage(const Message& message) NN_NOEXCEPT
{
    this->AcceptTouchMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax>& state) NN_NOEXCEPT
    {
        const auto fingerId = static_cast<int>(message.GetTouchFingerId());

        ::nn::hid::debug::TouchScreenAutoPilotState<
            ::nn::hid::TouchStateCountMax> value = {};

        for (int i = 0; i < state.count; ++i)
        {
            if (state.touches[i].fingerId == fingerId)
            {
                continue;
            }

            value.touches[value.count] = state.touches[i];
            value.count += 1;
        }

        return value;
    });
}

void InputDirector::AcceptHomeButtonMessage(const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsHomeButtonInitialized)
    {
        ::nn::hid::system::InitializeHomeButton();

        m_IsHomeButtonInitialized = true;
    }

    ::nn::hid::debug::HomeButtonAutoPilotState&
        autoPilotState = m_HomeButtonAutoPilotState;

    autoPilotState.buttons = message.GetHomeButtons();

    if (autoPilotState.buttons.IsAllOff())
    {
        ::nn::hid::debug::UnsetHomeButtonAutoPilotState();

        m_IsHomeButtonAutoPilotEnabled = false;
    }
    else
    {
        ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);

        m_IsHomeButtonAutoPilotEnabled = true;
    }
}

void InputDirector::AcceptCaptureButtonMessage(
    const Message& message) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsCaptureButtonInitialized)
    {
        ::nn::hid::system::InitializeCaptureButton();

        m_IsCaptureButtonInitialized = true;
    }

    ::nn::hid::debug::CaptureButtonAutoPilotState&
        autoPilotState = m_CaptureButtonAutoPilotState;

    autoPilotState.buttons = message.GetCaptureButtons();

    if (autoPilotState.buttons.IsAllOff())
    {
        ::nn::hid::debug::UnsetCaptureButtonAutoPilotState();

        m_IsCaptureButtonAutoPilotEnabled = false;
    }
    else
    {
        ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState);

        m_IsCaptureButtonAutoPilotEnabled = true;
    }
}

void InputDirector::AcceptDebugPadMessage(
    const Message& message,
    ::nn::hid::debug::DebugPadAutoPilotState(*function)(
        const Message&,
        const ::nn::hid::debug::DebugPadAutoPilotState&) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsDebugPadInitialized)
    {
        ::nn::hid::InitializeDebugPad();

        m_IsDebugPadInitialized = true;
    }

    m_DebugPadAutoPilotState = function(message, m_DebugPadAutoPilotState);

    ::nn::hid::debug::SetDebugPadAutoPilotState(m_DebugPadAutoPilotState);
}

void InputDirector::AcceptDebugPadButtonMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptDebugPadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::DebugPadAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::DebugPadAutoPilotState value = state;

        value.buttons = message.GetDebugPadButtons();

        return value;
    });
}

void InputDirector::AcceptDebugPadStickMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptDebugPadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::DebugPadAutoPilotState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::DebugPadAutoPilotState value = state;

        const uint8_t stickId = message.GetGamePadStickId();

        if (stickId == 0u)
        {
            value.analogStickL.x = message.GetGamePadStickPositionX();
            value.analogStickL.y = message.GetGamePadStickPositionY();
        }
        else if (stickId == 1u)
        {
            value.analogStickR.x = message.GetGamePadStickPositionX();
            value.analogStickR.y = message.GetGamePadStickPositionY();
        }

        return value;
    });
}

void InputDirector::AcceptDebugPadPowerMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptDebugPadMessage(message, [] (
        const Message& message,
        const ::nn::hid::debug::DebugPadAutoPilotState& state) NN_NOEXCEPT
    {
        if (!message.GetDebugPadConnectionState())
        {
            return ::nn::hid::debug::DebugPadAutoPilotState();
        }
        else
        {
            ::nn::hid::debug::DebugPadAutoPilotState value = state;
            value.attributes.Set<::nn::hid::DebugPadAttribute::IsConnected>();
            return value;
        }
    });
}

void InputDirector::AcceptAbstractedPadMessage(
    const Message& message,
    ::nn::hid::debug::AbstractedPadState(*function)(
        const Message&,
        const ::nn::hid::debug::AbstractedPadState&) NN_NOEXCEPT
    ) NN_NOEXCEPT
{
    if (!m_IsAbstractedPadInitialized)
    {
        ::nn::hid::debug::UnsetAllAutoPilotVirtualPadState();
        for (auto& state : m_AbstractedPadAutoPilotStates)
        {
            state = ::nn::hid::debug::AbstractedPadState();
        }

        m_IsAbstractedPadInitialized = true;
    }

    const int deviceIndex = message.GetAbstractedPadId();
    if (deviceIndex < 0 || deviceIndex >= ::nn::hid::debug::AbstractedVirtualPadIdCountMax)
    {
        return;
    }

    ::nn::hid::debug::AbstractedPadState& abstractedPadState =
        m_AbstractedPadAutoPilotStates[deviceIndex];
    abstractedPadState = function(message, abstractedPadState);

    ::nn::hid::debug::AbstractedVirtualPadId virtualPadId =
        static_cast<::nn::hid::debug::AbstractedVirtualPadId>(deviceIndex);

    if (abstractedPadState.attributes.Test<::nn::hid::debug::AbstractedPadAttribute::IsConnected>())
    {
        ::nn::hid::debug::SetAutoPilotVirtualPadState(virtualPadId, abstractedPadState);
    }
    else
    {
        ::nn::hid::debug::UnsetAutoPilotVirtualPadState(virtualPadId);
    }
}

void InputDirector::AcceptAbstractedPadDeviceDataMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptAbstractedPadMessage(message, [](
        const Message& message,
        const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::AbstractedPadState value = state;
        value.deviceType    = message.GetAbstractedPadDeviceType();
        value.interfaceType = message.GetAbstractedPadInterfaceType();
        value.attributes    = message.GetAbstractedPadAttributes();
        return value;
    });
}

void InputDirector::AcceptAbstractedPadColorMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptAbstractedPadMessage(message, [](
        const Message& message,
        const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::AbstractedPadState value = state;

        uint8_t colorType = message.GetAbstractedPadColorType();
        if (!colorType)
        {
            value.color.main = message.GetAbstractedPadColor();
        }
        else
        {
            value.color.sub = message.GetAbstractedPadColor();
        }

        return value;
    });
}

void InputDirector::AcceptAbstractedPadPowerStateMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptAbstractedPadMessage(message, [](
        const Message& message,
        const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::AbstractedPadState value = state;
        value.powerInfo = message.GetAbstractedPadPowerInfo();
        return value;
    });
}

void InputDirector::AcceptAbstractedPadButtonMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptAbstractedPadMessage(message, [](
        const Message& message,
        const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::AbstractedPadState value = state;
        value.buttons = message.GetAbstractedPadButtons();
        return value;
    });
}

void InputDirector::AcceptAbstractedPadStickMessage(
    const Message& message) NN_NOEXCEPT
{
    this->AcceptAbstractedPadMessage(message, [](
        const Message& message,
        const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
    {
        ::nn::hid::debug::AbstractedPadState value = state;
        const uint8_t stickId = message.GetGamePadStickId();

        if (stickId == 0u)
        {
            value.analogStickL.x = message.GetGamePadStickPositionX();
            value.analogStickL.y = message.GetGamePadStickPositionY();
        }
        else if (stickId == 1u)
        {
            value.analogStickR.x = message.GetGamePadStickPositionX();
            value.analogStickR.y = message.GetGamePadStickPositionY();
        }

        return value;
    });
}

void InputDirector::PrepareTouchMessage() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsTouchScreenInitialized)
    {
        ::nn::hid::InitializeTouchScreen();

        m_IsTouchScreenInitialized = true;
    }

    m_TouchMessageIndex = 0;
    m_TouchMessageCount = 0;

    ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax> nextState = {};
    ::nn::hid::GetTouchScreenState(&nextState);

    if (m_IsTouchScreenAutoPilotEnabled)
    {
        if (nextState.count == 0 && m_TouchScreenAutoPilotState.count == 0)
        {
            ::nn::hid::debug::UnsetTouchScreenAutoPilotState();

            m_IsTouchScreenAutoPilotEnabled = false;
        }

        return;
    }

    ::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax
        >& prevState = m_TouchScreenState;

    int nextIndex = 0;
    int prevIndex = 0;

    while (NN_STATIC_CONDITION(true))
    {
        if (prevIndex == prevState.count)
        {
             for (int i = nextIndex; i < nextState.count; ++i)
             {
                 const ::nn::hid::TouchState& touch = nextState.touches[i];
                 Message& message = m_TouchMessages[m_TouchMessageCount++];
                 message.Reset();
                 message.SetType(MessageType_TouchBegan);
                 message.SetTouchFingerId(static_cast<uint8_t>(touch.fingerId));
                 message.SetTouchPosition(static_cast<int16_t>(touch.x),
                                          static_cast<int16_t>(touch.y));
             }

             break;
        }

        if (nextIndex == nextState.count)
        {
            for (int i = prevIndex; i < prevState.count; ++i)
            {
                 const ::nn::hid::TouchState& touch = prevState.touches[i];
                 Message& message = m_TouchMessages[m_TouchMessageCount++];
                 message.Reset();
                 message.SetType(MessageType_TouchEnded);
                 message.SetTouchFingerId(static_cast<uint8_t>(touch.fingerId));
            }

            break;
        }

        const auto& nextTouch = nextState.touches[nextIndex];
        const auto& prevTouch = prevState.touches[prevIndex];

        if (nextTouch.fingerId != prevTouch.fingerId)
        {
            Message& message = m_TouchMessages[m_TouchMessageCount++];
            message.Reset();
            message.SetType(MessageType_TouchEnded);
            message.SetTouchFingerId(static_cast<uint8_t>(prevTouch.fingerId));
            ++prevIndex;
        }
        else
        {
            if (nextTouch.x != prevTouch.x || nextTouch.y != prevTouch.y)
            {
                Message& message = m_TouchMessages[m_TouchMessageCount++];
                message.Reset();
                message.SetType(MessageType_TouchMoved);
                message.SetTouchFingerId(
                    static_cast<uint8_t>(nextTouch.fingerId));
                message.SetTouchPosition(static_cast<int16_t>(nextTouch.x),
                                         static_cast<int16_t>(nextTouch.y));
            }

            ++nextIndex;
            ++prevIndex;
        }
    }

    prevState = nextState;
}

void InputDirector::PrepareDebugPadMessage() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsDebugPadInitialized)
    {
        ::nn::hid::InitializeDebugPad();

        m_IsDebugPadInitialized = true;
    }

    m_DebugPadMessageIndex = 0;
    m_DebugPadMessageCount = 0;

    ::nn::hid::DebugPadState nextState = {};
    ::nn::hid::GetDebugPadState(&nextState);

    if (m_IsDebugPadAutoPilotEnabled)
    {
        if (m_DebugPadAutoPilotState.attributes == m_DebugPadState.attributes &&
            m_DebugPadAutoPilotState.buttons.IsAllOff() &&
            m_DebugPadAutoPilotState.analogStickL.x == 0 &&
            m_DebugPadAutoPilotState.analogStickL.y == 0 &&
            m_DebugPadAutoPilotState.analogStickR.x == 0 &&
            m_DebugPadAutoPilotState.analogStickR.y == 0)
        {
            ::nn::hid::debug::UnsetDebugPadAutoPilotState();

            m_IsDebugPadAutoPilotEnabled = false;
        }

        return;
    }

    ::nn::hid::DebugPadState& prevState = m_DebugPadState;

    if (nextState.attributes.Test<
            ::nn::hid::DebugPadAttribute::IsConnected>() !=
        prevState.attributes.Test<
            ::nn::hid::DebugPadAttribute::IsConnected>())
    {
        Message& message = m_DebugPadMessages[m_DebugPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_DebugPadPower);
        message.SetDebugPadConnectionState(
            nextState.attributes.Test<
                ::nn::hid::DebugPadAttribute::IsConnected>());
    }

    if (nextState.buttons != prevState.buttons)
    {
        Message& message = m_DebugPadMessages[m_DebugPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_DebugPadButton);
        message.SetDebugPadButtons(nextState.buttons);
    }

    if (nextState.analogStickL.x != prevState.analogStickL.x ||
        nextState.analogStickL.y != prevState.analogStickL.y)
    {
        Message& message = m_DebugPadMessages[m_DebugPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_DebugPadStick);
        message.SetGamePadStickId(0u);
        message.SetGamePadStickPosition(
            nextState.analogStickL.x, nextState.analogStickL.y);
    }

    if (nextState.analogStickR.x != prevState.analogStickR.x ||
        nextState.analogStickR.y != prevState.analogStickR.y)
    {
        Message& message = m_DebugPadMessages[m_DebugPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_DebugPadStick);
        message.SetGamePadStickId(1u);
        message.SetGamePadStickPosition(
            nextState.analogStickR.x, nextState.analogStickR.y);
    }

    prevState = nextState;
}

void InputDirector::PrepareAbstractedPadMessage() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    m_AbstractedPadMessageIndex = 0;
    m_AbstractedPadMessageCount = 0;

    int nextHandleCount;
    ::nn::hid::debug::AbstractedPadHandle nextHandles[
        ::nn::hid::debug::AbstractedPadHandleCountMax] = {};
    ::nn::hid::debug::AbstractedPadState nextStates[
        ::nn::hid::debug::AbstractedPadHandleCountMax] = {};
    bool isKnownHandle[::nn::hid::debug::AbstractedPadHandleCountMax] = {};

    if(::nn::hid::debug::GetAbstractedPadsState(
        &nextHandleCount, nextHandles, nextStates,
        nn::hid::debug::AbstractedPadHandleCountMax).IsFailure())
    {
        return;
    }

    // 既存のパッドを更新
    for (int prevIndex = 0;
        prevIndex < ::nn::hid::debug::AbstractedPadHandleCountMax; ++prevIndex)
    {
        if (!m_AbstractedPadInfo[prevIndex].isEnabled)
        {
            continue;
        }

        int foundIndex = -1;
        for (int nextIndex = 0; nextIndex < nextHandleCount; ++nextIndex)
        {
            if (m_AbstractedPadInfo[prevIndex].handle.value == nextHandles[nextIndex].value)
            {
                foundIndex = nextIndex;
                break;
            }
        }

        if (foundIndex == -1)
        {
            m_AbstractedPadInfo[prevIndex].isEnabled = false;

            Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
            message.Reset();
            message.SetType(MessageType_AbstractedPadDeviceData);
            message.SetAbstractedPadId(prevIndex);
            ::nn::hid::debug::AbstractedPadAttributeSet attribute;
            attribute.Reset();
            message.SetAbstractedPadAttribute(attribute);
        }
        else
        {
            isKnownHandle[foundIndex] = true;

            this->PrepareUpdatingAbstractedPadMessage(
                prevIndex, m_AbstractedPadInfo[prevIndex].state, nextStates[foundIndex]);

            m_AbstractedPadInfo[prevIndex].state = nextStates[foundIndex];
        }
    }

    // 新規のパッドを追加
    for (int nextIndex = 0; nextIndex < nextHandleCount; ++nextIndex)
    {
        if (isKnownHandle[nextIndex])
        {
            continue;
        }

        for (int prevIndex = 0;
            prevIndex < ::nn::hid::debug::AbstractedPadHandleCountMax; ++prevIndex)
        {
            if (m_AbstractedPadInfo[prevIndex].isEnabled)
            {
                continue;
            }

            this->PrepareNewAbstractedPadMessage(prevIndex, nextStates[nextIndex]);

            m_AbstractedPadInfo[prevIndex].handle = nextHandles[nextIndex];
            m_AbstractedPadInfo[prevIndex].state = nextStates[nextIndex];
            m_AbstractedPadInfo[prevIndex].isEnabled = true;
            break;
        }
    }

}

void InputDirector::PrepareUpdatingAbstractedPadMessage(
    int id,
    const ::nn::hid::debug::AbstractedPadState& prevState,
    const ::nn::hid::debug::AbstractedPadState& nextState) NN_NOEXCEPT
{
    if (nextState.powerInfo.isPowered != prevState.powerInfo.isPowered ||
        nextState.powerInfo.isCharging != prevState.powerInfo.isCharging ||
        nextState.powerInfo.batteryLevel != prevState.powerInfo.batteryLevel)
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadPowerState);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadPowerInfo(nextState.powerInfo);
    }

    if (nextState.buttons != prevState.buttons)
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadButton);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadButtons(nextState.buttons);
    }

    if (nextState.analogStickL.x != prevState.analogStickL.x ||
        nextState.analogStickL.y != prevState.analogStickL.y)
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadStick);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadStickId(0u);
        message.SetAbstractedPadStickPosition(
            nextState.analogStickL.x, nextState.analogStickL.y);
    }

    if (nextState.analogStickR.x != prevState.analogStickR.x ||
        nextState.analogStickR.y != prevState.analogStickR.y)
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetAbstractedPadId(id);
        message.SetType(MessageType_AbstractedPadStick);
        message.SetAbstractedPadStickId(1u);
        message.SetAbstractedPadStickPosition(
            nextState.analogStickR.x, nextState.analogStickR.y);
    }
}

void InputDirector::PrepareNewAbstractedPadMessage(
    int id, const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
{
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadColor);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadColorType(0u);
        message.SetAbstractedPadColorData(state.color.main);
    }
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadColor);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadColorType(1u);
        message.SetAbstractedPadColorData(state.color.sub);
    }
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadPowerState);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadPowerInfo(state.powerInfo);
    }
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadButton);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadButtons(state.buttons);
    }
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadStick);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadStickId(0u);
        message.SetAbstractedPadStickPosition(
            state.analogStickL.x, state.analogStickL.y);
    }
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetAbstractedPadId(id);
        message.SetType(MessageType_AbstractedPadStick);
        message.SetAbstractedPadStickId(1u);
        message.SetAbstractedPadStickPosition(
            state.analogStickR.x, state.analogStickR.y);
    }

    // 接続ステータスを持つ MessageType_AbstractedPadDeviceData のメッセージは最後に送る
    {
        Message& message = m_AbstractedPadMessages[m_AbstractedPadMessageCount++];
        message.Reset();
        message.SetType(MessageType_AbstractedPadDeviceData);
        message.SetAbstractedPadId(id);
        message.SetAbstractedPadDeviceType(state.deviceType);
        message.SetAbstractedPadInterfaceType(state.interfaceType);
        message.SetAbstractedPadAttribute(state.attributes);
    }
}

bool InputDirector::AcquireCaptureStopMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_NeedsToStopCapture)
    {
        return false;
    }

    pOutValue->Reset();

    pOutValue->SetType(MessageType_CaptureStop);

    switch (m_CaptureBorderType)
    {
    case BorderType_Right:
        pOutValue->SetCapturePosition(
            0,
            static_cast<int16_t>(
                static_cast<float>(m_CaptureBorderLength) / m_DisplayHeight *
                m_MouseState.y));
        break;

    case BorderType_Left:
        pOutValue->SetCapturePosition(
            m_DisplayWidth,
            static_cast<int16_t>(
                static_cast<float>(m_CaptureBorderLength) / m_DisplayHeight *
                m_MouseState.y));
        break;

    case BorderType_Top:
        pOutValue->SetCapturePosition(
            static_cast<int16_t>(
                static_cast<float>(m_CaptureBorderLength) / m_DisplayWidth *
                m_MouseState.x),
            m_MouseState.y);
        break;

    case BorderType_Bottom:
        pOutValue->SetCapturePosition(
            static_cast<int16_t>(
                static_cast<float>(m_CaptureBorderLength) / m_DisplayWidth *
                m_MouseState.x),
            0);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    m_NeedsToStopCapture = false;

    return true;
}

bool InputDirector::AcquireTouchMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_TouchMessageIndex >= m_TouchMessageCount)
    {
        return false;
    }
    else
    {
        *pOutValue = m_TouchMessages[m_TouchMessageIndex++];

        return true;
    }
}

bool InputDirector::AcquireDebugPadMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_DebugPadMessageIndex >= m_DebugPadMessageCount)
    {
        return false;
    }
    else
    {
        *pOutValue = m_DebugPadMessages[m_DebugPadMessageIndex++];

        return true;
    }
}

bool InputDirector::AcquireAbstractedPadMessage(Message* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_AbstractedPadMessageIndex >= m_AbstractedPadMessageCount)
    {
        return false;
    }
    else
    {
        *pOutValue = m_AbstractedPadMessages[m_AbstractedPadMessageIndex++];

        return true;
    }
}

void ReceiverThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(arg);
    auto pThreadArgs = reinterpret_cast<ThreadArgs*>(arg);
    ::nn::os::Mutex* pMutex = pThreadArgs->pMutex;
    ::nn::os::Event* pSocketEvent = pThreadArgs->pSocketEvent;
    SocketChannel* pSocketChannel = pThreadArgs->pSocketChannel;
    PingPongServer* pPingPongServer = pThreadArgs->pPingPongServer;
    InputDirector* pInputDirector = pThreadArgs->pInputDirector;
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES_NOT_NULL(pSocketEvent);
    NN_SDK_REQUIRES_NOT_NULL(pSocketChannel);
    NN_SDK_REQUIRES_NOT_NULL(pPingPongServer);
    NN_SDK_REQUIRES_NOT_NULL(pInputDirector);

    static Storage<Socket> s_SocketStorage;
    Socket* pSocket = &::nn::util::Get(s_SocketStorage);
    new(pSocket) Socket();
    pSocket->SetPortName(PortName);

    while (NN_STATIC_CONDITION(true))
    {
        SocketChannel channel;

        if (!IsSuccess(pSocket->CreateSocketChannel(&channel)))
        {
            ::nn::os::SleepThread(ConnectionTrialInterval);

            continue;
        }

        {
            ::std::lock_guard<decltype(*pMutex)> locker(*pMutex);

            *pSocketChannel = channel;

            pPingPongServer->Reset();

            pInputDirector->Reset();
        }

        pSocketEvent->Signal();

        while (NN_STATIC_CONDITION(true))
        {
            Message message;

            if (!IsSuccess(message.ReadFrom(channel)))
            {
                break;
            }

            if (pPingPongServer->AcceptMessage(message))
            {
                continue;
            }

            if (pInputDirector->AcceptMessage(message))
            {
                continue;
            }

            channel.Close();

            break;
        }
    }
}

void SenderThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(arg);
    auto pThreadArgs = reinterpret_cast<ThreadArgs*>(arg);
    ::nn::os::Mutex* pMutex = pThreadArgs->pMutex;
    ::nn::os::Event* pSocketEvent = pThreadArgs->pSocketEvent;
    SocketChannel* pSocketChannel = pThreadArgs->pSocketChannel;
    PingPongServer* pPingPongServer = pThreadArgs->pPingPongServer;
    InputDirector* pInputDirector = pThreadArgs->pInputDirector;
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES_NOT_NULL(pSocketEvent);
    NN_SDK_REQUIRES_NOT_NULL(pSocketChannel);
    NN_SDK_REQUIRES_NOT_NULL(pPingPongServer);
    NN_SDK_REQUIRES_NOT_NULL(pInputDirector);

    MessageWriter<PingPongServer> pingPongServerMessageWriter;
    pingPongServerMessageWriter.SetMutex(pMutex);
    pingPongServerMessageWriter.SetSocketChannel(pSocketChannel);
    pingPongServerMessageWriter.SetSource(pPingPongServer);

    MessageWriter<InputDirector> inputDirectorMessageWriter;
    inputDirectorMessageWriter.SetMutex(pMutex);
    inputDirectorMessageWriter.SetSocketChannel(pSocketChannel);
    inputDirectorMessageWriter.SetSource(pInputDirector);

    while (NN_STATIC_CONDITION(true))
    {
        pSocketEvent->Wait();

        ::nn::os::TimerEvent timerEvent(::nn::os::EventClearMode_AutoClear);

        timerEvent.StartPeriodic(0, PollingInterval);

        while (NN_STATIC_CONDITION(true))
        {
            timerEvent.Wait();

            if (!IsSuccess(pingPongServerMessageWriter.Flush()))
            {
                break;
            }

            pInputDirector->PrepareMessage();

            if (!IsSuccess(inputDirectorMessageWriter.Flush()))
            {
                break;
            }
        }
    }
}

} // namespace

}} // namespace nn::cs
