﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Common.h>

#include <nn/ae.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/os.h>
#include <nn/socket.h>

#include <glv.h>

#include <nv/nv_MemoryManagement.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

// For declearation of 'glslcSetAllocator'
#include <nvnTool/nvnTool_GlslcInterface.h>

#include "DummyStarter_Processor.h"
#include "DummyStarter_RootContext.h"

namespace {
    Processor g_Processor;

    // 処理スレッドのスタック
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_Stack[64 * 1024];

    // socket 用メモリ
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    // グラフィックス用メモリ
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

    void* Allocate( size_t size ) NN_NOEXCEPT
    {
        return ::malloc( size );
    }
}

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

//------------------------------------------------------------------------------
// GLSLC 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void GlslcFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

void InitializePeripherals() NN_NOEXCEPT
{
    // this memory allocation will be used from the nvn graphics systems at runtime.
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::InitializeGraphics(Allocate(GraphicsSystemReservedMemorySize),
                           GraphicsSystemReservedMemorySize );

    // this memory allocation interface will be used when compiling of shader code at runtime.
    glslcSetAllocator(GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL);
}

void FinalizePeripherals() NN_NOEXCEPT
{
}

const glv::InitialConfiguration    LocalGfxConfiguration = glv::InitialConfiguration( 512 * 1024, 1 * 1024 * 1024, 8 );
const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
    glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary
);

//------------------------------------------------------------------------------
// メイン
//------------------------------------------------------------------------------

void ProcessThread(void*) NN_NOEXCEPT
{
    g_Processor.Process();
}

void DummyStarterMain() NN_NOEXCEPT
{
    glv::Style::standard().color.set(glv::Color(0.6f, 1.0f), 0.4f); // SmokyGray 調で不透明

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    RootContext *context = new RootContext(width, height);
    context->SetProcessor(&g_Processor);

    glv::ApplicationFrameworkRegisterLoopCallback(context);
    glv::Application::run();

    delete context;

    glv::Application::quit();
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
}

extern "C" void nnMain()
{
    // AE ライブラリの初期化
    nn::ae::InitializeAsSystemApplication();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::os::ThreadType thread;
    nn::os::CreateThread(&thread, ProcessThread, nullptr, g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&thread);

    InitializePeripherals();

    glv::ApplicationFrameworkInitialize( LocalHidConfiguration, LocalGfxConfiguration );

    DummyStarterMain();

    FinalizePeripherals();

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    nn::socket::Finalize();
}

