﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/web/web_ShowArgPrivate.h>

#include <nn/web/common/web_CommonApi.h>
#include <nn/web/common/web_CommonArgData.h>
#include <nn/web/common/web_CommonTypes.h>

namespace nn { namespace web {

//------------------------------------------------------------------------
ShowNewsPageArg::ShowNewsPageArg() NN_NOEXCEPT
: m_Data()
{
    // 初期化
    common::CommonArgData::InitializeData(m_Data, common::ShimKind::Web);

    // デフォルトでアノニマスモードにする
    const bool isAnonymousMode = true;
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::IsAnonymousMode, isAnonymousMode);

    // RequestUrlと同じFQDNのURLを許可する
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::AllowFirstDomain, true);

    // private 呼び出しが行われることを設定
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::IsCalledFromPrivateApi, true);

    // ECジャンプを有効にする
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::EcJumpEnabled, true);
}

//------------------------------------------------------------------------
ShowNewsPageArg::ShowNewsPageArg(const char* pRequestUrl) NN_NOEXCEPT
: ShowNewsPageArg()
{
    // リクエストURLをセット
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::RequestURL,
        RequestUrlMaxLength, pRequestUrl);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetBootAsMediaPlayer(bool bootAsMediaPlayer) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::BootAsMediaPlayer, bootAsMediaPlayer);
    // エラーページで URL が表示されることを防ぐため、動画直接起動の場合に限りページ情報を無効化する
    // refs: BROWSER1BTS-1432
    const bool usePageInfo = !bootAsMediaPlayer;
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::UsePageInfo, usePageInfo);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetCallbackUrl(const char* pCallbackUrl) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::CallbackURL,
        CallbackUrlMaxLength, pCallbackUrl);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetCallbackableUrl(const char* pCallbackableUrl) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::CallbackableURL,
        CallbackableUrlMaxLength, pCallbackableUrl);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetWhitelist(const char* pWhitelist) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::Whitelist,
        WhitelistMaxLength, pWhitelist);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetPointerEnabled(const bool pointerEnabled) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::UseStickPointer, pointerEnabled);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetLeftStickMode(const WebLeftStickMode mode) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::LeftStickMode, nn::web::common::LeftStickMode(mode));
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetUid(const nn::account::Uid& uid) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::Uid, uid);

    // 有効なUidだったらアカウントモード(非アノニマス)にし
    // 無効なUidだったらアノニマスモードにする
    const bool isAnonymousMode = (uid == ::nn::account::InvalidUid);
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::IsAnonymousMode, isAnonymousMode);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetFooterEnabled(const bool footerEnabled) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::FooterEnabled, footerEnabled);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetBackgroundKind(const WebBackgroundKind kind) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::BackgroundKind, common::BackgroundKind(kind));
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetBootDisplayKind(const WebBootDisplayKind bootDisplayKind) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::BootDisplayKind, common::BootDisplayKind(bootDisplayKind));
}

//------------------------------------------------------------------------
void ShowNewsPageArg::SetScreenShotEnabled(const bool screenShotEnabled) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::ScreenShotEnabled, screenShotEnabled);
}

//------------------------------------------------------------------------
void ShowNewsPageArg::PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT
{
    common::CommonApi::PushStorageToInChannel(handle, m_Data, DataSize);
}

//------------------------------------------------------------------------
void ShowWebPageArgPrivate::SetScreenShotEnabled(const bool screenShotEnabled) NN_NOEXCEPT
{
    common::CommonArgData::SetData(m_Data, common::CommonArgData::ArgKind::ScreenShotEnabled, screenShotEnabled);
}


}} // namespace nn::web
