﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include <nn/rid/rid_ControllerUpdateApi.h>

namespace nn { namespace rid {
    namespace {
        const int64_t ProgressScale = 100;

        bool GetUniquePadId(hid::system::UniquePadId* outValue, bluetooth::Address& address)
        {
            hid::system::UniquePadId padList[hid::system::UniquePadIdCountMax];
            int padCount = hid::system::ListUniquePads(padList, hid::system::UniquePadIdCountMax);
            for (int i = 0; i < padCount; i++)
            {
                bluetooth::Address gotAddress;
                if (hid::system::GetUniquePadBluetoothAddress(&gotAddress, padList[i]).IsSuccess())
                {
                    if (gotAddress == address)
                    {
                        *outValue = padList[i];
                        return true;
                    }
                }
            }
            return false;
        }

        Result TryConnectAllRegisteredDevices() NN_NOEXCEPT
        {
            os::MultiWaitType multiWait;
            os::SystemEventType timeoutEvent;
            os::MultiWaitHolderType timeoutEventHolder;
            os::SystemEventType connectionEvent;
            os::MultiWaitHolderType connectionEventHolder;

            os::InitializeMultiWait(&multiWait);
            NN_UTIL_SCOPE_EXIT{ os::FinalizeMultiWait(&multiWait); };

            // コントローラーへの接続処理がタイムアウトしたときのシステムイベント
            hid::system::BindConnectionTriggerTimeoutEvent(&timeoutEvent, os::EventClearMode_ManualClear);
            NN_UTIL_SCOPE_EXIT{ os::DestroySystemEvent(&timeoutEvent); };
            os::InitializeMultiWaitHolder(&timeoutEventHolder, &timeoutEvent);
            NN_UTIL_SCOPE_EXIT{ os::FinalizeMultiWaitHolder(&timeoutEventHolder); };
            os::LinkMultiWaitHolder(&multiWait, &timeoutEventHolder);

            // コントローラーの接続状態の変化を検知したときのシステムイベント
            hid::system::BindUniquePadConnectionEvent(&connectionEvent, os::EventClearMode_ManualClear);
            NN_UTIL_SCOPE_EXIT{ os::DestroySystemEvent(&connectionEvent); };
            os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
            NN_UTIL_SCOPE_EXIT{ os::FinalizeMultiWaitHolder(&connectionEventHolder); };
            os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

            NN_UTIL_SCOPE_EXIT{ os::UnlinkAllMultiWaitHolder(&multiWait); };

            // 登録されているコントローラー一覧を取得
            hid::system::RegisteredDevice registeredList[hid::system::RegisteredDeviceCountMax];
            int registeredCount = hid::system::GetRegisteredDevices(registeredList, hid::system::RegisteredDeviceCountMax);

            // 登録されているコントローラーに接続を試行する
            for (int i = 0; i < registeredCount; i++)
            {
                const int MaxRetryCount = 5;
                for (int j = 0; j < MaxRetryCount; j++)
                {
                    // 登録されているコントローラーに接続トリガーを送信する
                    NN_RESULT_TRY(hid::system::SendConnectionTrigger(registeredList[i].address))
                        NN_RESULT_CATCH(hid::system::ResultBluetoothAlreadyConnected)
                        {
                            // 接続済みなので次のコントローラーへ進む
                            break;
                        }
                    NN_RESULT_END_TRY;

                    auto pHolder = os::WaitAny(&multiWait);

                    // コントローラーへの接続処理がタイムアウトした場合
                    if (pHolder == &timeoutEventHolder)
                    {
                        os::ClearSystemEvent(&timeoutEvent);
                        continue;
                    }
                    // コントローラーの接続状態の変化を検知した場合
                    else if (pHolder == &connectionEventHolder)
                    {
                        os::ClearSystemEvent(&connectionEvent);
                        hid::system::UniquePadId uniquePadId;
                        if (GetUniquePadId(&uniquePadId, registeredList[i].address))
                        {
                            // 接続されたので次のコントローラーへ進む
                            break;
                        }
                        continue;
                    }
                }
            }

            NN_RESULT_SUCCESS;
        }
    }

    ControllerUpdater::ControllerUpdater() NN_NOEXCEPT : m_Mutex(false), m_IsCancelled(false)
    {
        m_Progress = { ControllerUpdateProgress::State::DoNothing, 0, 0 };

        // ファームウェア更新機能を初期化
        hid::system::InitializeFirmwareUpdate();
    }

    Result ControllerUpdater::Execute() NN_NOEXCEPT
    {
        Result result = ExecuteImpl();
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultControllerUpdateCancelled)
            {
                UpdateProgress(ControllerUpdateProgress::State::Cancelled);
            }
            NN_RESULT_CATCH_ALL
            {
                UpdateProgress(ControllerUpdateProgress::State::Failed);
            }
        NN_RESULT_END_TRY

        return result;
    }

    void ControllerUpdater::Cancel() NN_NOEXCEPT
    {
        m_IsCancelled = true;
    }

    ControllerUpdateProgress ControllerUpdater::GetProgress() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        return m_Progress;
    }

    Result ControllerUpdater::ExecuteImpl() NN_NOEXCEPT
    {
        bool isUpdateApplied = false;
        m_IsCancelled = false;
        UpdateProgress(ControllerUpdateProgress::State::DoNothing);

        // 全ての登録されているコントローラーに接続を試みる
        NN_RESULT_DO(TryConnectAllRegisteredDevices());

        // 登録されているコントローラー一覧を取得
        hid::system::RegisteredDevice registeredList[hid::system::RegisteredDeviceCountMax];
        int registeredCount = hid::system::GetRegisteredDevices(registeredList, hid::system::RegisteredDeviceCountMax);

        for (int i = 0; i < registeredCount; i++)
        {
            UpdateProgress(ControllerUpdateProgress::State::Applying, i * ProgressScale, registeredCount * ProgressScale);

            if (m_IsCancelled)
            {
                NN_RESULT_THROW(ResultControllerUpdateCancelled());
            }

            // UniquePadId を取得する
            hid::system::UniquePadId uniquePadId;
            if (!GetUniquePadId(&uniquePadId, registeredList[i].address))
            {
                // 接続されていないので、次のコントローラーへ進む
                continue;
            }

            // ファームウェア更新があるかどうかを確認
            bool isUpdateAvailable;
            while (NN_STATIC_CONDITION(true))
            {
                NN_RESULT_TRY(hid::system::IsFirmwareUpdateAvailable(&isUpdateAvailable, uniquePadId))
                    NN_RESULT_CATCH(hid::system::ResultFirmwareVersionReading)
                    {
                        // ファームウェアバージョンの読み込み中だった場合は 50 msec 待ってからリトライする
                        os::SleepThread(TimeSpan::FromMilliSeconds(50));
                        continue;
                    }
                NN_RESULT_END_TRY

                break;
            }

            if (!isUpdateAvailable)
            {
                // 更新が不要なので、次のコントローラーへ進む
                continue;
            }

            // ファームウェア更新を開始
            hid::system::FirmwareUpdateDeviceHandle handle;
            NN_RESULT_DO(hid::system::StartFirmwareUpdate(&handle, uniquePadId));

            hid::system::FirmwareUpdateState state;
            do
            {
                os::SleepThread(TimeSpan::FromSeconds(1));

                // ファームウェア更新の経過状況を取得
                NN_RESULT_DO(hid::system::GetFirmwareUpdateState(&state, handle));
                UpdateProgress(ControllerUpdateProgress::State::Applying, (i + state.progress / 100.0) * ProgressScale, registeredCount * ProgressScale);
            } while (state.stage != hid::system::FirmwareUpdateStage_Completed);

            isUpdateApplied = true;
            UpdateProgress(ControllerUpdateProgress::State::Applying, (i + 1) * ProgressScale, registeredCount * ProgressScale);
        }

        if (isUpdateApplied)    // ファームウェア更新を適用したコントローラーがあった場合
        {
            UpdateProgress(ControllerUpdateProgress::State::Completed);
        }
        else                    // ファームウェア更新が行われなかった場合
        {
            UpdateProgress(ControllerUpdateProgress::State::NeedNoUpdate);
        }

        NN_RESULT_SUCCESS;
    }

    void ControllerUpdater::UpdateProgress(ControllerUpdateProgress::State state, int64_t done, int64_t total) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        m_Progress.state = state;
        m_Progress.done = done;
        m_Progress.total = total;
    }
}}
