﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_ProtectedFileEncryptor.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/TargetConfigs/build_Os.h>
#include <nn/nn_SdkLog.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/spl/spl_Api.h>

namespace {

}

namespace nn
{
namespace repair
{
    class SplProtectedFileEncryptor : public IProtectedFileEncryptor
    {
    public:
        SplProtectedFileEncryptor();
        virtual ~SplProtectedFileEncryptor();

        virtual nn::Result GenerateEncryptedKey(Key128 * pOut) override;
        virtual nn::Result LoadKey(Key128 encrpytedKey) override;
        virtual nn::Result CreateNewBlock(void * pOut, int64_t index, size_t size) override;
        virtual nn::Result DeProcessDataBlock(void * pOut, const BlockHeader & header, int64_t index, const void * buffer, size_t size) override;
        virtual nn::Result ProcessDataBlock(BlockHeader * pOutHeader, void * pOut, int64_t index, const void * buffer, size_t size) override;
        virtual nn::Result CalculateCmac(Cmac * pOut, void * buffer, size_t size) override;

    private:
        void FillRandomBytes(void *buffer, size_t size);

        int m_AesKeySlot;
        nn::spl::AccessKey m_AccessKey;
        Key128 m_EncryptedKey;
    };

    nn::Result CreateSplProtectedFileEncryptor(std::shared_ptr<IProtectedFileEncryptor>* pOut)
    {
        pOut->reset(new SplProtectedFileEncryptor());
        NN_RESULT_SUCCESS;
    }

    SplProtectedFileEncryptor::SplProtectedFileEncryptor()
    {
        const Key128 keySource = {
            {0x8d, 0x72, 0xc1, 0x00, 0xf3, 0xbe, 0xf1, 0xba, 0xa3, 0xa8, 0xca, 0x63, 0x4e, 0x9d, 0x19, 0xa5}
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(&m_AccessKey, keySource.data, keySource.KEY_SIZE, 0, 0));
    }

    SplProtectedFileEncryptor::~SplProtectedFileEncryptor()
    {
    }

    nn::Result SplProtectedFileEncryptor::GenerateEncryptedKey(Key128 * pOut)
    {
        NN_RESULT_DO(
            nn::spl::GenerateRandomBytes(pOut->data, pOut->KEY_SIZE));

        NN_RESULT_SUCCESS;
    }

    nn::Result SplProtectedFileEncryptor::LoadKey(Key128 encrpytedKey)
    {
        m_EncryptedKey = encrpytedKey;

        NN_RESULT_SUCCESS;
    }
    nn::Result SplProtectedFileEncryptor::CreateNewBlock(void * pOut, int64_t index, size_t size)
    {
        NN_UNUSED(index);
        FillRandomBytes(pOut, size);
        NN_RESULT_SUCCESS;
    }

    nn::Result SplProtectedFileEncryptor::DeProcessDataBlock(void * pOut, const BlockHeader & header, int64_t index, const void * buffer, size_t size)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&m_AesKeySlot));
        NN_UTIL_SCOPE_EXIT{
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(m_AesKeySlot));
        };
        NN_RESULT_DO(nn::spl::LoadAesKey(m_AesKeySlot, m_AccessKey, m_EncryptedKey.data, m_EncryptedKey.KEY_SIZE));

        Sha256Hash hash;

        nn::crypto::Sha256Generator sha256;
        sha256.Initialize();
        sha256.Update(&index, sizeof(index));
        sha256.Update(header.initialVector.data, header.initialVector.IV_SIZE);
        sha256.Update(buffer, size);
        sha256.GetHash(hash.data, hash.SIZE);

        Cmac cmac;
        NN_RESULT_DO(
            nn::spl::ComputeCmac(cmac.data, cmac.CMAC_SIZE, m_AesKeySlot, hash.data, hash.SIZE));

        if (cmac != header.cmac)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        NN_RESULT_DO(
            nn::spl::ComputeCtr(pOut, size, m_AesKeySlot, buffer, size, header.initialVector.data, header.initialVector.IV_SIZE));

        NN_RESULT_SUCCESS;
    }
    nn::Result SplProtectedFileEncryptor::ProcessDataBlock(BlockHeader * pOutHeader, void * pOut, int64_t index, const void * buffer, size_t size)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&m_AesKeySlot));
        NN_UTIL_SCOPE_EXIT{
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(m_AesKeySlot));
        };
        NN_RESULT_DO(nn::spl::LoadAesKey(m_AesKeySlot, m_AccessKey, m_EncryptedKey.data, m_EncryptedKey.KEY_SIZE));

        FillRandomBytes(pOutHeader->initialVector.data, pOutHeader->initialVector.IV_SIZE);

        NN_RESULT_DO(
            nn::spl::ComputeCtr(pOut, size, m_AesKeySlot, buffer, size, pOutHeader->initialVector.data, pOutHeader->initialVector.IV_SIZE));

        Sha256Hash hash;
        nn::crypto::Sha256Generator sha256;
        sha256.Initialize();
        sha256.Update(&index, sizeof(index));
        sha256.Update(pOutHeader->initialVector.data, pOutHeader->initialVector.IV_SIZE);
        sha256.Update(pOut, size);
        sha256.GetHash(hash.data, hash.SIZE);

        NN_RESULT_DO(
            nn::spl::ComputeCmac(pOutHeader->cmac.data, pOutHeader->cmac.CMAC_SIZE, m_AesKeySlot, hash.data, hash.SIZE));

        NN_RESULT_SUCCESS;
    }
    nn::Result SplProtectedFileEncryptor::CalculateCmac(Cmac * pOut, void * buffer, size_t size)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&m_AesKeySlot));
        NN_UTIL_SCOPE_EXIT{
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::DeallocateAesKeySlot(m_AesKeySlot));
        };
        NN_RESULT_DO(nn::spl::LoadAesKey(m_AesKeySlot, m_AccessKey, m_EncryptedKey.data, m_EncryptedKey.KEY_SIZE));

        NN_RESULT_DO(
            nn::spl::ComputeCmac(pOut->data, pOut->CMAC_SIZE, m_AesKeySlot, buffer, size));
        NN_RESULT_SUCCESS;
    }
    void SplProtectedFileEncryptor::FillRandomBytes(void * buffer, size_t size)
    {
        nn::spl::GenerateRandomBytes(buffer, size);
    }
}
}
