﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_ProtectedFile.h>

#include <algorithm>

#include <nn/TargetConfigs/build_Os.h>
#include <nn/nn_SdkLog.h>

#if defined NN_BUILD_CONFIG_OS_HORIZON
#include <nn/spl/spl_Api.h>
#endif

namespace {
    nn::Result LoadHeader(nn::repair::ProtectedFileHeader *pOut, std::shared_ptr<nn::repair::IFile> file)
    {
        nn::repair::ProtectedFileHeader header;
        size_t readSize;

        NN_RESULT_DO(
            file->Read(&readSize, 0, &header, sizeof(header)));
        NN_ABORT_UNLESS_EQUAL(readSize, sizeof(header));

        if (std::memcmp(nn::repair::ProtectedFileHeader::SIGNATURE, header.Signature, nn::repair::ProtectedFileHeader::SIGNATURE_SIZE) != 0)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        if (header.Version != nn::repair::ProtectedFileHeader::CURRENT_VERSION)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        *pOut = header;

        NN_RESULT_SUCCESS;
    }
}

namespace nn
{
namespace repair
{
    ProtectedFileHeader ProtectedFileHeader::Make(Cmac headerCmac, const AuthenticationArchiveContent &content, uint64_t version, uint64_t dataSize, uint64_t blockSize, uint64_t numBlocks)
    {
        ProtectedFileHeader ret;
        std::memcpy(ret.Signature, ProtectedFileHeader::SIGNATURE, ProtectedFileHeader::SIGNATURE_SIZE);
        ret.HeaderCmac = headerCmac;
        ret.AuthticationContent = content;
        ret.Version = version;
        ret.DataSize = dataSize;
        ret.BlockSize = blockSize;
        ret.NumberOfBlocks = numBlocks;

        return ret;
    }

    const char ProtectedFileHeader::SIGNATURE[ProtectedFileHeader::SIGNATURE_SIZE] = "NRARCHIVE";

    ProtectedFile::ProtectedFile()
        : m_IsFinished(false),
        m_Mode(0),
        m_NumBlocks(0),
        m_DataSize(0),
        m_pEncryptor(nullptr)
    {
    }

    nn::Result ProtectedFile::OpenWrite(std::shared_ptr<FileSystem> fs, const char * path, std::shared_ptr<IProtectedFileEncryptor> pEncryptor, const Key128 &key, const AuthenticationArchiveContent &content)
    {
        m_IsFinished = false;
        m_Mode = nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend;
        m_NumBlocks = 0;
        m_DataSize = 0;

        m_pEncryptor = pEncryptor;

        m_EncryptedKey = key;
        m_AuthContent = content;

        NN_RESULT_DO(
            m_pEncryptor->LoadKey(m_EncryptedKey));

        NN_RESULT_DO(
            fs->OpenFile(&m_RawFile, path, m_Mode));

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::OpenRead(std::shared_ptr<FileSystem> fs, const char* path, std::shared_ptr<IProtectedFileEncryptor> pEncryptor, const Key128 &key)
    {
        m_IsFinished = false;
        m_Mode = nn::fs::OpenMode_Read;
        m_NumBlocks = 0;
        m_DataSize = 0;

        m_pEncryptor = pEncryptor;

        m_EncryptedKey = Key128::MakeZero();
        AuthenticationArchiveContent zero = {};
        m_AuthContent = zero;

        NN_RESULT_DO(
            fs->OpenFile(&m_RawFile, path, m_Mode));

        NN_RESULT_DO(
            LoadHeader(key));

        NN_RESULT_SUCCESS;
    }

    void ProtectedFile::Close()
    {
        if (m_RawFile)
        {
            if (!m_IsFinished && (m_Mode & nn::fs::OpenMode_Write))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(Finish());
            }
            m_RawFile->Close();
            m_RawFile.reset();
        }
    }

    nn::Result ProtectedFile::Write(int64_t offset, const void * buffer, size_t size, bool flush)
    {
        if ( !(m_Mode & nn::fs::OpenMode_Write) )
        {
            NN_RESULT_THROW(nn::fs::ResultInvalidOperationForOpenMode());
        }

        int64_t endOffset = offset + size;

        // loop per block
        for (int64_t currentOffset = nn::util::align_down(offset, BlockSize);
            currentOffset < endOffset;
            currentOffset += BlockSize)
        {
            NN_ABORT_UNLESS(nn::util::is_aligned(currentOffset, BlockSize));
            auto currentIndex = currentOffset / BlockSize;
            auto offsetInBlock = std::max(offset, currentOffset) - currentOffset;
            auto endOffsetInBlock = std::min<int64_t>(BlockSize, endOffset - currentOffset);
            auto sizeInBlock = static_cast<size_t>(endOffsetInBlock - offsetInBlock);
            auto offsetInBuffer = currentOffset + offsetInBlock - offset;
            auto bufferInBlock = &(reinterpret_cast<const uint8_t*>(buffer)[offsetInBuffer]);

            NN_RESULT_DO(
                WriteBlock(currentIndex, offsetInBlock, bufferInBlock, sizeInBlock, flush));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::Read(size_t * pOut, int64_t offset, void * buffer, size_t size)
    {
        NN_ABORT_UNLESS_NOT_NULL(pOut);

        if (m_Mode != nn::fs::OpenMode_Read)
        {
            NN_RESULT_THROW(nn::fs::ResultInvalidOperationForOpenMode());
        }

        if (m_DataSize < static_cast<int64_t>(offset + size))
        {
            NN_RESULT_THROW(nn::fs::ResultOutOfRange());
        }

        int64_t endOffset = offset + size;
        std::unique_ptr<uint8_t[]> blockBuffer(new uint8_t[BlockSize]);

        // loop per block
        for (int64_t currentOffset = nn::util::align_down(offset, BlockSize);
            currentOffset < endOffset;
            currentOffset += BlockSize)
        {
            NN_ABORT_UNLESS(nn::util::is_aligned(currentOffset, BlockSize));
            auto currentIndex = currentOffset / BlockSize;
            auto offsetInBlock = std::max(offset, currentOffset) - currentOffset;
            auto endOffsetInBlock = std::min<int64_t>(BlockSize, endOffset - currentOffset);
            auto sizeInBlock = static_cast<size_t>(endOffsetInBlock - offsetInBlock);
            auto offsetInBuffer = currentOffset + offsetInBlock - offset;
            auto bufferInBlock = &(reinterpret_cast<uint8_t*>(buffer)[offsetInBuffer]);

            NN_RESULT_DO(
                ReadBlock(blockBuffer.get(), currentIndex, BlockSize));

            std::memcpy(bufferInBlock, blockBuffer.get() + offsetInBlock, sizeInBlock);
        }

        *pOut = size;

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::GetSize(int64_t * pOut)
    {
        NN_RESULT_DO(m_RawFile->GetSize(pOut));
        NN_RESULT_SUCCESS;
    }

    int64_t ProtectedFile::GetDataSize()
    {
        return m_DataSize;
    }

    int64_t ProtectedFile::GetBlockCount()
    {
        return m_NumBlocks;
    }

    nn::Result ProtectedFile::Finish()
    {
        NN_ABORT_UNLESS(!m_IsFinished, "This file is already finished up.");

        NN_RESULT_DO(WriteHeader());

        m_IsFinished = true;

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::LoadHeader(const Key128 &encryptedKey)
    {
        ProtectedFileHeader header;
        size_t readSize;

        NN_RESULT_DO(
            m_RawFile->Read(&readSize, 0, &header, sizeof(header)));
        NN_ABORT_UNLESS_EQUAL(readSize, sizeof(header));

        Cmac expectedCmac = header.HeaderCmac;

        NN_RESULT_DO(
            m_pEncryptor->LoadKey(encryptedKey));

        header.HeaderCmac = Cmac::MakeZero();
        Cmac actualCmac;
        NN_RESULT_DO(
            m_pEncryptor->CalculateCmac(&actualCmac, &header, sizeof(header)));

        if (expectedCmac != actualCmac)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        if (std::memcmp(ProtectedFileHeader::SIGNATURE, header.Signature, ProtectedFileHeader::SIGNATURE_SIZE) != 0)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        if (header.Version != ProtectedFileHeader::CURRENT_VERSION)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        if (header.BlockSize != BlockSize)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        m_DataSize = header.DataSize;
        m_NumBlocks = header.NumberOfBlocks;

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::WriteHeader()
    {
        auto cmac = Cmac::MakeZero();
        auto header = ProtectedFileHeader::Make(cmac, m_AuthContent, ProtectedFileHeader::CURRENT_VERSION, m_DataSize, BlockSize, m_NumBlocks);

        auto newCmac = Cmac::MakeZero();
        m_pEncryptor->CalculateCmac(&newCmac, &header, sizeof(header));
        header.HeaderCmac = newCmac;

        NN_RESULT_DO(
            this->m_RawFile->Write(0, &header, sizeof(header), true));

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::ReadBlock(void * pOut, int64_t blockIndex, size_t size)
    {
        NN_ABORT_UNLESS(IsValidBlockIndex(blockIndex), "Failed to verify block index: %lld", blockIndex);
        NN_ABORT_UNLESS(size == BlockSize);
        NN_ABORT_UNLESS_NOT_NULL(pOut);

        std::unique_ptr<uint8_t[]> processedBuffer(new uint8_t[size]);
        auto rawBlockOffset = sizeof(ProtectedFileHeader) + blockIndex * (BlockSize + sizeof(BlockHeader));

        BlockHeader header;
        size_t readSize;
        NN_RESULT_DO(
            m_RawFile->Read(&readSize, rawBlockOffset, &header, sizeof(header)));
        NN_ABORT_UNLESS_EQUAL(sizeof(header), readSize);

        NN_RESULT_DO(
            m_RawFile->Read(&readSize, rawBlockOffset + sizeof(header), processedBuffer.get(), size));
        NN_ABORT_UNLESS_EQUAL(size, readSize);

        NN_RESULT_DO(
            m_pEncryptor->DeProcessDataBlock(pOut, header, blockIndex, processedBuffer.get(), size));

        NN_RESULT_SUCCESS;
    }

    nn::Result ProtectedFile::WriteBlock(int64_t blockIndex, int64_t offset, const void * buffer, size_t size, bool flush)
    {
        auto rawBlockOffset = sizeof(ProtectedFileHeader) + blockIndex * (BlockSize + sizeof(BlockHeader));
        std::unique_ptr<uint8_t[]> rawBlockBuffer(new uint8_t[BlockSize]);
        std::unique_ptr<uint8_t[]> proccessedBuffer(new uint8_t[BlockSize]);

        if (IsValidBlockIndex(blockIndex))
        {
            NN_RESULT_DO(
                this->ReadBlock(rawBlockBuffer.get(), blockIndex, BlockSize));
        }
        else
        {
            NN_RESULT_DO(
                m_pEncryptor->CreateNewBlock(rawBlockBuffer.get(), blockIndex, BlockSize));
        }

        std::memset(proccessedBuffer.get(), 0, BlockSize);
        std::memcpy(rawBlockBuffer.get() + offset, buffer, size);

        BlockHeader header = {};
        NN_RESULT_DO(
            m_pEncryptor->ProcessDataBlock(&header, proccessedBuffer.get(), blockIndex, rawBlockBuffer.get(), BlockSize));

        NN_RESULT_DO(
            m_RawFile->Write(rawBlockOffset, &header, sizeof(header), false));

        NN_RESULT_DO(
            m_RawFile->Write(rawBlockOffset + sizeof(header), proccessedBuffer.get(), BlockSize, flush));

        UpdateMetaData(blockIndex, static_cast<size_t>(offset + size));

        NN_RESULT_SUCCESS;
    }

    void ProtectedFile::UpdateMetaData(int64_t blockIndex, size_t writeSize)
    {
        m_NumBlocks = std::max(blockIndex + 1, m_NumBlocks);
        m_DataSize = std::max<int64_t>(m_DataSize, blockIndex * BlockSize + writeSize);
    }

    bool ProtectedFile::IsValidBlockIndex(int64_t blockIndex)
    {
        return 0 <= blockIndex && blockIndex < this->GetBlockCount();
    }

    nn::Result ProtectedFile::Flush()
    {
        return m_RawFile->Flush();
    }

    nn::Result CreateProtectedFile(std::shared_ptr<ProtectedFile> *pOut, std::shared_ptr<FileSystem> fs, const char * path, std::shared_ptr<nn::repair::IProtectedFileEncryptor> pEncryptor, const Key128 &key, const AuthenticationArchiveContent &content)
    {
        NN_RESULT_DO(
            fs->CreateFile(path));

        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());
        NN_RESULT_DO(
            file->OpenWrite(fs, path, pEncryptor, key, content));

        *pOut = file;

        NN_RESULT_SUCCESS;
    }

    nn::Result OpenProtectedFile(std::shared_ptr<ProtectedFile>* pOut, std::shared_ptr<FileSystem> fs, const char * path, std::shared_ptr<nn::repair::IProtectedFileEncryptor> pEncryptor, const Key128 & key)
    {
        auto file = std::shared_ptr<ProtectedFile>(new ProtectedFile());
        NN_RESULT_DO(
            file->OpenRead(fs, path, pEncryptor, key));

        *pOut = file;

        NN_RESULT_SUCCESS;
    }

    nn::Result LoadEncryptedKey(AuthenticationArchiveContent * pOut, std::string protectedFilePath, std::shared_ptr<FileSystem> fs)
    {
        std::shared_ptr<IFile> file;
        NN_RESULT_DO(
            fs->OpenFile(&file, protectedFilePath.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{
            file->Close();
        };

        ProtectedFileHeader header = {};
        NN_RESULT_DO(
            LoadHeader(&header, file));

        *pOut = header.AuthticationContent;

        NN_RESULT_SUCCESS;
    }
}
}
