﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_FileSystem.h>
#include <nn/repair/repair_IFile.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <memory>


namespace nn
{
    namespace repair
    {
        class NnFsFile : public IFile
        {
        public:
            virtual ~NnFsFile() {}

            nn::Result Open(const char *path, int mode);

            virtual nn::Result Write(int64_t offset, const void * buffer, size_t size, bool flush) override;
            virtual nn::Result Read(size_t * pOut, int64_t offset, void * buffer, size_t size) override;
            virtual nn::Result GetSize(int64_t * pOut) override;
            virtual nn::Result Flush() override;
            virtual void Close() override;
        private:
            nn::fs::FileHandle m_Handle;
        };

        nn::Result NnFsFile::Open(const char * path, int mode)
        {
            NN_RESULT_DO(
                nn::fs::OpenFile(&m_Handle, path, mode));

            NN_RESULT_SUCCESS;
        }

        void NnFsFile::Close()
        {
            nn::fs::CloseFile(m_Handle);
        }

        nn::Result NnFsFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
        {
            nn::fs::WriteOption option = {};
            if (flush)
            {
                option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);
            }

            NN_RESULT_DO(
                nn::fs::WriteFile(m_Handle, offset, buffer, size, option));

            NN_RESULT_SUCCESS;
        }

        nn::Result NnFsFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
        {
            NN_RESULT_DO(
                nn::fs::ReadFile(pOut, m_Handle, offset, buffer, size));

            NN_RESULT_SUCCESS;
        }

        nn::Result NnFsFile::GetSize(int64_t* pOut)
        {
            NN_RESULT_DO(
                nn::fs::GetFileSize(pOut, m_Handle));

            NN_RESULT_SUCCESS;
        }

        nn::Result NnFsFile::Flush()
        {
            NN_RESULT_DO(
                nn::fs::FlushFile(m_Handle));
            NN_RESULT_SUCCESS;
        }

        class NnFsFileSystem : public FileSystem
        {
        public:
            NnFsFileSystem()
            {
            }

            virtual nn::Result DeleteFile(const char * path) override
            {
                return nn::fs::DeleteFile(path);
            }

            virtual nn::Result CreateFile(const char * path) override
            {
                return nn::fs::CreateFile(path, 0);
            }

            virtual nn::Result OpenFile(std::shared_ptr<IFile> *pOut, const char * path, int mode) override
            {
                auto file = new NnFsFile();

                auto result = file->Open(path, mode);
                if (result.IsFailure())
                {
                    delete file;
                    NN_RESULT_THROW(result);
                }

                *pOut = std::shared_ptr<IFile>(file);

                NN_RESULT_SUCCESS;
            }

            virtual nn::Result GetFileSize(int64_t *pOut, const char * path) override
            {
                nn::fs::FileHandle handle;

                NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(handle);
                };

                NN_RESULT_DO(
                    nn::fs::GetFileSize(pOut, handle));

                NN_RESULT_SUCCESS;
            }

            virtual nn::Result Exists(bool * pOut, const char * path) override
            {
                nn::fs::FileHandle handle;

                NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
                    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                {
                    *pOut = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_END_TRY;
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(handle);
                };

                *pOut = true;

                NN_RESULT_SUCCESS;
            }

        };

        FileSystem* CreateNnfsFileSystem()
        {
            return new NnFsFileSystem();
        }
    }
}
