﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_Api.h>
#include <nn/repair/repair_ProtectedFileEncryptor.h>
#include <nn/repair/repair_IFile.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/TargetConfigs/build_Os.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <string>
#include "..\..\..\Include\nn\repair\repair_Authentication.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/spl/spl_Api.h>
#include "repair_DevKeyPair1.h"
#include "repair_DevKeyPair3.h"
#include "repair_ProdKeyPair1.h"
#include "repair_ProdKeyPair3.h"
#include "repair_DevPubKey2.h"
#include "repair_DevPubKey4.h"
#include "repair_ProdPubKey2.h"
#include "repair_ProdPubKey4.h"
#endif

namespace {
    uint8_t PublicExponent[] = { 0x00, 0x01, 0x00, 0x01 };
}

namespace nn
{
namespace repair
{
    nn::Result RepairAuthentication::Initialize(std::shared_ptr<IAuthenticationKeySource> pKeySource)
    {
        m_pKeySource = pKeySource;

        NN_RESULT_SUCCESS;
    }

    nn::Result RepairAuthentication::MakeSessionId(Id128 * pOut)
    {
        NN_RESULT_DO(
            m_pKeySource->GenerateSessionId(pOut));

        NN_RESULT_SUCCESS;
    }

    nn::Result RepairAuthentication::MakeBackupRequestMessage(BackupRequestMessage *pOut, const Id128 &sessionId, const Key128 &protectedFileKey)
    {
        Key2048 publicKey2;
        NN_RESULT_DO(m_pKeySource->GetPublicKey2(&publicKey2));

        {
            nn::crypto::RsaOaepEncryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> encrpytor;

            encrpytor.Initialize(
                publicKey2.data, publicKey2.KEY_SIZE,
                PublicExponent, sizeof(PublicExponent));
            encrpytor.UpdateLabel("", 0);

            Sha256Hash seed;
            NN_RESULT_DO(
                m_pKeySource->GenerateOaepSeed(&seed));

            encrpytor.Encrypt(
                pOut->sessionId.data, pOut->sessionId.SIZE,
                sessionId.data, sessionId.SIZE,
                seed.data, seed.SIZE);
        }

        {
            nn::crypto::RsaOaepEncryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> encrpytor;

            encrpytor.Initialize(
                publicKey2.data, publicKey2.KEY_SIZE,
                PublicExponent, sizeof(PublicExponent));
            encrpytor.UpdateLabel("", 0);

            Sha256Hash seed;
            NN_RESULT_DO(
                m_pKeySource->GenerateOaepSeed(&seed));

            encrpytor.Encrypt(
                pOut->sessionKey.data, pOut->sessionKey.SIZE,
                protectedFileKey.data, protectedFileKey.KEY_SIZE,
                seed.data, seed.SIZE);
        }

        Key2048 privateKey1;
        Key2048 publicKey1;
        NN_RESULT_DO(m_pKeySource->GetPublicKey1(&publicKey1));
        NN_RESULT_DO(m_pKeySource->GetPrivateKey1(&privateKey1));

        nn::crypto::Rsa2048Pkcs1Sha256Signer signer;
        signer.Initialize(
            publicKey1.data, publicKey1.KEY_SIZE,
            privateKey1.data, privateKey1.KEY_SIZE);
        signer.Update(pOut->sessionId.data, pOut->sessionId.SIZE);
        signer.Update(pOut->sessionKey.data, pOut->sessionKey.SIZE);

        signer.Sign(pOut->sign.data, pOut->sign.SIZE);

        NN_RESULT_DO( nn::repair::GetDeviceIdHex(pOut->devIdHex) );

        NN_RESULT_SUCCESS;
    }

    nn::Result RepairAuthentication::LoadBackupResponseMessage(AuthenticationArchiveContent * pOut, const BackupResponseMessage & response, const Id128 &sessionId)
    {
        NN_SDK_LOG("load backup response\n");

        Key2048 publicKey4;
        NN_RESULT_DO(m_pKeySource->GetPublicKey4(&publicKey4));

        nn::crypto::Rsa2048Pkcs1Sha256Verifier verifier;
        verifier.Initialize(
            publicKey4.data, publicKey4.KEY_SIZE,
            PublicExponent, sizeof(PublicExponent));
        verifier.Update(response.iv.data, response.iv.IV_SIZE);
        verifier.Update(response.sessionId.data, response.sessionId.SIZE);
        verifier.Update(response.sessionKey.data, response.sessionKey.SIZE);
        if (!verifier.Verify(response.sign.data, response.sign.SIZE))
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        Key2048 privateKey3;
        Key2048 publicKey3;
        NN_RESULT_DO(m_pKeySource->GetPrivateKey3(&privateKey3));
        NN_RESULT_DO(m_pKeySource->GetPublicKey3(&publicKey3));

        pOut->iv = response.iv;

        Id128 receivedSessionId;
        {
            nn::crypto::RsaOaepDecryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> decriptor;
            decriptor.Initialize(publicKey3.data, publicKey3.KEY_SIZE,
                privateKey3.data, privateKey3.KEY_SIZE);
            decriptor.UpdateLabel("", 0);

            auto size = decriptor.Decrypt(receivedSessionId.data, receivedSessionId.SIZE, response.sessionId.data, response.sessionId.SIZE);
            if (size != receivedSessionId.SIZE)
            {
                NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
            }
        }

        if (receivedSessionId != sessionId)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        {
            nn::crypto::RsaOaepDecryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> decriptor;
            decriptor.Initialize(publicKey3.data, publicKey3.KEY_SIZE,
                privateKey3.data, privateKey3.KEY_SIZE);
            decriptor.UpdateLabel("", 0);

            auto size = decriptor.Decrypt(pOut->encryptedSessionKey.data, pOut->encryptedSessionKey.KEY_SIZE, response.sessionKey.data, response.sessionKey.SIZE);
            if (size != pOut->encryptedSessionKey.KEY_SIZE)
            {
                NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result RepairAuthentication::MakeRestoreRequestMessage(RestoreRequestMessage * pOut, const Id128 &sessionId, const AuthenticationArchiveContent & content)
    {
        Key2048 publicKey2;
        NN_RESULT_DO(m_pKeySource->GetPublicKey2(&publicKey2));

        pOut->iv = content.iv;

        {
            nn::crypto::RsaOaepEncryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> encrpytor;

            encrpytor.Initialize(
                publicKey2.data, publicKey2.KEY_SIZE,
                PublicExponent, sizeof(PublicExponent));
            encrpytor.UpdateLabel("", 0);

            Sha256Hash seed;
            NN_RESULT_DO(
                m_pKeySource->GenerateOaepSeed(&seed));

            encrpytor.Encrypt(
                pOut->sessionId.data, pOut->sessionId.SIZE,
                sessionId.data, sessionId.SIZE,
                seed.data, seed.SIZE);
        }

        {
            nn::crypto::RsaOaepEncryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> encrpytor;

            encrpytor.Initialize(
                publicKey2.data, publicKey2.KEY_SIZE,
                PublicExponent, sizeof(PublicExponent));
            encrpytor.UpdateLabel("", 0);

            Sha256Hash seed;
            NN_RESULT_DO(
                m_pKeySource->GenerateOaepSeed(&seed));

            encrpytor.Encrypt(
                pOut->sessionKey.data, pOut->sessionKey.SIZE,
                content.encryptedSessionKey.data, content.encryptedSessionKey.KEY_SIZE,
                seed.data, seed.SIZE);
        }

        Key2048 privateKey1;
        Key2048 publicKey1;
        NN_RESULT_DO(m_pKeySource->GetPublicKey1(&publicKey1));
        NN_RESULT_DO(m_pKeySource->GetPrivateKey1(&privateKey1));

        nn::crypto::Rsa2048Pkcs1Sha256Signer signer;
        signer.Initialize(
            publicKey1.data, publicKey1.KEY_SIZE,
            privateKey1.data, privateKey1.KEY_SIZE);
        signer.Update(pOut->iv.data, pOut->iv.IV_SIZE);
        signer.Update(pOut->sessionId.data, pOut->sessionId.SIZE);
        signer.Update(pOut->sessionKey.data, pOut->sessionKey.SIZE);

        signer.Sign(pOut->sign.data, pOut->sign.SIZE);

        NN_RESULT_DO( nn::repair::GetDeviceIdHex(pOut->devIdHex) );

        NN_RESULT_SUCCESS;
    }

    nn::Result RepairAuthentication::LoadRestoreResponseMessage(Key128 * pOut, const RestoreResponseMessage & response, const Id128 & sessionId)
    {
        NN_SDK_LOG("load backup response\n");

        Key2048 publicKey4;
        NN_RESULT_DO(m_pKeySource->GetPublicKey4(&publicKey4));

        nn::crypto::Rsa2048Pkcs1Sha256Verifier verifier;
        verifier.Initialize(
            publicKey4.data, publicKey4.KEY_SIZE,
            PublicExponent, sizeof(PublicExponent));
        verifier.Update(response.sessionId.data, response.sessionId.SIZE);
        verifier.Update(response.sessionKey.data, response.sessionKey.SIZE);
        if (!verifier.Verify(response.sign.data, response.sign.SIZE))
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        Key2048 privateKey3;
        Key2048 publicKey3;
        NN_RESULT_DO(m_pKeySource->GetPrivateKey3(&privateKey3));
        NN_RESULT_DO(m_pKeySource->GetPublicKey3(&publicKey3));

        Id128 receivedSessionId;
        {
            nn::crypto::RsaOaepDecryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> decriptor;
            decriptor.Initialize(publicKey3.data, publicKey3.KEY_SIZE,
                privateKey3.data, privateKey3.KEY_SIZE);
            decriptor.UpdateLabel("", 0);

            auto size = decriptor.Decrypt(receivedSessionId.data, receivedSessionId.SIZE, response.sessionId.data, response.sessionId.SIZE);
            if (size != receivedSessionId.SIZE)
            {
                NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
            }
        }

        if (receivedSessionId != sessionId)
        {
            NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
        }

        {
            nn::crypto::RsaOaepDecryptor<Key2048::KEY_SIZE, nn::crypto::Sha256Generator> decriptor;
            decriptor.Initialize(publicKey3.data, publicKey3.KEY_SIZE,
                privateKey3.data, privateKey3.KEY_SIZE);
            decriptor.UpdateLabel("", 0);

            auto size = decriptor.Decrypt(pOut->data, pOut->KEY_SIZE, response.sessionKey.data, response.sessionKey.SIZE);
            if (size != pOut->KEY_SIZE)
            {
                NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
            }
        }

        NN_RESULT_SUCCESS;
    }

#if !defined(NN_BUILD_CONFIG_OS_HORIZON)

    nn::Result CreateSplAuthenticationKeySource(std::shared_ptr<IAuthenticationKeySource>* pOut)
    {
        NN_UNUSED(pOut);
        return nn::fs::ResultNotImplemented();
    }

#else

    nn::Result DecryptBySpl(void* outBuffer, size_t outSize, void* sourceBuffer, size_t sourceSize, void* encryptedKey, size_t keySize, void* iv, size_t ivSize)
    {
        Key128 key;
        NN_RESULT_DO(
            nn::spl::DecryptAesKey(key.data, key.KEY_SIZE, encryptedKey, keySize, 0, 0));
        NN_UTIL_SCOPE_EXIT{
            std::memset(key.data, 0, key.KEY_SIZE);
        };

        nn::crypto::Aes128CtrDecryptor decryptor;
        decryptor.Initialize(key.data, key.KEY_SIZE, iv, ivSize);
        auto decryptSize = decryptor.Update(outBuffer, outSize, sourceBuffer, sourceSize);
        NN_ABORT_UNLESS_EQUAL(decryptSize, sourceSize);

        NN_RESULT_SUCCESS;
    }

    class SplKeySourceDev : public IAuthenticationKeySource
    {
        virtual nn::Result GetPrivateKey1(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevKeyPair1_EncryptedPrivateExponent, sizeof(DevKeyPair1_EncryptedPrivateExponent),
                    DevKeyPair1_EncryptedKey, sizeof(DevKeyPair1_EncryptedKey),
                    DevKeyPair1_IV, sizeof(DevKeyPair1_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey1(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevKeyPair1_EncryptedModulus, sizeof(DevKeyPair1_EncryptedModulus),
                    DevKeyPair1_EncryptedKey, sizeof(DevKeyPair1_EncryptedKey),
                    DevKeyPair1_IV, sizeof(DevKeyPair1_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey2(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevPubKey2_EncryptedModulus, sizeof(DevPubKey2_EncryptedModulus),
                    DevPubKey2_EncryptedKey, sizeof(DevPubKey2_EncryptedKey),
                    DevPubKey2_IV, sizeof(DevPubKey2_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPrivateKey3(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevKeyPair3_EncryptedPrivateExponent, sizeof(DevKeyPair3_EncryptedPrivateExponent),
                    DevKeyPair3_EncryptedKey, sizeof(DevKeyPair3_EncryptedKey),
                    DevKeyPair3_IV, sizeof(DevKeyPair3_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey3(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevKeyPair3_EncryptedModulus, sizeof(DevKeyPair3_EncryptedModulus),
                    DevKeyPair3_EncryptedKey, sizeof(DevKeyPair3_EncryptedKey),
                    DevKeyPair3_IV, sizeof(DevKeyPair3_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey4(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    DevPubKey4_EncryptedModulus, sizeof(DevPubKey4_EncryptedModulus),
                    DevPubKey4_EncryptedKey, sizeof(DevPubKey4_EncryptedKey),
                    DevPubKey4_IV, sizeof(DevPubKey4_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GenerateSessionId(Id128 * pOut) override
        {
            NN_RESULT_DO(
                nn::spl::GenerateRandomBytes(pOut, sizeof(Id128)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GenerateOaepSeed(Sha256Hash * pOut) override
        {
            NN_RESULT_DO(
                nn::spl::GenerateRandomBytes(pOut->data, pOut->SIZE));

            NN_RESULT_SUCCESS;
        }
    };

    class SplKeySourceProd : public IAuthenticationKeySource
    {
        virtual nn::Result GetPrivateKey1(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdKeyPair1_EncryptedPrivateExponent, sizeof(ProdKeyPair1_EncryptedPrivateExponent),
                    ProdKeyPair1_EncryptedKey, sizeof(ProdKeyPair1_EncryptedKey),
                    ProdKeyPair1_IV, sizeof(ProdKeyPair1_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey1(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdKeyPair1_EncryptedModulus, sizeof(ProdKeyPair1_EncryptedModulus),
                    ProdKeyPair1_EncryptedKey, sizeof(ProdKeyPair1_EncryptedKey),
                    ProdKeyPair1_IV, sizeof(ProdKeyPair1_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey2(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdPubKey2_EncryptedModulus, sizeof(ProdPubKey2_EncryptedModulus),
                    ProdPubKey2_EncryptedKey, sizeof(ProdPubKey2_EncryptedKey),
                    ProdPubKey2_IV, sizeof(ProdPubKey2_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPrivateKey3(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdKeyPair3_EncryptedPrivateExponent, sizeof(ProdKeyPair3_EncryptedPrivateExponent),
                    ProdKeyPair3_EncryptedKey, sizeof(ProdKeyPair3_EncryptedKey),
                    ProdKeyPair3_IV, sizeof(ProdKeyPair3_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey3(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdKeyPair3_EncryptedModulus, sizeof(ProdKeyPair3_EncryptedModulus),
                    ProdKeyPair3_EncryptedKey, sizeof(ProdKeyPair3_EncryptedKey),
                    ProdKeyPair3_IV, sizeof(ProdKeyPair3_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetPublicKey4(Key2048 * pOut) override
        {
            NN_RESULT_DO(
                DecryptBySpl(
                    pOut->data, pOut->KEY_SIZE,
                    ProdPubKey4_EncryptedModulus, sizeof(ProdPubKey4_EncryptedModulus),
                    ProdPubKey4_EncryptedKey, sizeof(ProdPubKey4_EncryptedKey),
                    ProdPubKey4_IV, sizeof(ProdPubKey4_IV)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GenerateSessionId(Id128 * pOut) override
        {
            NN_RESULT_DO(
                nn::spl::GenerateRandomBytes(pOut, sizeof(Id128)));

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GenerateOaepSeed(Sha256Hash * pOut) override
        {
            NN_RESULT_DO(
                nn::spl::GenerateRandomBytes(pOut->data, pOut->SIZE));

            NN_RESULT_SUCCESS;
        }
    };

    nn::Result CreateSplAuthenticationKeySource(std::shared_ptr<IAuthenticationKeySource>* pOut)
    {
        if (nn::spl::IsDevelopment())
        {
            pOut->reset(new SplKeySourceDev());
        }
        else
        {
            pOut->reset(new SplKeySourceProd());
        }


        NN_RESULT_SUCCESS;
    }

    nn::Result WriteBackupRequestMessage(const std::string requestFilePath, const BackupRequestMessage &package, std::shared_ptr<FileSystem> fileSystem)
    {
        fileSystem->CreateFile(requestFilePath.c_str());

        std::shared_ptr<IFile> file;
        NN_RESULT_DO(
            fileSystem->OpenFile(&file, requestFilePath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{
            file->Close();
        };

        file->Write(0, &package, sizeof(package), true);

        NN_RESULT_SUCCESS;
    }

    nn::Result WriteRestoreRequestMessage(const std::string requestFilePath, const RestoreRequestMessage &package, std::shared_ptr<FileSystem> fileSystem)
    {
        fileSystem->CreateFile(requestFilePath.c_str());

        std::shared_ptr<IFile> file;
        NN_RESULT_DO(
            fileSystem->OpenFile(&file, requestFilePath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{
            file->Close();
        };

        file->Write(0, &package, sizeof(package), true);

        NN_RESULT_SUCCESS;
    }

    template<typename DataType>
    nn::Result WaitAuthenticationResponse(DataType *pOut, const std::string &responseFilePath, std::shared_ptr<FileSystem> fileSystem)
    {
        nn::Result result;
        const int RetryCount = 30;
        for (int i = 0; i < RetryCount; i++)
        {
            NN_SDK_LOG("wait response %d/%d\n", i + 1, RetryCount);

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

            bool exists;

            result = fileSystem->Exists(&exists, responseFilePath.c_str());

            if (result.IsFailure() || !exists)
            {
                continue;
            }

            int64_t fileSize;
            NN_RESULT_DO(
                fileSystem->GetFileSize(&fileSize, responseFilePath.c_str()));

            if (fileSize != sizeof(*pOut))
            {
                continue;
            }

            std::shared_ptr<IFile> file;
            NN_RESULT_DO(
                fileSystem->OpenFile(&file, responseFilePath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{
                file->Close();
            };

            size_t readSize;
            NN_RESULT_DO(
                file->Read(&readSize, 0, pOut, sizeof(*pOut)));

            if (readSize != sizeof(*pOut))
            {
                continue;
            }

            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW(nn::fs::ResultFileNotFound());
    }

    nn::Result RequestBackup(AuthenticationArchiveContent *pOut, Key128 inputKey, const std::string &requestFilePath, const std::string &responseFilePath, std::shared_ptr<FileSystem> fileSystem)
    {
        std::shared_ptr<IAuthenticationKeySource> keySource;
        NN_RESULT_DO(
            nn::repair::CreateSplAuthenticationKeySource(&keySource));

        nn::repair::RepairAuthentication auth;
        auth.Initialize(keySource);

        Id128 sessionId;
        NN_RESULT_DO(
            auth.MakeSessionId(&sessionId));

        NN_SDK_LOG("made session id.\n");

        BackupRequestMessage request;
        NN_RESULT_DO(
            auth.MakeBackupRequestMessage(&request, sessionId, inputKey));

        NN_SDK_LOG("made backup request message\n");

        NN_RESULT_DO(
            WriteBackupRequestMessage(requestFilePath, request, fileSystem));

        NN_SDK_LOG("wrote backup request message\n");

        BackupResponseMessage response = {};
        NN_RESULT_DO(
            WaitAuthenticationResponse(&response, responseFilePath.c_str(), fileSystem));

        NN_SDK_LOG("received backup response message\n");

        NN_RESULT_DO(
            auth.LoadBackupResponseMessage(pOut, response, sessionId));

        NN_SDK_LOG("received encrypted key\n");

        NN_RESULT_SUCCESS;
    }

    nn::Result RequestRestore(Key128 *pOut, const AuthenticationArchiveContent &content, std::string restoreRequestPath, std::string restoreResponsePath, std::shared_ptr<FileSystem> fileSystem)
    {
        std::shared_ptr<IAuthenticationKeySource> keySource;
        NN_RESULT_DO(
            nn::repair::CreateSplAuthenticationKeySource(&keySource));

        nn::repair::RepairAuthentication auth;
        auth.Initialize(keySource);

        Id128 sessionId;
        NN_RESULT_DO(
            auth.MakeSessionId(&sessionId));

        NN_SDK_LOG("made session id.\n");

        RestoreRequestMessage message;
        NN_RESULT_DO(
            auth.MakeRestoreRequestMessage(&message, sessionId, content));

        NN_SDK_LOG("made backup request message\n");

        NN_RESULT_DO(
            WriteRestoreRequestMessage(restoreRequestPath, message, fileSystem));

        NN_SDK_LOG("wrote backup request message\n");

        RestoreResponseMessage response = {};
        NN_RESULT_DO(
            WaitAuthenticationResponse(&response, restoreResponsePath.c_str(), fileSystem));

        NN_SDK_LOG("received backup response message\n");

        NN_RESULT_DO(
            auth.LoadRestoreResponseMessage(pOut, response, sessionId));

        NN_SDK_LOG("received encrypted key\n");

        NN_RESULT_SUCCESS;
    }

    nn::Result RequestBackupByCallback(AuthenticationArchiveContent *pOut, Key128 inputKey,
                                       BackupRequestMessageWriter requestWriter,
                                       BackupResponseMessageReader reader)
    {
        std::shared_ptr<IAuthenticationKeySource> keySource;
        NN_RESULT_DO(
            nn::repair::CreateSplAuthenticationKeySource(&keySource));

        nn::repair::RepairAuthentication auth;
        auth.Initialize(keySource);

        Id128 sessionId;
        NN_RESULT_DO(
            auth.MakeSessionId(&sessionId));

        NN_SDK_LOG("made session id.\n");

        std::shared_ptr<BackupRequestMessage> request(new BackupRequestMessage());
        std::memset(request.get(), 0, sizeof(BackupRequestMessage));

        NN_RESULT_DO(
            auth.MakeBackupRequestMessage(request.get(), sessionId, inputKey));

        NN_SDK_LOG("made backup request message\n");

        NN_RESULT_DO(
            requestWriter(*request));

        NN_SDK_LOG("wrote backup request message\n");

        std::shared_ptr<BackupResponseMessage> response(new BackupResponseMessage());
        std::memset(response.get(), 0, sizeof(BackupResponseMessage));
        NN_RESULT_DO(
            reader(response.get()));

        NN_SDK_LOG("received backup response message\n");

        NN_RESULT_DO(
            auth.LoadBackupResponseMessage(pOut, *response, sessionId));

        NN_SDK_LOG("received encrypted key\n");

        NN_RESULT_SUCCESS;
    }

    nn::Result RequestRestoreByCallback(Key128 *pOut, const AuthenticationArchiveContent &content, RestoreRequestMessageWriter requestWriter, RestoreResponseMessageReader reader)
    {
        std::shared_ptr<IAuthenticationKeySource> keySource;
        NN_RESULT_DO(
            nn::repair::CreateSplAuthenticationKeySource(&keySource));

        nn::repair::RepairAuthentication auth;
        auth.Initialize(keySource);

        Id128 sessionId;
        NN_RESULT_DO(
            auth.MakeSessionId(&sessionId));

        NN_SDK_LOG("made session id.\n");

        RestoreRequestMessage message;
        NN_RESULT_DO(
            auth.MakeRestoreRequestMessage(&message, sessionId, content));

        NN_SDK_LOG("made restore request message\n");

        NN_RESULT_DO(
            requestWriter(message));

        NN_SDK_LOG("wrote restore request message\n");

        RestoreResponseMessage response = {};
        NN_RESULT_DO(
            reader(&response));

        NN_SDK_LOG("received restore response message\n");

        NN_RESULT_DO(
            auth.LoadRestoreResponseMessage(pOut, response, sessionId));

        NN_SDK_LOG("received encrypted key\n");

        NN_RESULT_SUCCESS;
    }

#endif
}
}
