﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair.h>
#include <nn/result/result_HandlingUtility.h>
#include "detail/repair_DeviceId.h"
#include "detail/repair_Directory.h"
#include "detail/repair_MessageReporter.h"
#include "detail/repair_SerialNumber.h"
#include "detail/repair_Transporter.h"
#include "detail/repair_FirmwareVersion.h"
#include "detail/repair_ExtractMigration.h"
#include "detail/repair_UnsafeExtract.h"
#include "detail/repair_Surveyor.h"

namespace nn { namespace repair {

void Initialize() NN_NOEXCEPT
{
}

void Finalize() NN_NOEXCEPT
{
}

nn::Result ExportSaveDataSecurely(
        const char* path, const BlackListStruct* pList) NN_NOEXCEPT
{
    NN_RESULT_DO(ExportSaveDataSecurely(path, pList, nullptr));
    NN_RESULT_SUCCESS;
}

nn::Result ExportSaveDataSecurely(
        const char* path,
        const BlackListStruct* pList,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 出力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    // セーブデータを抜き出す
    detail::Transporter transporter(path, true);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };
    NN_RESULT_DO(transporter.Export(pList));

    NN_RESULT_SUCCESS;
}

nn::Result ExportSaveData(
        const char* path, const BlackListStruct* pList) NN_NOEXCEPT
{
    NN_RESULT_DO(ExportSaveData(path, pList, nullptr));
    NN_RESULT_SUCCESS;
}

nn::Result GetTransportRateInfo(uint64_t *pOutSize, int64_t *pOutElapsed, int64_t *pOutCount)
{
    *pOutSize     = detail::Surveyor::TotalSizeSum();
    *pOutElapsed  = detail::Surveyor::ElapsedTimeSum();
    *pOutCount    = detail::Surveyor::TotalFileCount();

    NN_RESULT_SUCCESS;
}


nn::Result DetectMigration(
    const char* path,
    const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 出力先ディレクトリの存在を確認
    const std::string RootPath(path);

    // セーブデータを抜き出す
    detail::ExtractMigration dealer(nn::fs::SaveDataSpaceId::ProperSystem, path);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };

    // ユーザ移行中の場合に処理を行う
    bool isInMigration = dealer.GetMigrationIsActive();

    if(isInMigration)
    {
        NN_RESULT_DO(dealer.Export());
        return nn::repair::ResultInMigration();
    }

    NN_RESULT_SUCCESS;
}


nn::Result ExportSaveData(
        const char* path,
        const BlackListStruct* pList,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 出力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    // セーブデータを抜き出す
    detail::Transporter transporter(path, false);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };
    NN_RESULT_DO(transporter.Export(pList));

    NN_RESULT_SUCCESS;
}

nn::Result UnsafeExtract(
        const char* path,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 出力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    // セーブデータを抜き出す
    detail::UnsafeExtract transporter(path);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };
    NN_RESULT_DO(transporter.Export());

    NN_RESULT_SUCCESS;
}


nn::Result ImportSaveDataSecurely(const char* path) NN_NOEXCEPT
{
    NN_RESULT_DO(ImportSaveDataSecurely(path, nullptr));
    NN_RESULT_SUCCESS;
}

nn::Result ImportSaveDataSecurely(
        const char* path, const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 入力元ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    // セーブデータを取り込む
    detail::Transporter transporter(path, true);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };
    NN_RESULT_DO(transporter.Import());

    NN_RESULT_SUCCESS;
}

nn::Result ImportSaveData(const char* path) NN_NOEXCEPT
{
    NN_RESULT_DO(ImportSaveData(path, nullptr));
    NN_RESULT_SUCCESS;
}

nn::Result ImportSaveData(
        const char* path, const ReportCallbackStruct* pCallback) NN_NOEXCEPT
{
    // 入力元ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    // セーブデータを取り込む
    detail::Transporter transporter(path, false);
    detail::RegisterReportCallBack(pCallback);
    NN_UTIL_SCOPE_EXIT
    {
        detail::UnregisterReportCallBack();
    };
    NN_RESULT_DO(transporter.Import());

    NN_RESULT_SUCCESS;
}

nn::Result GetDeviceId(char* pOutDeviceId) NN_NOEXCEPT
{
    detail::DeviceId id;
    NN_RESULT_DO(id.GetValue(pOutDeviceId));

    NN_RESULT_SUCCESS;
}

nn::Result GetDeviceIdHex(char* pOutDeviceId) NN_NOEXCEPT
{
    detail::DeviceId id;
    NN_RESULT_DO(id.GetHexValue(pOutDeviceId));

    NN_RESULT_SUCCESS;
}

nn::Result GetSerialNumber(char* pOutSerialNumber) NN_NOEXCEPT
{
    detail::SerialNumber number;
    NN_RESULT_DO(number.GetValue(pOutSerialNumber));

    NN_RESULT_SUCCESS;
}

nn::Result IsAccountExported(bool *pOut, const char* path) NN_NOEXCEPT
{
    // 入力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    detail::Transporter transporter(path);
    NN_RESULT_DO(transporter.ExportedId(pOut, ACCOUNT_SAVEDATA_ID));

    NN_RESULT_SUCCESS;
}

nn::Result IsAccountImported(bool *pOut, const char* path) NN_NOEXCEPT
{
    // 入力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {

        return ResultInvalidPath();
    }

    detail::Transporter transporter(path);
    NN_RESULT_DO(transporter.ImportedId(pOut, ACCOUNT_SAVEDATA_ID));

    NN_RESULT_SUCCESS;
}

nn::Result IsVirtualAccountExported(bool *pOut, const char* path) NN_NOEXCEPT
{
    // 入力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    detail::Transporter transporter(path);
    NN_RESULT_DO(transporter.ExportedId(pOut, VA_SAVEDATA_ID));

    NN_RESULT_SUCCESS;
}

bool IsAvailableAutoCup() NN_NOEXCEPT
{
    detail::FirmwareVersion fw_version;

    return fw_version.IsAvailableAutoCup();
}

nn::Result ShowCorruptedData(
        bool *pOutHasCorruption, const char* path, ReportFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT
{
    // 入力先ディレクトリの存在を確認
    const std::string RootPath(path);
    detail::Directory root(RootPath);
    if (!root.IsFound())
    {
        return ResultInvalidPath();
    }

    detail::Transporter transporter(path);
    NN_RESULT_DO(transporter.ShowRecordedData(pOutHasCorruption, CorruptionRecordName, pFunction, pParameter));

    NN_RESULT_SUCCESS;
}

bool IsAvailableMigration() NN_NOEXCEPT
{
    detail::FirmwareVersion fw_version;

    return fw_version.IsAvailableMigration();
}

}} // namespace nn::repair

